package io.agora.avc.app.setting

import android.app.Application
import androidx.lifecycle.MutableLiveData
import io.agora.avc.R
import io.agora.avc.app.developer.DeveloperOptions
import io.agora.avc.base.AppViewModel
import io.agora.avc.biz.AppController
import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.bo.AssistantInfo
import io.agora.avc.bo.LocalUser
import io.agora.avc.bo.NoticeCode
import io.agora.avc.bo.Room
import io.agora.avc.manager.logupload.LogUploader
import io.agora.avc.manager.notice.NoticeManager
import io.agora.avc.repository.DeveloperRepository
import io.agora.avc.repository.RoomRepository
import io.agora.avc.repository.UserRepository
import io.agora.avc.utils.StringUtils
import io.agora.avc.widget.LoadingStatusTextButton
import io.agora.frame.base.livedata.EventLiveData
import io.agora.frame.base.livedata.StatusEvent
import io.agora.logger.LogConverter
import io.agora.logger.Logger
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import okhttp3.ResponseBody
import javax.inject.Inject

class RoomSettingsViewModel @Inject constructor(
    application: Application,
    appController: AppController,
    private val userRepository: UserRepository,
    private val roomRepository: RoomRepository,
    private val developerRepository: DeveloperRepository,
    private val logUploader: LogUploader,
    private val noticeManager: NoticeManager
) : AppViewModel(application, appController) {
    val uploadLogLiveData = EventLiveData<Boolean>()
    val roomInfoChangedLiveData = MutableLiveData<Room>()
    val assistantChangedLiveData = MutableLiveData<AssistantInfo?>()
    val assistantStatusLiveData = MutableLiveData<Int>()
    val localUserLiveData = MutableLiveData<LocalUser>()
    val bizConnectChangedLiveData = MutableLiveData<Boolean>()
    val developerOptionChangedLiveData = MutableLiveData<DeveloperOptions>()
    val resolutionLiveData = EventLiveData<Int?>()

    override fun onResume() {
        super.onResume()
        notifyRoomInfoChanged(roomRepository.getRoom())
        queryLocalUser()
        queryBizConnected()
        queryMediaSetting()
        queryDeveloperOption()
        queryAssistantChanged()
        queryIsLocalUserApplyingAssistant()
    }

    private fun queryIsLocalUserApplyingAssistant() {
        userRepository.isLocalUserApplyingAssistant.let {
            if (it) {
                notifyAssistantStatusChanged(LoadingStatusTextButton.State.LOADING)
            }
        }
    }

    private fun queryDeveloperOption() {
        developerOptionChangedLiveData.postValue(developerRepository.getDeveloperOptions())
    }

    private fun queryLocalUser() {
        getLocalUser()?.let {
            localUserLiveData.postValue(it)
        }
    }

    private fun queryBizConnected() {
        bizConnectChangedLiveData.postValue(appController.bizConnected())
    }

    private fun queryMediaSetting() {
        resolutionLiveData.postValue(roomRepository.getRoom()?.resolution)
    }

    fun getHostName(): String? {
        val hostId = roomRepository.getRoom()?.hostUid
        val hostName = roomRepository.getRoom()?.hostName
        val localUser = getLocalUser()
        if (hostId == null || hostId?.isEmpty()) {
            return null
        }
        if (hostId.isNotEmpty() && hostName == null) {
            return StringUtils.getString(R.string.unknown_user)
        }
        if (localUser?.uid == hostId && hostName != null) {
            return localUser.name
        }
        return null
    }

    fun uploadLog(platform: String, deviceName: String) {
        statusEvent.postValue(StatusEvent.Status.LOADING)
        userRepository.getLocalUser(platform, deviceName)
            .flatMap { user ->
                logUploader.uploadLog(user.streamId)
            }
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(object : io.reactivex.Observer<ResponseBody> {
                override fun onSubscribe(d: Disposable) {

                }

                override fun onNext(it: ResponseBody) {
                    statusEvent.postValue(StatusEvent.Status.SUCCESS)
                    uploadLogLiveData.value = true
                }

                override fun onError(e: Throwable) {
                    statusEvent.postValue(StatusEvent.Status.ERROR)
                    uploadLogLiveData.value = false
                }

                override fun onComplete() {

                }
            })
    }

    fun changeRoomVideo(enable: Boolean?) {
        if (!checkBizConnected()) {
            roomInfoChangedLiveData.postValue(roomRepository.getRoom())
            return
        }
        enable?.let {
            Logger.i(TAG, "I changed the room video:${LogConverter.enable(it)}")
            appController?.setRoomVideo(enable)
        }
    }

    fun changeRoomAudio(enable: Boolean?) {
        if (!checkBizConnected()) {
            roomInfoChangedLiveData.postValue(roomRepository.getRoom())
            return
        }
        enable?.let {
            Logger.i(TAG, "I changed the room audio:${LogConverter.enable(it)}")
            appController?.setRoomAudio(enable)
        }
    }

    fun changeRoomHost(apply: Boolean) {
        if (!checkBizConnected()) {
            return
        }
        if (apply) {
            appController?.applyHost()
        } else {
            appController?.giveUpHost()
        }
    }

    fun changeVideoDimensions(resolution: Int) {
        appController?.setResolution(resolution)
    }

    override fun onEventReceived(arg: MessageEvent) {
        when (arg.type) {
            AppEvent.ROOM_INFO_CHANGED.ordinal -> {
                notifyRoomInfoChanged(roomRepository.getRoom())
            }
            AppEvent.BIZ_STATUS_CHANGED.ordinal -> {
                notifyRoomInfoChanged(roomRepository.getRoom())
                queryBizConnected()
            }
            AppEvent.LOCAL_USER_CHANGED.ordinal -> {
                if (localUserLiveData.value == null
                    || localUserLiveData.value?.isHost != getLocalUser()?.isHost
                ) {
                    queryLocalUser()
                }
            }
            AppEvent.ROOM_ASSISTANT_CHANGED.ordinal -> {
                queryAssistantChanged()
            }
            AppEvent.WAIT_ASSISTANT_RESULT.ordinal -> {
                notifyAssistantStatusChanged(LoadingStatusTextButton.State.LOADING)
            }
            AppEvent.APPLY_ASSISTANT_OPERATION_RESULT.ordinal -> {
                (arg.obj as? Boolean)?.apply {
                    notifyAssistantStatusChanged(
                        if (this) LoadingStatusTextButton.State.SUCCESS else LoadingStatusTextButton.State.FAIL
                    )
                }
            }
        }
    }

    private fun notifyRoomInfoChanged(room: Room?) {
        roomInfoChangedLiveData.postValue(room)
    }

    private fun queryAssistantChanged() {
        assistantChangedLiveData.postValue(roomRepository.getAssistantInfo())
    }

    private fun checkBizConnected(): Boolean {
        return appController.bizConnected().apply {
            if (!this) {
                noticeManager.notice(NoticeCode.CODE_SETTING_DEFAULT_ROOM_SETTING_FAIL)
            }
        }
    }

    private fun notifyAssistantStatusChanged(status: Int) {
        assistantStatusLiveData.postValue(status)
    }

    override fun getUIEvents(): Array<AppEvent>? {
        return arrayOf(
            AppEvent.ROOM_INFO_CHANGED,
            AppEvent.LOCAL_USER_CHANGED,
            AppEvent.ROOM_ASSISTANT_CHANGED,
            AppEvent.WAIT_ASSISTANT_RESULT,
            AppEvent.APPLY_ASSISTANT_OPERATION_RESULT,
        )
    }

    fun saveDeveloperVisible(show: Boolean) {
        developerRepository.getDeveloperOptions().showDeveloper = show
        queryDeveloperOption()
    }

    fun cancelAssistant() {
        Logger.i(TAG, "cancelAssistant request")
        appController.cancelAssistant()
    }

    companion object {
        private const val TAG = "[VM][RoomSetting]"
    }
}