package io.agora.avc.app.web

import android.content.Context
import android.net.Uri
import android.net.http.SslError
import android.util.AttributeSet
import android.webkit.*
import io.agora.logger.Logger

class AgoraWebView : WebView {

    private var shouldOverrideUrlListener: OnShouldOverrideUrlListener? = null
    private var pageFinishedListener: OnPageFinishedListener? = null
    private var webChromeClientListener: OnWebChromeClientListener? = null

    constructor(context: Context) : super(context) {
        initialize(context)
    }

    constructor(context: Context, attrs: AttributeSet?) : super(context, attrs) {
        initialize(context)
    }

    constructor(context: Context, attrs: AttributeSet?, defStyleAttr: Int) : super(
        context,
        attrs,
        defStyleAttr
    ) {
        initialize(context)
    }

    private fun initialize(context: Context?) {
        //init settings
        settings.javaScriptEnabled = true
        settings.setSupportZoom(false)
        //false-web width/height unit,using device-independent (CSS) pixels.
        settings.useWideViewPort = true
        settings.loadWithOverviewMode = true
        settings.allowFileAccess = false
        isVerticalScrollBarEnabled = false
        //Adapt web page width
        setInitialScale(100)
        //config webChromeClient
        webChromeClient = AgoraWebChromeClient()
        //config webViewClient
        webViewClient = AgoraWebViewClient()
    }

    /**
     * can override shouldOverrideUrlLoading method to intercept user requests
     */
    inner class AgoraWebViewClient : WebViewClient() {

        override fun onRenderProcessGone(
            view: WebView?,
            detail: RenderProcessGoneDetail?
        ): Boolean {
            Logger.e(TAG, "WebView render process gone")
            return super.onRenderProcessGone(view, detail)
        }

        /**
         * Report web resource loading error to the host application.
         */
        override fun onReceivedError(
            view: WebView?,
            request: WebResourceRequest?,
            error: WebResourceError?
        ) {
            super.onReceivedError(view, request, error)
            if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.M) {
                Logger.e(TAG, "On request: ${request?.url}, receive error: ${error?.description}")
            } else {
                Logger.e(TAG, "Received error,while loading a resource.")
            }

        }

        /**
         * Notify the host application that an HTTP error has been received from the server
         * while loading a resource. HTTP errors have status codes >= 400.
         */
        override fun onReceivedHttpError(
            view: WebView?,
            request: WebResourceRequest?,
            errorResponse: WebResourceResponse?
        ) {
            if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.LOLLIPOP) {
                Logger.e(
                    TAG,
                    "On request: ${request?.url}, receive http error: ${errorResponse?.reasonPhrase}"
                )
            } else {
                Logger.e(TAG, "Received http error,while loading a resource.")
            }
        }

        /**
         * Notify the host application that an SSL error occurred while loading a resource.
         */
        override fun onReceivedSslError(
            view: WebView?,
            handler: SslErrorHandler?,
            error: SslError?
        ) {
            super.onReceivedSslError(view, handler, error)
            Logger.e(TAG, "Received ssl error,while loading a resource.")
        }

        override fun shouldOverrideUrlLoading(view: WebView?, url: String?): Boolean {
            if (shouldOverrideUrlLoading(Uri.parse(url))) {
                return false
            }
            return true
        }

        override fun shouldOverrideUrlLoading(
            view: WebView?,
            request: WebResourceRequest?
        ): Boolean {
            if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.LOLLIPOP) {
                if (shouldOverrideUrlLoading(request?.url)) {
                    return false
                }
            }
            return true
        }

        private fun shouldOverrideUrlLoading(uri: Uri?): Boolean {
            Logger.i(TAG, "should override url loading of uri:$uri")
            shouldOverrideUrlListener?.apply {
                if (onShouldOverrideUrl(uri)) {
                    return true
                }
            }
            if (whiteList.contains(uri?.host)) {
                return true
            }
            Logger.w(TAG, "intercept url loading of host:${uri?.host}")
            return false
        }

        override fun onPageFinished(view: WebView?, url: String?) {
            pageFinishedListener?.onPageFinished(view, url)
        }

        override fun onSafeBrowsingHit(
            view: WebView?,
            request: WebResourceRequest?,
            threatType: Int,
            callback: SafeBrowsingResponse?
        ) {
            super.onSafeBrowsingHit(view, request, threatType, callback)
            Logger.w(TAG, "Received onSafeBrowsingHit callback")
        }

        override fun onLoadResource(view: WebView?, url: String?) {
            super.onLoadResource(view, url)
            Logger.w(TAG, "on load resource url:$url")
        }
    }

    /**
     * can override WebChromeClient to config debug mode
     */
    inner class AgoraWebChromeClient : WebChromeClient() {
        override fun onProgressChanged(view: WebView?, newProgress: Int) {
            webChromeClientListener?.onProgressChanged(view, newProgress)
        }
    }

    fun setOnShouldOverrideUrlListener(listener: OnShouldOverrideUrlListener?) {
        this.shouldOverrideUrlListener = listener
    }

    interface OnShouldOverrideUrlListener {
        fun onShouldOverrideUrl(uri: Uri?): Boolean
    }

    fun setOnPageFinishedListener(listener: OnPageFinishedListener?) {
        this.pageFinishedListener = listener
    }

    interface OnPageFinishedListener {
        fun onPageFinished(view: WebView?, url: String?)
    }

    fun setOnWebChromeClientListener(listener: OnWebChromeClientListener?) {
        this.webChromeClientListener = listener
    }

    interface OnWebChromeClientListener {
        fun onProgressChanged(view: WebView?, newProgress: Int)
    }

    companion object {
        private const val TAG = "[UI][AgoraWebView]"
        private val whiteList: ArrayList<String> by lazy {
            arrayListOf("open.work.weixin.qq.com", "avc-staging.sh.agoralab.co")
        }
    }
}