package io.agora.avc.app.web

import android.net.Uri
import android.os.Bundle
import android.view.View
import android.webkit.WebView
import androidx.core.net.toUri
import io.agora.avc.R
import io.agora.avc.auth.signin.wework.*
import io.agora.avc.databinding.FragmentWebBinding
import io.agora.avc.manager.splite.SPLiteProxy
import io.agora.avc.utils.ThreadUtils
import io.agora.avc.widget.COMPANY_AGORA
import io.agora.avc.widget.COMPANY_EASEMOB
import io.agora.avc.widget.TYPE_COMPANY
import io.agora.frame.base.NovelFragment
import io.agora.logger.Logger

class WeworkWebFragment : NovelFragment<WebViewModel, FragmentWebBinding>(),
    WebAppInterface.WebAppResponse {

    override fun getLayoutId(): Int {
        return R.layout.fragment_web
    }

    override fun initialize(savedInstanceState: Bundle?) {
        mBinding.actionBar.setTitle("")
        val webAppInterface = WebAppInterface(requireContext())
        webAppInterface.setWebAppResponse(this)
        mBinding.webView.addJavascriptInterface(webAppInterface, INTERFACE_NAME)
        mBinding.webView.setOnShouldOverrideUrlListener(object :
            AgoraWebView.OnShouldOverrideUrlListener {
            override fun onShouldOverrideUrl(uri: Uri?): Boolean {
                Logger.i(TAG, "on should override url:$uri")
                mViewModel?.handleWeworkLogin(uri)
                return false
            }
        })
        mBinding.webView.setOnPageFinishedListener(object : AgoraWebView.OnPageFinishedListener {
            override fun onPageFinished(view: WebView?, url: String?) {
                Logger.i(TAG, "page:$url load finish")
                mBinding?.actionBar?.setTitle(view?.title ?: "")
            }
        })
        mBinding.webView.setOnWebChromeClientListener(object :
            AgoraWebView.OnWebChromeClientListener {
            override fun onProgressChanged(view: WebView?, newProgress: Int) {
                mBinding?.progressBar?.progress = newProgress
                if (newProgress == 100) {
                    mBinding?.progressBar?.visibility = View.GONE
                } else {
                    mBinding?.progressBar?.visibility = View.VISIBLE
                }
            }
        })

        val type = arguments?.getInt(TYPE_COMPANY)
        Logger.i(TAG, "receive company type:$type")
        if (type != null) {
            mBinding.webView.loadDataWithBaseURL(
                getBaseUrl(type),
                createWeworkLoginHtml(type),
                "text/html; charset=utf-8",
                "UTF-8",
                null
            )
        }
    }

    private fun createWeworkLoginHtml(type: Int): String {
        var appId: String = AGORA_APP_ID
        var agentId: String = getAgoraAppId()
        val redirectUri: String = getRedirectUri()
        var content = getString(R.string.easemob_login)

        if (!isAgora(type)) {
            appId = EASEMOB_APP_ID
            agentId = getEasemobAgentId()
            content = getString(R.string.agora_login)
        }
        return """
            <!DOCTYPE html>
            <head>
                <title>企业微信登录</title>
                <meta content="width=device-width,initial-scale=1,user-scalable=no" name="viewport">
                <script type="text/javascript" src="https://rescdn.qqmail.com/node/ww/wwopenmng/js/sso/wwLogin-1.0.0.js"></script>
                <style type="text/css">
                    html,
                    body {
                        width: 100%;
                        height: 100%;
                        background-color: #ffffff;
                        overflow-x: hidden;
                        overflow-y: auto;
                        margin: 0;
                    }
                    
                    .main-container {
                        display: flex;
                        flex-direction: column;
                        align-items: center;
                    }
                    
                    #login_container {
                        width: 100%;
                        height: 100%;
                        display: -webkit-flex;
                        display: flex;
                        flex-direction: row;
                        justify-content: center;
                        align-items: center;
                    }
                    
                    #btn_switch {
                        font-size: 16px;
                        text-decoration: underline;
                        color: #0087FF;
                    }
                    
                    #login_container iframe {}
                </style>
            </head>

            <body style="width: 100%; height: 100%; display: flex; align-items: center; justify-content: center;">
                <div class="main-container">
                    <div id="login_container"></div>
                    <div>
                        <span id="btn_switch" onClick="switchUrl()">$content</span>
                    </div>
                </div>
            </body>
            <script type="text/javascript">
                var obj = window.WwLogin({
                    "id": "login_container",
                    "appid": "$appId",
                    "agentid": "$agentId",
                    "redirect_uri": "$redirectUri",
                    "state": "",
                    "href": "",
                });
                function switchUrl(){
                    console.log("switch URl")
                    Android.switchUrl(window.location.href);
                }
            </script>
            </html>
            
        """.trimIndent()
    }

    override fun onBackPressed() {
        super.onBackPressed()
    }

    override fun onDestroyView() {
        mBinding.webView.setOnShouldOverrideUrlListener(null)
        mBinding.webView.setOnPageFinishedListener(null)
        mBinding.webView.setOnWebChromeClientListener(null)
        super.onDestroyView()
    }

    private fun isAgoraLoginUrl(url: String): Boolean {
        val uri = url.toUri()
        val appId = uri?.getQueryParameter(PARAMETER_APP_ID)
        return appId == AGORA_APP_ID
    }

    override fun onSwitchUrl(url: String) {
        ThreadUtils.runOnUiThread {
            val type = if (isAgoraLoginUrl(url)) {
                COMPANY_EASEMOB
            } else {
                COMPANY_AGORA
            }
            mBinding.webView.loadDataWithBaseURL(
                getBaseUrl(type),
                createWeworkLoginHtml(type),
                "text/html; charset=utf-8",
                "UTF-8",
                null
            )
        }
    }

    private fun isAgora(type: Int): Boolean {
        return type == COMPANY_AGORA
    }

    companion object {
        private const val TAG = "[UI][WeworkWeb]"
        private const val INTERFACE_NAME = "Android"
        private const val REDIRECT_URI_DEBUG =
            "https%3a%2f%2favc-staging.sh.agoralab.co%2favc_private_test%2flogin"
        private const val REDIRECT_URI_RELEASE = "https%3a%2f%2fvideocall.agora.io%2flogin"

        private const val WEWORK_AUTH_URL_RELEASE =
            "https://open.work.weixin.qq.com/wwopen/sso/qrConnect" +
                    "?appid=$AGORA_APP_ID" +
                    "&agentid=$AGENT_ID_AGORA_RELEASE" +
                    "&redirect_uri=$REDIRECT_URI_RELEASE"
        private const val WEWORK_AUTH_URL_DEBUG =
            "https://open.work.weixin.qq.com/wwopen/sso/qrConnect" +
                    "?appid=$AGORA_APP_ID" +
                    "&agentid=$AGENT_ID_AGORA_DEBUG" +
                    "&redirect_uri=$REDIRECT_URI_DEBUG"

        private const val PARAMETER_CODE = "code"
        private const val PARAMETER_APP_ID = "appid"
        private const val EASEMOB_AUTH_URL_RELEASE =
            "https://open.work.weixin.qq.com/wwopen/sso/qrConnect" +
                    "?appid=$EASEMOB_APP_ID" +
                    "&agentid=$AGENT_ID_EASEMOB_RELEASE" +
                    "&redirect_uri=$REDIRECT_URI_RELEASE"
        private const val EASEMOB_AUTH_URL_DEBUG =
            "https://open.work.weixin.qq.com/wwopen/sso/qrConnect" +
                    "?appid=$EASEMOB_APP_ID" +
                    "&agentid=$AGENT_ID_EASEMOB_DEBUG" +
                    "&redirect_uri=$REDIRECT_URI_DEBUG"

        fun getBaseUrl(type: Int): String {
            return if (type == COMPANY_AGORA) getWeWorkAuthUrl() else getEasemobAuthUrl()
        }

        private fun getWeWorkAuthUrl(): String {
            return if (SPLiteProxy.isTestServer()) WEWORK_AUTH_URL_DEBUG else WEWORK_AUTH_URL_RELEASE
        }

        private fun getEasemobAuthUrl(): String {
            return if (SPLiteProxy.isTestServer()) EASEMOB_AUTH_URL_DEBUG else EASEMOB_AUTH_URL_RELEASE
        }

        fun getAgoraAppId(): String {
            return if (SPLiteProxy.isTestServer()) AGENT_ID_AGORA_DEBUG else AGENT_ID_AGORA_RELEASE
        }

        fun getEasemobAgentId(): String {
            return if (SPLiteProxy.isTestServer()) AGENT_ID_EASEMOB_DEBUG else AGENT_ID_EASEMOB_RELEASE
        }

        fun getRedirectUri(): String {
            return if (SPLiteProxy.isTestServer()) REDIRECT_URI_DEBUG else REDIRECT_URI_RELEASE
        }

    }
}