package io.agora.avc.auth.signin

import android.app.Activity
import android.content.Context
import com.google.gson.JsonObject
import io.agora.avc.auth.signin.google.GoogleSignInManager
import io.agora.avc.auth.signin.wework.WeWorkManager
import io.agora.avc.biz.AppController
import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.AppEventBus
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.bo.InnerInfo
import io.agora.avc.bo.InnerSession
import io.agora.avc.bo.NoticeCode
import io.agora.avc.bo.UserSource
import io.agora.avc.manager.notice.NoticeEvent
import io.agora.avc.manager.notice.NoticeManager
import io.agora.avc.manager.notice.NoticeType
import io.agora.avc.net.BaseObserver
import io.agora.avc.net.api.ApiService
import io.agora.avc.net.bean.BaseNo
import io.agora.avc.push.PushManager
import io.agora.avc.repository.DeveloperRepository
import io.agora.avc.repository.RoomRepository
import io.agora.avc.repository.UserRepository
import io.agora.frame.data.IDataRepository
import io.agora.logger.Logger
import io.reactivex.Observable
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.schedulers.Schedulers
import javax.inject.Inject

class SignInManagerImpl @Inject constructor(
    private val dataRepository: IDataRepository,
    private val userRepository: UserRepository,
    private val developerRepository: DeveloperRepository,
    private val appEventBus: AppEventBus,
    private val noticeManager: NoticeManager,
    private val roomRepository: RoomRepository,
    private val appController: AppController,
    private val pushManager: PushManager,
    private val googleSignInManager: GoogleSignInManager,
    private val weWorkManager: WeWorkManager,
) : SignInManager {
    override fun initialize(context: Context) {
        weWorkManager.initWWAPI(context)
    }

    override fun signInGoogle(activity: Activity) {
        googleSignInManager.signIn(activity)
    }

    override fun signInWework(type: Int) {
        weWorkManager.loginViaWeWork(type)
    }

    override fun signOut() {
        userRepository.getLocalUser()
            .map {
                developerRepository.loginWeWork(false)
                it.token ?: ""
            }
            .flatMap {
                delInnerSession()
            }
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(object : BaseObserver<BaseNo>() {
                override fun onSuccess(t: BaseNo?) {
                    Logger.i(TAG, "Successfully to logout via WeWork")
                    userRepository.removeInnerInfo()
                    appEventBus.notifyObservers(
                        MessageEvent(
                            AppEvent.LOCAL_USER_CHANGED.ordinal,
                            true
                        )
                    )
                }

                override fun onFail(t: BaseNo?) {
                    Logger.i(TAG, "Failed to logout via WeWork")
                    appEventBus.notifyObservers(
                        MessageEvent(
                            AppEvent.LOCAL_USER_CHANGED.ordinal,
                            true
                        )
                    )
                }
            })
    }

    private fun delInnerSession(): Observable<BaseNo> {
        return dataRepository.getRetrofitService(ApiService::class.java)
            .delInnerSession()
    }

    /**
     * developer mode login weWork
     */
    override fun signInByTester() {
        weWorkManager.loginWeWorkForDeveloper(true)
    }

    override fun signOutByTester() {
        weWorkManager.loginWeWorkForDeveloper(false)
    }

    override fun queryAccountInfo(
        code: String,
        source: Int,
        platform: String,
        deviceName: String
    ) {
        val data = JsonObject()
        data.addProperty("code", code)
        data.addProperty("source", source)
        data.addProperty("platform", platform)
        data.addProperty("deviceName", deviceName)

        dataRepository.getRetrofitService(ApiService::class.java)
            .queryInnerSession(data)
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(object : BaseObserver<InnerSession>() {
                override fun onSuccess(t: InnerSession?) {
                    Logger.i(TAG, "Successfully to query inner session:$t")
                    if (t != null && t.success) {
                        val info = InnerInfo(t.alias, t.department, t.name, t.innerSession, source)
                        userRepository.saveInnerInfo(info)
                        pushManager.getToken()
                        appEventBus.notifyObservers(
                            MessageEvent(
                                AppEvent.LOCAL_USER_CHANGED.ordinal,
                                true
                            )
                        )
                        appEventBus.notifyObservers(
                            MessageEvent(
                                AppEvent.WEWORK_LOGIN.ordinal,
                                true
                            )
                        )
                    }
                }

                override fun onFail(t: InnerSession?) {
                    Logger.e(TAG, "Failed to query inner session")
                    appEventBus.notifyObservers(
                        MessageEvent(
                            AppEvent.LOCAL_USER_CHANGED.ordinal,
                            true
                        )
                    )
                }
            })
    }

    override fun isAppSupportGoogle(): Boolean {
        return googleSignInManager.isDeviceSupported()
    }

    override fun isAppSupportWeiXin(): Boolean {
        // TODO: add
        return false
    }

    override fun isAppSupportWework(): Boolean {
        return weWorkManager.isWWAppInstalled()
    }

    override fun deInitialize() {
        weWorkManager.deInitWWAPI()
    }

    /**
     * verify session
     */
    override fun verifyAccount(session: String?, source: Int) {
        Logger.i(TAG, "verify session:$session, source:$source")
        if (source == UserSource.UNKNOWN.value) {
            handleTokenExpired()
            return
        }
        dataRepository.getRetrofitService(ApiService::class.java)
            .verifySession()
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(object : BaseObserver<InnerSession>() {
                override fun onSuccess(resp: InnerSession?) {
                    if (resp == null) {
                        Logger.e(TAG, "failed to verify session, session is empty")
                        return
                    }
                    Logger.i(TAG, "Successfully to synchronize session:$resp")
                    pushManager.getToken()
                    if (resp.name != null) {
                        val info =
                            InnerInfo(resp.alias, resp.department, resp.name, session, source)
                        if (!developerRepository.getDeveloperOptions().weWorkLogin) {
                            userRepository.saveInnerInfo(info)
                        }
                    }
                }

                override fun onFail(resp: InnerSession?) {
                    Logger.e(TAG, "Failed to synchronize session,session invalid")
                    if (resp?.code != RESIGNED_USER &&
                        resp?.code != SESSION_INVALID &&
                        resp?.code != TOKEN_EXPIRED
                    ) {
                        return
                    }
                    handleTokenExpired()
                }
            })
    }

    private fun handleTokenExpired() {
        Logger.i(TAG, "handle token expired")
        userRepository.removeInnerInfo()
        appEventBus.notifyObservers(MessageEvent(AppEvent.LOCAL_USER_CHANGED.ordinal, true))
        if (roomRepository.getRoom() == null) {
            noticeManager.notice(NoticeEvent(NoticeCode.CODE_JOIN_ROOM_LOGIN_EXPIRED))
        } else {
            appController.leaveRoom()
            noticeManager.notice(
                NoticeEvent(
                    NoticeCode.CODE_JOIN_ROOM_LOGIN_EXPIRED,
                    NoticeType.DIALOG
                )
            )
        }
    }

    companion object {
        private const val TAG = "[SignIn][Client]"
        private const val STATE = "io.agora"

        private const val RESIGNED_USER = 1018
        private const val SESSION_INVALID = 1023
        private const val TOKEN_EXPIRED = 11003
    }
}
