package io.agora.avc.biz

import android.content.Context
import android.util.Pair
import android.view.TextureView
import android.view.View
import androidx.core.os.bundleOf
import com.agora.valoran.Constants
import com.agora.valoran.IValoranEvents
import com.agora.valoran.bean.*
import com.google.gson.reflect.TypeToken
import io.agora.avc.MyApplication
import io.agora.avc.R
import io.agora.avc.app.address.AddressBookResponse
import io.agora.avc.app.address.MemberNode
import io.agora.avc.app.group.ContactGroup
import io.agora.avc.app.group.MeetupResponse
import io.agora.avc.base.AppContainer
import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.AppEventBus
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.biz.vlr.IVlrClient
import io.agora.avc.bo.*
import io.agora.avc.bo.valoran.ARoomUser
import io.agora.avc.config.KEY_REQUEST_ID
import io.agora.avc.extensions.*
import io.agora.avc.functions.RetryForever
import io.agora.avc.manager.logupload.LogUploader
import io.agora.avc.manager.media.MediaPlayer
import io.agora.avc.manager.notice.NoticeEvent
import io.agora.avc.manager.notice.NoticeManager
import io.agora.avc.manager.notice.NoticeType
import io.agora.avc.manager.notice.NotifyResult
import io.agora.avc.manager.quality.QualityManager
import io.agora.avc.manager.splite.SPLite
import io.agora.avc.manager.time.TimeManager
import io.agora.avc.manager.upgrade.AppUpgradeManager
import io.agora.avc.push.PushDataRepository
import io.agora.avc.push.PushManager
import io.agora.avc.repository.*
import io.agora.avc.utils.EncryptUtils
import io.agora.avc.utils.GsonUtils
import io.agora.avc.utils.StringUtils
import io.agora.logger.Logger
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import java.net.URLEncoder
import java.util.*
import java.util.concurrent.atomic.AtomicBoolean
import java.util.concurrent.atomic.AtomicInteger
import javax.inject.Inject

/**
 *  TODO: maybe need to use single thread
 */
class AppControllerImp @Inject constructor(
    private val vlrClient: IVlrClient,
    private val appEventBus: AppEventBus,
    private val userRepository: UserRepository,
    private val roomRepository: RoomRepository,
    private val chatRepository: ChatRepository,
    private val timeManager: TimeManager,
    private val qualityManager: QualityManager,
    private val noticeManager: NoticeManager,
    private val appConfigRepository: AppConfigRepository,
    private val problemRepository: ProblemRepository,
    private val spLite: SPLite,
    private val pushDataRepository: PushDataRepository,
    private val addressBookRepository: AddressBookRepository,
    private val notificationRepository: NotificationRepository,
    private val mediaPlayer: MediaPlayer,
    private val logUploader: LogUploader,
    private val appUpgradeManager: AppUpgradeManager,
    private val pushManager: PushManager,
) : AppController {
    private val vlrEventObserver: IValoranEvents = VlrEventObserver()
    private val initialized: AtomicBoolean = AtomicBoolean(false)
    private var roomState: Int = 0
    private var bizState: Int = 0
    private var lastNetworkType: Int = 0
    private var logType: Int = LOG_IMPORTANT//0-all 1-important
    private var lastRequestId: String? = null
    private var appContainer: AppContainer? = null
    private var audioEnabled: AtomicBoolean = AtomicBoolean(true)
    private var userConfigured: AtomicBoolean = AtomicBoolean(false)
    private var pendingRoom: Room? = null
    private var currentCallSeq: Int = 0
    private var operationSeqId = AtomicInteger(1)

    override fun initialize(
        context: Context,
        isEnvProduction: Boolean,
        logPath: String,
        customType: Int
    ) {
        if (context is MyApplication) {
            appContainer = context.appContainer
        }
        if (initialized.compareAndSet(false, true)) {
            vlrClient.initialize(context, vlrEventObserver, isEnvProduction, logPath, customType)
            if (appConfigRepository.getDevConfig().wifiAp) {
                setWifiAp(true)
            }
        }
    }

    override fun bizConnected(): Boolean {
        return bizState == Constants.BizConnState.getValue(Constants.BizConnState.BIZ_CONNECTED)
    }

    override fun roomConnected(): Boolean {
        return roomState == Constants.RoomState.getValue(Constants.RoomState.ROOM_CONNECTED)
    }

    /*** IAppClient override start***/
    /*** User Action***/
    override fun configUser(streamId: Int, uid: String?, access_token: String?) {
        if (userConfigured.compareAndSet(false, true)) {
            Logger.i(
                TAG,
                "configUser request, " +
                        "streamId:$streamId, " +
                        "uid:$uid, " +
                        "token:${EncryptUtils.encrypt(access_token ?: "")}"
            )
            vlrClient.configUser(streamId, uid, access_token)
        }
    }

    override fun configClientInfo(
        uuid: String,
        lang: Int,
        feature: Long,
        appVer: String,
        deviceInfo: String,
        osVersion: String
    ) {
        Logger.i(
            TAG,
            "configClientInfo request, uuid:$uuid, lang:$lang, feature:$feature, appVer:$appVer, deviceInfo:$deviceInfo, osVersion:$osVersion"
        )
        vlrClient.configClientInfo(uuid, lang, feature, appVer, deviceInfo, osVersion)
    }

    override fun joinRoom(rid: String?, pwd: String?, mode: Int, info: UserJoinInfo?) {
        Logger.i(
            TAG, "joinRoom request, " +
                    "rid:$rid, pwd:$pwd, " +
                    "mode:$mode, " +
                    "info:${GsonUtils.toJson(info)}"
        )
        val channelParams = arrayListOf(
            "{\"che.video.render.texture.mode\":0}"
        )
        info?.apply {
            rtcChannelParams = channelParams.toTypedArray()
        }

        vlrClient.joinRoom(rid, pwd, mode, info)
    }

    override fun leaveRoom() {
        Logger.i(TAG, "leaveRoom request")
        vlrClient.leaveRoom()
    }

    private fun cleanResource() {
        alignRoomMode()
        releaseRoom()
        chatRepository.clearMessage()
        problemRepository.clearIssue()
        addressBookRepository.clear()
        notificationRepository.clear()
        lastRequestId = null
    }

    private fun alignRoomMode() {
        val isUserLogin = getLocalUser()?.isThirdPartyLoggedIn ?: false
        if (appConfigRepository.getRoomMode() == RoomMode.AGORA.value && !isUserLogin) {
            Logger.w(TAG, "sync room mode")
            appConfigRepository.saveRoomMode(RoomMode.NORMAL.value)
            val event = MessageEvent(AppEvent.ROOM_MODE_CHANGED.ordinal)
            notifyObservers(event)
        }
    }

    private fun releaseRoom() {
        roomRepository.releaseRoom()
        setRoomInfo(null)
    }

    /**
     * Prevent the api call from not taking effect, therefore, call false first
     * https://jira.agoralab.co/browse/CSD-23729
     */
    override fun setLocalAudio(on: Boolean) {
        Logger.i(TAG, "setLocalAudio request, status:$on")
        if (on && audioEnabled.compareAndSet(false, true)) {
            disableLocalAudio()
            enableLocalAudio()
        }
        vlrClient.setLocalAudio(on)
    }

    override fun setLocalVideo(on: Boolean) {
        Logger.i(TAG, "setLocalVideo request, status:$on")
        vlrClient.setLocalVideo(on)
    }

    override fun startRenderVideo(target: Int, view: View?, mode: Int) {
        Logger.i(
            TAG,
            "startRenderVideo request, target:$target, view:${view?.hashCode()}, mode:$mode"
        )
        vlrClient.startRenderVideo(target, view, mode)
    }

    override fun stopRenderVideo(target: Int) {
        Logger.i(TAG, "stopRenderVideo request, target:$target")
        vlrClient.stopRenderVideo(target)
    }

    override fun setResolution(resolution: Int) {
        Logger.i(TAG, "setResolution request, resolution:$resolution")
        roomRepository.getRoom()?.resolution = resolution
        vlrClient.setResolution(resolution)
    }

    override fun setMajorUser(target: Int) {
        Logger.i(TAG, "setMajorUser request, target:$target")
        vlrClient.setMajorUser(target)
    }

    override fun disableLocalAudio() {
        Logger.i(TAG, "disable local audio")
        vlrClient.disableLocalAudio()
    }

    override fun enableLocalAudio() {
        Logger.i(TAG, "enable local audio")
        vlrClient.enableLocalAudio()
    }

    /*** Remote Control***/
    override fun setRemoteAudio(target: Int, on: Boolean) {
        Logger.i(TAG, "setRemoteAudio request, target:$target, on:$on")
        vlrClient.setRemoteAudio(target, on)
    }

    override fun setRemoteVideo(target: Int, on: Boolean) {
        Logger.i(TAG, "setRemoteVideo request, target:$target, on:$on")
        vlrClient.setRemoteVideo(target, on)
    }

    override fun kickRemote(target: Int) {
        Logger.i(TAG, "kickRemote request, target:$target")
        vlrClient.kickRemote(target)
    }

    override fun acceptRemoteRequest(type: Int, tag: String?) {
        Logger.i(TAG, "acceptRemoteRequest request, type:$type, tag:$tag")
        vlrClient.acceptRemoteRequest(type, tag)
    }

    override fun refuseRemoteRequest(type: Int, tag: String?) {
        Logger.i(TAG, "refuseRemoteRequest request, type:$type, tag:$tag")
        vlrClient.refuseRemoteRequest(type, tag)
    }

    override fun setMediaProxy(ips: Array<String>, domain: String) {
        Logger.i(TAG, "set media proxy")
        vlrClient.setMediaProxy(ips, domain)
    }

    /*** Room Control***/
    override fun applyHost() {
        Logger.i(TAG, "applyHost request")
        vlrClient.applyHost()
    }

    override fun giveUpHost() {
        Logger.i(TAG, "giveUpHost request")
        vlrClient.giveUpHost()
    }

    override fun setRoomAudio(on: Boolean) {
        Logger.i(TAG, "setRoomAudio request, on:$on")
        vlrClient.setRoomAudio(on)
    }

    override fun setRoomVideo(on: Boolean) {
        Logger.i(TAG, "setRoomVideo request, on:$on")
        vlrClient.setRoomVideo(on)
    }

    override fun applyScreenShare(seq: Int, hasWatermark: Boolean) {
        Logger.i(TAG, "applyScreenShare request, seq:$seq, watermark:$hasWatermark")
        vlrClient.applyScreenShare(seq, hasWatermark)
    }

    override fun cancelScreenShare() {
        Logger.i(TAG, "cancelScreenShare request")
        vlrClient.cancelScreenShare()
    }

    override fun sendChat(msg: String?) {
        Logger.i(TAG, "sendChat request, msg:$msg")
        vlrClient.sendChat(msg)
    }

    override fun resendChat(id: Int) {
        Logger.i(TAG, "resendChat request, id:$id")
        vlrClient.resendChat(id)
    }

    override fun setTemporaryLeave(leave: Boolean) {
        Logger.i(TAG, "setTemporaryLeave request, leave:$leave")
        if (!leave) {
            vlrClient.setTemporaryLeave(leave)
        } else if (getLocalUser()?.isAssistant == true) {
            vlrClient.setTemporaryLeave(leave)
        } else if (roomRepository.getRoomUserList().size < 50 || getLocalUser()?.isMediaOn() == true) {
            vlrClient.setTemporaryLeave(leave)
        } else {
            Logger.w(TAG, "can not send leave event")
        }
    }

    override fun startCloudRecording() {
        Logger.i(TAG, "startCloudRecording request")
        vlrClient.startCloudRecording()
    }

    override fun stopCloudRecording() {
        Logger.i(TAG, "stopCloudRecording request")
        vlrClient.stopCloudRecording()
    }

    override fun startIssueDumping(type: Int, path: String) {
        Logger.i(TAG, "startIssueDumping request, type:$type, path:$path")
        vlrClient.startIssueDumping(type, path)
    }

    override fun stopIssueDumping(type: Int) {
        Logger.i(TAG, "stopIssueDumping request, type:$type")
        vlrClient.stopIssueDumping(type)
    }

    override fun applyAssistant(type: Int) {
        Logger.i(TAG, "applyAssistant request, type:$type")
        vlrClient.applyAssistant(type)
    }

    override fun cancelAssistant() {
        Logger.i(TAG, "cancelAssistant request")
        vlrClient.cancelAssistant()
    }

    override fun enableAssist(listenOriginSound: Boolean) {
        Logger.i(TAG, "enableAssist request, listenOriginSound:$listenOriginSound")
        vlrClient.enableAssist(listenOriginSound)
    }

    override fun disableAssist() {
        Logger.i(TAG, "disableAssist request")
        vlrClient.disableAssist()
    }

    override fun getAddressBook(seq: Int, departmentId: String) {
        Logger.i(TAG, "getAddressBook request, seq:$seq, departmentId:$departmentId")
        vlrClient.getAddressBook(seq, departmentId)
    }

    override fun searchAddressBook(seq: Int, content: String) {
        Logger.i(TAG, "searchAddressBook request, seq:$seq, content:$content")
        val param = URLEncoder.encode(content, "UTF-8")
        vlrClient.searchAddressBook(seq, param)
    }

    /**
     * invite users who have not joined the room
     */
    override fun inviteContacts(seq: Int, sources: ArrayList<SimpleRoomUser>) {
        this.currentCallSeq = seq
        val sb = StringBuilder()
        val sbFilter = StringBuilder()
        val roomUserList = roomRepository.getRoomUserList()
        val inviteUserList = arrayListOf<SimpleRoomUser>()
        sources?.forEach { simpleRoomUser ->
            if (simpleRoomUser == null) {
                return@forEach
            }
            for (user in roomUserList) {
                if (user.isAttendee && user.thirdPartyUid == simpleRoomUser?.thirdPartyId) {
                    sbFilter.append(simpleRoomUser?.thirdPartyId).append(",")
                    return@forEach
                }
            }
            inviteUserList.add(simpleRoomUser)
            sb.append(simpleRoomUser?.thirdPartyId).append(",")
        }
        Logger.i(TAG, "inviteContacts request, seq:$seq, invite:$sb, except:$sbFilter")
        //if list is empty, auto return result
        if (inviteUserList.isEmpty()) {
            MessageEvent(
                AppEvent.CALL_RESULT_EVENT.ordinal,
                NotifyResult(
                    Constants.NOTIFY_INVITE_CONTACTS_OPERATION_RESULT,
                    true,
                    0,
                    seq
                )
            ).apply {
                notifyObservers(this)
            }
            return
        }
        vlrClient.inviteContacts(seq, inviteUserList)
    }

    override fun answerIncoming(
        seq: Int,
        pickUp: Boolean,
        fromCallKit: Boolean,
        params: IncomingData?
    ) {
        Logger.i(
            TAG,
            "answerIncoming request, seq:$seq, pickUp:$pickUp, fromCallKit:$fromCallKit, params:$params"
        )
        if (!pickUp) {
            pushManager.rejectIncoming()
        }
        vlrClient.answerIncoming(seq, pickUp, fromCallKit, params)
    }

    override fun createMeetup(seq: Int, name: String, memberIds: ArrayList<String>) {
        var sb = StringBuilder()
        memberIds?.forEach {
            sb.append(it).append(",")
        }
        Logger.i(TAG, "createMeetup request, seq:$seq, name:$name, memberIds:$sb")
        vlrClient.createMeetup(seq, name, memberIds)
    }

    override fun deleteMeetup(seq: Int, id: String) {
        Logger.i(TAG, "deleteMeetup request, seq:$seq, id:$id")
        vlrClient.deleteMeetup(seq, id)
    }

    override fun updateMeetup(
        seq: Int,
        id: String,
        name: String,
        newIds: ArrayList<String>?,
        deleteIds: ArrayList<String>?
    ) {
        var newIdSB = StringBuilder()
        newIds?.forEach {
            newIdSB.append(it).append(",")
        }
        var deleteIdSB = StringBuilder()
        deleteIds?.forEach {
            deleteIdSB.append(it).append(",")
        }
        Logger.i(
            TAG,
            "updateMeetup request, seq:$seq, id:$id, name:$name, new:$newIdSB,del:$deleteIdSB"
        )
        vlrClient.updateMeetup(seq, id, name, newIds, deleteIds)
    }

    override fun getMeetupList(seq: Int, page: Int, pageSize: Int) {
        Logger.i(TAG, "getMeetupList request, seq:$seq, page:$page, size:$pageSize")
        vlrClient.getMeetupList(seq, page, pageSize)
    }

    override fun getMeetupDetail(seq: Int, id: String) {
        Logger.i(TAG, "getMeetupDetail request, seq:$seq, id:$id")
        vlrClient.getMeetupDetail(seq, id)
    }

    /*** Others ***/
    override fun configAudioEffects(effects: List<Pair<Constants.EffectType, String>>) {
        Logger.i(TAG, "configAudioEffects request")
        vlrClient.configAudioEffects(effects)
    }

    override fun adjustPlaybackSignalVolume(volume: Int) {
        Logger.i(TAG, "adjust playback signal volume:$volume")
        if (volume < 100 || volume > 400) {
            Logger.e(TAG, "failed to adjust remote audio volume, volume not in range")
        }
        vlrClient.adjustPlaybackSignalVolume(volume)
    }

    override fun setParameters(type: Int, params: String?) {
        Logger.i(TAG, "setParameters request, type:$type, params:$params")
        vlrClient.setParameters(type, params)
    }

    override fun switchCamera() {
        Logger.i(TAG, "switchCamera request")
        vlrClient.switchCamera()
    }

    override fun getCallId(): String? {
        Logger.i(TAG, "getCallId request")
        return vlrClient.getCallId()
    }

    override fun createTextureView(context: Context): TextureView {
        Logger.i(TAG, "createTextureView request")
        return vlrClient.createTextureView(context)
    }

    override fun changeToAudioMode() {
        Logger.i(TAG, "changeToAudioMode request")
        return vlrClient.changeToAudioMode()
    }

    override fun changeToVideoMode() {
        Logger.i(TAG, "changeToVideoMode request")
        return vlrClient.changeToVideoMode()
    }

    override fun biRecord(type: Int, successFlag: Int, code: Int, elapse: Int, param: String) {
        Logger.i(
            TAG,
            "biRecord request, type:$type, successFlag:$successFlag, code:$code, elapse:$elapse, param:$param"
        )
        vlrClient.biRecord(type, successFlag, code, elapse, param)
    }

    override fun setBackgroundColor(streamId: Int, color: Int) {
        Logger.i(TAG, "setBackgroundColor request, streamId:$streamId, color:$color")
        vlrClient.setBackgroundColor(streamId, color)
    }

    override fun startDumpVideoReceiveTrack(uid: Int, dumpFile: String) {
        Logger.i(TAG, "startDumpVideoReceiveTrack request, uid:$uid, dumpFile:$dumpFile")
        vlrClient.startDumpVideoReceiveTrack(uid, dumpFile)
    }

    override fun stopDumpVideoReceiveTrack() {
        Logger.i(TAG, "stopDumpVideoReceiveTrack request")
        vlrClient.stopDumpVideoReceiveTrack()
    }

    override fun tryNetworkPromptAction() {
        Logger.i(TAG, "tryNetworkPromptAction request")
        vlrClient.tryNetworkPromptAction()
    }

    override fun closeNetworkPrompt() {
        Logger.i(TAG, "closeNetworkPrompt request")
        vlrClient.closeNetworkPrompt()
    }

    override fun destroy() {
        Logger.i(TAG, "destroy request")
        initialized.set(false)
        audioEnabled.set(true)
        userConfigured.set(false)
        vlrClient.destroy()
    }

    override fun showMediaStatics(visible: Boolean) {
        Logger.i(TAG, "showMediaStatics request, visible:$visible")
        val param = if (visible) {
            " {\"show_media_statics\": true}"
        } else {
            " {\"show_media_statics\": false}"
        }
        vlrClient.setParameters(Constants.ParameterType.PARAMETER_VALORAN.value, param)
    }

    override fun openNetworkEvaluation(open: Boolean) {
        val param = if (open) {
            " {\"log_network_evaluation_detail\": true}"
        } else {
            " {\"log_network_evaluation_detail\": false}"
        }
        vlrClient.setParameters(Constants.ParameterType.PARAMETER_VALORAN.value, param)
    }

    override fun setWifiAp(open: Boolean) {
        val param = if (open) {
            " {\"rtc.sdk_ap_qos\": true}"
        } else {
            " {\"rtc.sdk_ap_qos\": false}"
        }
        vlrClient.setParameters(Constants.ParameterType.PARAMETER_RTC.value, param)
    }

    override fun connectRtmLink(connect: Boolean) {
        Logger.w(TAG, "connectRtmLink request, connect:$connect")
        val param = if (connect) {
            " {\"keep_biz_connection\": true}"
        } else {
            " {\"keep_biz_connection\": false}"
        }
        vlrClient.setParameters(Constants.ParameterType.PARAMETER_VALORAN.value, param)
    }

    override fun useMetaData(use: Boolean) {
        Logger.w(TAG, "useMetaData request, use:$use")
        val param = if (use) {
            " {\"keep_rtc_metadata\": true}"
        } else {
            " {\"keep_rtc_metadata\": false}"
        }
        vlrClient.setParameters(Constants.ParameterType.PARAMETER_VALORAN.value, param)
    }

    override fun useDataChannel(use: Boolean) {
        Logger.w(TAG, "useDataChannel request, use:$use")
        val param = if (use) {
            " {\"keep_rtc_data_channel\": true}"
        } else {
            " {\"keep_rtc_data_channel\": false}"
        }
        vlrClient.setParameters(Constants.ParameterType.PARAMETER_VALORAN.value, param)
    }

    override fun setRemoteLowerVolume(volume: Int) {
        Logger.w(TAG, "set remote lower volume request, volume:$volume")
        val param = " {\"assist_remotes_lower_volume\": $volume}"
        vlrClient.setParameters(Constants.ParameterType.PARAMETER_VALORAN.value, param)
    }

    override fun setAudioEnable(enable: Boolean) {
        Logger.w(TAG, "set audio enable:$enable")
        audioEnabled.set(enable)
    }

    override fun setEnableSpeakerphone(enable: Boolean) {
        vlrClient.setEnableSpeakerphone(enable)
    }

    private fun setRemoteRequestTimeout(time: Int) {
        Logger.w(TAG, "set remote request timeout request, time:$time")
        val param = " {\"remote_request_timeout_millisec\": $time}"
        vlrClient.setParameters(Constants.ParameterType.PARAMETER_VALORAN.value, param)
    }

    private fun setRemoteControlTimeout(time: Int) {
        Logger.w(TAG, "set remote control timeout request, time:$time")
        val param = " {\"remote_control_timeout_millisec\": $time}"
        vlrClient.setParameters(Constants.ParameterType.PARAMETER_VALORAN.value, param)
    }

    private fun setEncryptionMode(mode: Int) {
        Logger.w(TAG, "set encryption mode request, mode:$mode")
        val param = " {\"room_encryption_mode\": $mode}"
        vlrClient.setParameters(Constants.ParameterType.PARAMETER_VALORAN.value, param)
    }

    /*** IAppClient override end***/

    override fun registerObserver(eventObserver: Observer, uiEvents: Array<AppEvent>?) {
        appEventBus.addObserver(eventObserver, uiEvents)
    }

    override fun unregisterObserver(eventObserver: Observer, uiEvents: Array<AppEvent>?) {
        appEventBus.deleteObserver(eventObserver, uiEvents)
    }

    inner class VlrEventObserver : IValoranEvents {
        override fun onRoomState(state: Int, reason: Int) {
            Logger.i(TAG, "onRoomState response, state:$state, p1:$reason")
            roomState = state
            if (roomState == Constants.RoomState.getValue(Constants.RoomState.ROOM_CONNECTED)) {
                qualityManager.start()
            } else if (state == Constants.RoomState.getValue(Constants.RoomState.ROOM_DISCONNECTED)) {
                cleanResource()
                val event = MessageEvent(AppEvent.LEAVE_ROOM_EVENT.ordinal)
                notifyObservers(event)
            }
            val code = when (reason) {
                Constants.RoomStateReason.getValue(Constants.RoomStateReason.RTC_JOIN_FAILURE) -> {
                    NoticeCode.CODE_ROOM_ERROR
                }
                Constants.RoomStateReason.getValue(Constants.RoomStateReason.BIZ_PWD_NOT_MATCH) -> {
                    NoticeCode.CODE_JOIN_FAIL_WRONG_PASSWORD
                }
                Constants.RoomStateReason.getValue(Constants.RoomStateReason.RTC_CONNECTION_ABORT) -> {
                    NoticeCode.CODE_JOIN_CONNECTING_RTM_ERROR
                }
                Constants.RoomStateReason.getValue(Constants.RoomStateReason.BIZ_RID_INVALID) -> {
                    NoticeCode.CODE_JOIN_FAIL_ROOM_NAME_INVALID
                }
                Constants.RoomStateReason.getValue(Constants.RoomStateReason.BIZ_AGORA_ROOM_NO_PERMISSION) -> {
                    NoticeCode.CODE_JOIN_FAIL_AGORA_LOGIN_PERMISSION
                }
                Constants.RoomStateReason.getValue(Constants.RoomStateReason.BIZ_KICK_OUT) -> {
                    //notice by onNotify, design like this
                    null
                }
                Constants.RoomStateReason.getValue(Constants.RoomStateReason.BIZ_KICK_OUT_BY_SERVER) -> {
                    //notice by onNotify, design like this
                    null
                }
                Constants.RoomStateReason.getValue(Constants.RoomStateReason.BIZ_HTTP_FAILURE) -> {
                    NoticeCode.CODE_JOIN_CONNECTING_RTM_NODE_ERROR
                }
                Constants.RoomStateReason.getValue(Constants.RoomStateReason.BIZ_JOIN_TIME_OUT),
                Constants.RoomStateReason.getValue(Constants.RoomStateReason.BIZ_ID_INVALID),
                Constants.RoomStateReason.getValue(Constants.RoomStateReason.BIZ_TOKEN_ERROR) -> {
                    NoticeCode.CODE_JOIN_CONNECTING_RTM_ERROR
                }
                else -> {
                    null
                }
            }
            if (code != null) {
                noticeManager.notice(NoticeEvent(code, NoticeType.DIALOG))
            }
            if (pendingRoom != null) {
                val newRoom = pendingRoom
                pendingRoom = null
                notifyObservers(MessageEvent(AppEvent.CHANGE_ROOM_EVENT.ordinal, newRoom))
            }
        }

        override fun onLocalAudioState(on: Boolean) {
            Logger.i(
                TAG,
                "onLocalAudioState response, on:$on, audioState:${getLocalUser()?.audioState}"
            )
            val event = MessageEvent(AppEvent.LOCAL_AUDIO_CHANGED.ordinal)
            notifyObservers(event)
        }

        override fun onLocalVideoState(on: Boolean) {
            Logger.i(
                TAG,
                "onLocalVideoState response, on:$on, videoState:${getLocalUser()?.videoState}"
            )
            val event = MessageEvent(AppEvent.LOCAL_VIDEO_CHANGED.ordinal)
            notifyObservers(event)
        }

        override fun onBizState(state: Int) {
            Logger.i(TAG, "onBizState response, state:$state")
            bizState = state
            if (bizConnected() && roomRepository.getRoom()?.rtmJoinedBefore == false) {
                roomRepository.rtmJoinedBefore(true)
                notifyObservers(MessageEvent(AppEvent.ROOM_INFO_CHANGED.ordinal))
            }
            val event = MessageEvent(AppEvent.BIZ_STATUS_CHANGED.ordinal)
            notifyObservers(event)
        }

        override fun onNewChatMessage(msg: ChatMessage?) {
            Logger.i(TAG, "onNewChatMessage response, msg:$msg")
            val type =
                if (msg?.ownerUid?.isNotEmpty() == true && msg.ownerUid != getLocalUser()?.uid) {
                    Message.CHAT_LEFT
                } else {
                    Message.CHAT_RIGHT
                }
            val chatMsg = Message(
                msg?.chatId,
                type,
                msg?.ownerUid,
                getUserName(msg?.ownerUid, msg?.name, msg?.thirdPartyName, msg?.thirdPartyAlias),
                textContent = msg?.content,
                timestamp = msg?.ts,
                status = msg?.status,
            )
            chatRepository.addMessage(chatMsg)
            val event = MessageEvent(
                AppEvent.CHAT_MESSAGE_ADD.ordinal,
                data = bundleOf(Message.KEY to GsonUtils.toJson(chatMsg))
            )
            notifyObservers(event)
        }

        private fun getUserName(
            uid: String?,
            name: String?,
            thirdPartyName: String?,
            thirdPartyAlias: String?
        ): String? {
            if (uid == roomRepository.getAssistantInfo()?.user?.uid) {
                return StringUtils.getString(R.string.as_name)
            }
            if (getRoomInfo()?.isInternal() == false) {
                return name
            }
            return when {
                thirdPartyAlias?.isNotEmpty() == true -> {
                    "${thirdPartyName}(${thirdPartyAlias})"
                }
                thirdPartyName?.isNotEmpty() == true -> {
                    "$thirdPartyName"
                }
                else -> {
                    "$name"
                }
            }
        }

        override fun onChatMessageUpdate(index: Int, status: Int) {
            Logger.i(TAG, "onChatMessageUpdate response, index:$index, status:$status")
            val position = index - 1
            val messages = chatRepository.getMessages()
            if (position >= 0 && messages.size > position) {
                messages[position].status = status
                val event = MessageEvent(
                    AppEvent.CHAT_MESSAGE_UPDATE.ordinal,
                    position
                )
                notifyObservers(event)
            }
        }

        override fun onRoomInfoChanged(info: RoomInfo?) {
            Logger.i(TAG, "onRoomInfoChanged response, info:${GsonUtils.toJson(info)}")
            info?.let {
                val roomInfo = info.toARoomInfo()
                roomInfo.elapsedTime = timeManager.elapsedRealTime() - roomInfo.elapsedTime
                val newRoomInfo = roomRepository.updateRoomInfo(roomInfo)
                setRoomInfo(newRoomInfo)
                val event = MessageEvent(AppEvent.ROOM_INFO_CHANGED.ordinal)
                notifyObservers(event)
            }
        }

        /**
         * change event to notification
         */
        override fun onRemoteRequest(
            type: Int,
            sources: Array<out SimpleRoomUser>?,
            subType: Int,
            tag: String?
        ) {
            Logger.i(
                TAG,
                "onRemoteRequest response, type:$type, sources:${GsonUtils.toJson(sources)}, subType:$subType, tag:$tag"
            )
            var sb = StringBuilder()
            sources?.forEach {
                sb.append(getUserName(it.uid, it.name, it.thirdPartyName, it.thirdPartyAlias))
                    .append(",")
            }
            if (sb.isNotEmpty()) {
                sb = sb.deleteCharAt(sb.length - 1)
            }
            val noticeEvent = when (type) {
                Constants.RequestType.getValue(Constants.RequestType.AUDIO) -> {
                    NoticeEvent(
                        NoticeCode.CODE_PEER_INVITE_MICROPHONE_RECEIVE,
                        NoticeType.NOTIFICATION_B,
                        MeetingNotificationType.NO_2,
                        sb.toString()
                    )
                }
                Constants.RequestType.getValue(Constants.RequestType.VIDEO) -> {
                    NoticeEvent(
                        NoticeCode.CODE_PEER_INVITE_CAMERA_RECEIVE,
                        NoticeType.NOTIFICATION_B,
                        MeetingNotificationType.NO_6,
                        sb.toString()
                    )
                }
                Constants.RequestType.getValue(Constants.RequestType.ASSISTANT) -> {
                    lastRequestId = tag
                    Logger.i(TAG, "record lastRequestId:$lastRequestId")
                    val code = if (subType == Constants.TransLangType.EN.value) {
                        NoticeCode.CODE_AS_ROOM_ASK_HOST_EN
                    } else {
                        NoticeCode.CODE_AS_ROOM_ASK_HOST_CN
                    }
                    NoticeEvent(
                        code,
                        NoticeType.NOTIFICATION_B,
                        MeetingNotificationType.NO_8,
                        sb.toString(),
                        bundleOf(KEY_REQUEST_ID to tag)
                    )
                }
                else -> {
                    null
                }
            }
            if (noticeEvent is NoticeEvent) {
                noticeManager.notice(noticeEvent)
            }
        }

        override fun onRemoteRequestCanceled(type: Int, subType: Int, tag: String?) {
            Logger.i(
                TAG,
                "onRemoteRequestCanceled response, type:$type, subType:$subType, tag:$tag"
            )
            val noticeEvent = when (type) {
                Constants.RequestType.getValue(Constants.RequestType.AUDIO) -> {
                    NoticeEvent(
                        NoticeCode.CODE_PEER_INVITE_MICROPHONE_RECEIVE,
                        NoticeType.NOTIFICATION_B,
                        MeetingNotificationType.NO_2,
                        show = false
                    )
                }
                Constants.RequestType.getValue(Constants.RequestType.VIDEO) -> {
                    NoticeEvent(
                        NoticeCode.CODE_PEER_INVITE_CAMERA_RECEIVE,
                        NoticeType.NOTIFICATION_B,
                        MeetingNotificationType.NO_6,
                        show = false
                    )
                }
                Constants.RequestType.getValue(Constants.RequestType.ASSISTANT) -> {
                    if (lastRequestId != null && lastRequestId != tag) {
                        Logger.w(
                            TAG,
                            "do not process request cancel lastRequestId is not equal to tag"
                        )
                        null
                    } else {
                        NoticeEvent(
                            NoticeCode.CODE_AS_ROOM_ASK_HOST_EN,
                            NoticeType.NOTIFICATION_B,
                            MeetingNotificationType.NO_8,
                            data = bundleOf(KEY_REQUEST_ID to tag),
                            show = false
                        )
                    }
                }
                else -> {
                    null
                }
            }
            if (noticeEvent is NoticeEvent) {
                noticeManager.notice(noticeEvent)
            }
        }

        override fun onRoomAssistantStatus(
            has: Boolean,
            type: Int,
            owner: SimpleRoomUser?
        ) {
            Logger.i(
                TAG,
                "onRoomAssistantStatus response, has:$has, type:${type}, owner:${owner?.streamId}"
            )
            if (owner == null) {
                Logger.e(TAG, "npe occurs in room assistant status cb, owner is empty")
                return
            }
            val assistUser = if (isMySelf(owner)) {
                updateLocalUser(owner)
            } else {
                owner.toARoomUser()
            }

            if (assistUser?.thirdPartyName?.isNotEmpty() == true) {
                assistUser.isThirdPartyLoggedIn = true
            }

            val noticeEvent = if (has && assistUser != null && !assistUser.isMySelf()) {
                val code = if (type == Constants.TransLangType.EN.value) {
                    NoticeCode.CODE_AS_OPEN_TITLE_EN
                } else {
                    NoticeCode.CODE_AS_OPEN_TITLE_CN
                }
                NoticeEvent(
                    code,
                    NoticeType.NOTIFICATION_B,
                    MeetingNotificationType.NO_8,
                )
            } else if (!has && assistUser != null && assistUser.isMySelf()) {
                NoticeEvent(
                    NoticeCode.NOT_CODE,
                    NoticeType.NOTIFICATION_A,
                    MeetingNotificationType.NO_8,
                    show = false
                )
            } else if (!has && roomRepository.queryAssistState() != Constants.AssistState.DISABLED.value) {
                NoticeEvent(
                    NoticeCode.CODE_AS_ROOM_LEAVE_OUT,
                    NoticeType.NOTIFICATION_A,
                    MeetingNotificationType.NO_8,
                )
            } else if (!has && roomRepository.queryAssistState() == Constants.AssistState.DISABLED.value) {
                NoticeEvent(
                    NoticeCode.NOT_CODE,
                    NoticeType.NOTIFICATION_A,
                    MeetingNotificationType.NO_8,
                    show = false
                )
            } else {
                null
            }

            if (noticeEvent != null) {
                noticeManager.notice(noticeEvent)
            }

            if (has && assistUser != null) {
                roomRepository.saveAssistantInfo(AssistantInfo(assistUser, type))
            } else {
                roomRepository.saveAssistantInfo(null)
                roomRepository.saveAssistState(Constants.AssistState.NONE.value)
            }

            val event = MessageEvent(AppEvent.ROOM_ASSISTANT_CHANGED.ordinal)
            notifyObservers(event)
        }

        override fun onLocalAssistState(state: Int) {
            Logger.i(TAG, "onLocalAssistState response, state:${state}")
            roomRepository.saveAssistState(state)
            val event = MessageEvent(AppEvent.LOCAL_ASSISTANT_STATE_CHANGED.ordinal)
            notifyObservers(event)

            if (state == Constants.AssistState.ENABLE_WITHOUT_ORIGIN_SOUND.ordinal
                || state == Constants.AssistState.ENABLE_WITH_ORIGIN_SOUND.ordinal
            ) {
                noticeManager.notice(
                    NoticeEvent(
                        code = if (roomRepository.getAssistantInfo()?.type == Constants.TransLangType.ZH.value) {
                            NoticeCode.CODE_AS_OPEN_TITLE_CN
                        } else {
                            NoticeCode.CODE_AS_OPEN_TITLE_EN
                        },
                        type = NoticeType.NOTIFICATION_B,
                        notificationType = MeetingNotificationType.NO_8,
                        show = false,
                    )
                )
            }
        }

        /**
         * result code:
         * kShareResultFailureAlreadyExists:1
         */
        override fun onApplyScreenShare(
            seq: Int,
            result: Int,
            cname: String?,
            share_id: Int,
            token: String?,
            encryptionMode: Int,
            encryptionKey: String?,
            encryptionSalt: String?
        ) {
            Logger.i(
                TAG,
                "onApplyScreenShare response, seq:$seq, result:$result, cname:$cname, shareId:$share_id, token:$token, encryptionMode:$encryptionMode, encryptionKey:$encryptionKey, encryptionSalt:$encryptionSalt"
            )
            if (result == Constants.ShareResultType.getValue(Constants.ShareResultType.SUCCESS)) {
                val room = getRoomInfo()
                val localUser = getLocalUser()
                val roomId = room?.getChannelId()
                val streamId = localUser?.streamId
                val userName = localUser?.getConferenceNickname() ?: ""
                if (roomId == null || streamId == null) {
                    Logger.e(TAG, "can not start screen share, roomId or streamId is empty")
                    return
                }

                val config = appConfigRepository.getAppConfig() ?: spLite.getAppConfig()
                var ips: ArrayList<String>? = null
                var domain: String? = null
                config?.app?.rtcProxy?.let { proxy ->
                    if (proxy.enable) {
                        ips = proxy.ips
                        domain = proxy.domain
                    }
                }

                val shareInfo = ShareInfo(
                    seq,
                    roomId,
                    share_id,
                    streamId,
                    userName,
                    token,
                    encryptionMode,
                    encryptionKey,
                    encryptionSalt,
                    cname,
                    ips,
                    domain
                )
                val event = MessageEvent(AppEvent.SCREEN_SHARE_EVENT.ordinal, shareInfo)
                notifyObservers(event)
            } else {
                Logger.e(TAG, "can not start screen share, result:$result")
                val code = if (result == 1) {
                    NoticeCode.CODE_SCREEN_SHARING_ERROR
                } else {
                    null
                }
                if (code == null) {
                    return
                }
                noticeManager.notice(
                    NoticeEvent(
                        NoticeCode.CODE_SCREEN_SHARING_ERROR,
                        NoticeType.NOTIFICATION_C,
                        MeetingNotificationType.NO_3
                    )
                )
            }
        }

        override fun onRoomCloudRecordingStatus(
            enable: Boolean,
            elapsedTime: Long,
            user: SimpleRoomUser?
        ) {
            Logger.i(
                TAG,
                "onRoomCloudRecordingStatus response,enable:$enable, elapsedTime:$elapsedTime, user:$user"
            )
            user?.toARoomUser()?.let {
                if (enable) {
                    val beginTime = timeManager.elapsedRealTime() - elapsedTime
                    roomRepository.saveCloudRecordInfo(CloudRecordInfo(it, beginTime))
                } else {
                    noticeManager.notice(NoticeEvent(NoticeCode.CODE_CLOUD_RECORDING_END))
                    roomRepository.saveCloudRecordInfo(null)
                }
                val event = MessageEvent(AppEvent.CLOUD_RECORDING_STATUS_CHANGED.ordinal)
                notifyObservers(event)
            }
        }

        /**
         * audio dumping user changed
         */
        override fun onRoomIssueDumpingStatus(
            enable: Boolean,
            sources: Array<SimpleRoomUser>?,
            status: IntArray?
        ) {
            Logger.i(TAG, "onRoomIssueDumpingStatus response, enable:$enable, sources:$sources")
            sources?.toARoomUserList()?.let {
                if (enable) {
                    roomRepository.saveAudioDumpingUser(it)
                } else {
                    roomRepository.saveAudioDumpingUser(it)
                }
            }
            val event = MessageEvent(AppEvent.ROOM_DUMP_STATUS_CHANGED.ordinal)
            notifyObservers(event)
        }

        /**
         * do nothing
         */
        override fun onRoomScreenSharingStatus(sharing: Boolean, user: SimpleRoomUser?) {
            Logger.i(TAG, "onRoomScreenSharingStatus response, sharing:$sharing, p1:$user")
            if (!sharing) {
                noticeManager.notice(
                    NoticeEvent(
                        NoticeCode.CODE_SCREEN_SHARING_END,
                        NoticeType.NOTIFICATION_C,
                        MeetingNotificationType.NO_3,
                        getUserName(
                            user?.uid,
                            user?.name,
                            user?.thirdPartyName,
                            user?.thirdPartyAlias
                        )
                    )
                )
            } else {
                noticeManager.notice(
                    NoticeEvent(
                        NoticeCode.CODE_SCREEN_SHARING_START,
                        NoticeType.NOTIFICATION_C,
                        MeetingNotificationType.NO_3,
                        getUserName(
                            user?.uid,
                            user?.name,
                            user?.thirdPartyName,
                            user?.thirdPartyAlias
                        )
                    )
                )
            }
        }

        /**
         * last_mile detection outside the room
         */
        override fun onLastMileQuality(
            quality: Int,
            rtt: Int,
            upLossRate: Int,
            upJitter: Int,
            upBandWidth: Int,
            downLossRate: Int,
            downJitter: Int,
            downBandWidth: Int
        ) {
            Logger.d(TAG, "onLastMileQuality response, quality:$quality, rtt:$rtt")
            val lastMile = LastMile(
                quality,
                rtt,
                upLossRate,
                upJitter,
                upBandWidth,
                downLossRate,
                downJitter,
                downBandWidth
            )
            val event = MessageEvent(AppEvent.LAST_MILE_QUALITY_CHANGED.ordinal, lastMile)
            notifyObservers(event)
        }


        /**
         *  last_mile detection in the room
         */
        override fun onLocalNetworkStatus(
            quality: Int,
            lastMileDelay: Int,
            upLossRate: Int,
            downLossRate: Int
        ) {
            Logger.d(
                TAG,
                "onLocalNetworkStatus response, quality:$quality, lastMileDelay:$lastMileDelay, upLossRate:$upLossRate, downLossRate:$downLossRate"
            )
            val networkSignal = NetworkSignal(
                quality,
                lastMileDelay,
                upLossRate,
                downLossRate,
                bizConnected(),
                roomConnected()
            )
            val event = MessageEvent(AppEvent.ROOM_NETWORK_STATUS_CHANGED.ordinal, networkSignal)
            notifyObservers(event)
        }

        /**
         * Weak network tips
         */
        override fun onNetworkQualityPrompt(type: Int, action: Int, param: String?) {
            Logger.i(
                TAG,
                "onNetworkQualityPrompt response, type:$type, action:$action, param:$param"
            )
            if (type == Constants.PromptType.getValue(Constants.PromptType.NONE)) {
                cancelNetworkQualityPrompt()
            } else {
                getNetworkNoticeCode(type)?.let { noticeCode ->
                    noticeManager.onNetworkQualityPrompt(noticeCode, action, param)
                }
            }
            lastNetworkType = type
        }

        private fun getNetworkNoticeCode(type: Int): NoticeCode? {
            return when (type) {
                Constants.PromptType.getValue(Constants.PromptType.LOCAL_OFF) -> {
                    NoticeCode.CODE_NETWORK_LOCAL_DOWN
                }
                Constants.PromptType.getValue(Constants.PromptType.REMOTE_OFF) -> {
                    NoticeCode.CODE_NETWORK_PEER_DOWN
                }
                Constants.PromptType.getValue(Constants.PromptType.LOCAL_DOWN_WEAK) -> {
                    NoticeCode.CODE_NETWORK_LOCAL_POOR
                }
                Constants.PromptType.getValue(Constants.PromptType.REMOTE_UP_WEAK) -> {
                    NoticeCode.CODE_NETWORK_PEER_POOR
                }
                Constants.PromptType.getValue(Constants.PromptType.LOCAL_UP_WEAK) -> {
                    NoticeCode.CODE_NETWORK_LOCAL_POOR
                }
                Constants.PromptType.getValue(Constants.PromptType.REMOTE_DOWN_WEAK) -> {
                    NoticeCode.CODE_NETWORK_PEER_POOR
                }
                else -> {
                    null
                }
            }
        }

        private fun cancelNetworkQualityPrompt() {
            if (lastNetworkType != Constants.PromptType.getValue(Constants.PromptType.NONE)) {
                getNetworkNoticeCode(lastNetworkType)?.let { noticeCode ->
                    noticeManager.notice(
                        NoticeEvent(
                            code = noticeCode,
                            type = NoticeType.NOTIFICATION_A,
                            notificationType = MeetingNotificationType.NO_1,
                            show = false,
                        )
                    )
                }
            }
        }

        override fun onTokenBad() {
            Logger.i(TAG, "onTokenBad response")
            handleTokenBad()
        }

        /**
         * valoran proactive notification
         */
        override fun onNotify(type: Int, keys: Array<out String>?, values: Array<out String>?) {
            val assistantStreamId = roomRepository.getAssistantInfo()?.user?.streamId
            val keyData = StringBuilder()
            val valueData = StringBuilder()
            keys?.forEach {
                keyData.append(it).append(",")
            }
            values?.forEach {
                valueData.append(it).append(",")
            }
            Logger.i(TAG, "onNotify response, type:$type, keys:$keyData, values:$valueData")
            val map = HashMap<String, String>()
            if (keys != null && values != null) {
                keys.forEachIndexed { index, key ->
                    try {
                        map[key] = values[index]
                    } catch (e: Exception) {
                        Logger.e(TAG, "failed to analysis data from onNotify, type:$type")
                    }
                }
            }

            val sourceStreamId = map[Constants.KEY_SOURCE_STREAM_ID]
            val sourceThirdPartyName = map[Constants.KEY_SOURCE_THIRD_PARTY_NAME]
            val sourceThirdPartyAlias = map[Constants.KEY_SOURCE_THIRD_PARTY_ALIAS]
            val sourceName = when {
                sourceStreamId?.isNotEmpty() == true && assistantStreamId != null && sourceStreamId == assistantStreamId.toString() -> {
                    StringUtils.getString(R.string.as_name)
                }
                sourceThirdPartyAlias?.isNotEmpty() == true -> {
                    "$sourceThirdPartyName($sourceThirdPartyAlias)"
                }
                sourceThirdPartyName?.isNotEmpty() == true -> {
                    "$sourceThirdPartyName"
                }
                else -> {
                    "${map[Constants.KEY_SOURCE_NAME]}"
                }
            }

            val targetStreamId = map[Constants.KEY_TARGET_STREAM_ID]
            val targetThirdPartyName = map[Constants.KEY_TARGET_THIRD_PARTY_NAME]
            val targetThirdPartyAlias = map[Constants.KEY_TARGET_THIRD_PARTY_ALIAS]
            val targetName = when {
                targetStreamId?.isNotEmpty() == true && assistantStreamId != null && targetStreamId == assistantStreamId.toString() -> {
                    StringUtils.getString(R.string.as_name)
                }
                targetThirdPartyAlias?.isNotEmpty() == true -> {
                    "$targetThirdPartyName($targetThirdPartyAlias)"
                }
                targetThirdPartyName?.isNotEmpty() == true -> {
                    "$targetThirdPartyName"
                }
                else -> {
                    "${map[Constants.KEY_TARGET_NAME]}"
                }
            }

            val encryptionMode = map[Constants.KEY_ENCRYPTION_MODE] as? Int

            when (type) {
                Constants.NOTIFY_LOCAL_RESOLUTION_HAS_TURN_DOWN -> {
                    if ("resolution_label" == keys?.get(0)) {
                        val resolution = values?.get(0)
                        when (resolution) {
                            "360P" -> {
                                setResolution(0)
                            }
                            "480P" -> {
                                setResolution(1)
                            }
                            "720P" -> {
                                setResolution(2)
                            }
                        }
                        noticeManager.notice(
                            NoticeEvent(
                                code = NoticeCode.CODE_NETWROK_RESOLUTION_DONE,
                                type = NoticeType.TOAST,
                                obj = resolution
                            )
                        )
                    }
                }
                Constants.NOTIFY_ENCRYPTION_MODE_CHANGED -> {
                    val config = appConfigRepository.getAppConfig() ?: spLite.getAppConfig()
                    config?.app?.encryptionMode?.let {
                        config.app.encryptionMode = encryptionMode
                        appConfigRepository.saveAppConfig(config)
                    }
                }
                Constants.NOTIFY_INCOMING_NEW -> {
                    val rid = map[Constants.KEY_ROOM_RID]
                    val pwd = map[Constants.KEY_ROOM_PWD]
                    val ticket = map[Constants.KEY_ROOM_TICKET]
                    val avatar = map[Constants.KEY_SOURCE_AVATAR]
                    val requestId = map[Constants.KEY_REQUEST_ID]
                    pushManager.onIncomingNew(IncomingData().apply {
                        this.rid = rid
                        this.pwd = pwd
                        this.ticket = ticket
                        this.inviterName = sourceName
                        this.inviterAvatar = avatar
                        this.requestId = requestId
                    })
                }
                Constants.NOTIFY_INCOMING_CANCENLED -> {
                    pushManager.cancelIncoming()
                }
                Constants.NOTIFY_SERVER_REQUEST_UPLOAD_LOG -> {
                    uploadLog()
                }
                Constants.NOTIFY_ME_KICKED_OUT -> {
                    mediaPlayer.playBeKickOut()
                }
                Constants.NOTIFY_ME_KICKED_OUT_BY_SERVER -> {
                    mediaPlayer.playBeKickOut()
                }
                Constants.NOTIFY_ME_KICKED_OUT_BY_ASSISTANT -> {
                    mediaPlayer.playBeKickOut()
                }
            }
            noticeManager.onNotify(type, sourceName, targetName)
        }

        /**
         * users take the initiative to operate, the result is notified
         */
        override fun onNotifyResult(type: Int, success: Boolean, code: Int, seq: Int) {
            Logger.w(
                TAG,
                "onNotifyResult response, type:$type, success:$success, code:$code, seq:$seq"
            )
            when (type) {
                Constants.NOTIFY_APPLY_ASSISTANT_OPERATION_RESULT -> {
                    notifyObservers(
                        MessageEvent(
                            AppEvent.APPLY_ASSISTANT_OPERATION_RESULT.ordinal,
                            success
                        )
                    )
                }
                Constants.NOTIFY_INVITE_CONTACTS_OPERATION_RESULT -> {
                    if (currentCallSeq == seq) {
                        notifyObservers(
                            MessageEvent(
                                AppEvent.CALL_RESULT_EVENT.ordinal,
                                NotifyResult(type, success, code, seq)
                            )
                        )
                    }
                }
                Constants.NOTIFY_PICK_UP_INCOMING_OPERATION_RESULT -> {
                    pushManager.answerIncoming(type, success, code, seq)
                }
                Constants.NOTIFY_INVITE_CONTACTS_OPERATION_RESULT,
                Constants.NOTIFY_PICK_UP_INCOMING_OPERATION_RESULT -> {
                    if (code == Constants.NOTIFY_INCOMING_LOGIN_EXPIRED) {
                        Logger.e(TAG, "incoming login expired")
                        handleLoginExpired()
                    }
                }
            }
            noticeManager.onNotifyResult(type, success, code, seq)
        }

        override fun onMosResult(
            rid: String?,
            callId: String,
            localUser: SimpleRoomUser?,
            localMos: Int,
            videoMos: Int,
            users: Array<SimpleRoomUser>?,
            usersMos: IntArray?
        ) {
            Logger.i(
                TAG,
                "onMosResult response, rid:$rid, localUser:$localUser, localMos:$localMos"
            )
            if (rid == null) {
                Logger.e(TAG, "failed to resp mos result, due to rid is null")
                return
            }
            val user = localUser?.toARoomUser()
            if (user == null) {
                Logger.e(TAG, "failed to resp mos result, due to local user is null")
                return
            }
            val local = kotlin.Pair(user, localMos)
            val remote = arrayListOf<kotlin.Pair<ARoomUser, Int>>()
            users?.forEachIndexed { index, usr ->
                if (usersMos != null) {
                    remote.add(kotlin.Pair(usr.toARoomUser(), usersMos[index]))
                }
            }
            val audioScore = MosScore(rid, local, remote, videoMos, callId)
            //sdk problem, temporarily trigger the function
            val event = MessageEvent(AppEvent.CALL_RATING_EVENT.ordinal, audioScore)
            notifyObservers(event)
        }

        override fun onAddressBookResult(
            seq: Int,
            operationType: Int,
            resultType: Int,
            content: String?
        ) {
            Logger.i(
                TAG,
                "onAddressBookResult response, seq:$seq, operation:$operationType, resultType:$resultType"
            )
            if (resultType == Constants.AddressBookResultType.FAILURE_LOGIN_EXPIRED.value) {
                handleLoginExpired()
                return
            }

            when (operationType) {
                Constants.AddressBookOperationType.GET.value -> {
                    val node = if (content?.isNotEmpty() == true) {
                        addressBookRepository.analysisGetResponse(content)
                    } else {
                        null
                    }
                    val event = MessageEvent(
                        AppEvent.ADDRESS_BOOK_EVENT.ordinal,
                        AddressBookResponse(operationType, resultType, seq, node)
                    )
                    notifyObservers(event)
                }
                Constants.AddressBookOperationType.SEARCH.value -> {
                    val node = if (content?.isNotEmpty() == true) {
                        addressBookRepository.analysisSearchResponse(content)
                    } else {
                        null
                    }
                    val event = MessageEvent(
                        AppEvent.ADDRESS_BOOK_EVENT.ordinal,
                        AddressBookResponse(operationType, resultType, seq, node)
                    )
                    notifyObservers(event)
                }
                else -> {
                }
            }
        }

        override fun onMeetupOperationResult(
            seq: Int,
            id: String?,
            operationType: Int,
            resultType: Int
        ) {
            Logger.i(
                TAG,
                "onMeetupOperationResult response, seq:$seq, id:$id, ot:$operationType, rt:$resultType"
            )

            if (resultType == Constants.MeetupResultType.FAILURE_LOGIN_EXPIRED.value) {
                handleLoginExpired()
                return
            }

            val eventType = when (operationType) {
                Constants.MeetupOperationType.CREATE.value -> {
                    AppEvent.MEETUP_CREATE_RESULT_EVENT.ordinal
                }
                Constants.MeetupOperationType.UPDATE.value -> {
                    AppEvent.MEETUP_UPDATE_RESULT_EVENT.ordinal
                }
                Constants.MeetupOperationType.DELETE.value -> {
                    AppEvent.MEETUP_DELETE_RESULT_EVENT.ordinal
                }
                else -> {
                    null
                }
            }

            if (eventType == null) {
                Logger.e(TAG, "onMeetupOperationResult response unknown, seq:$seq")
                return
            }

            val data = MeetupResponse(
                operationType,
                resultType,
                seq,
                id
            )
            notifyObservers(MessageEvent(eventType, data))
        }

        override fun onMeetupGetListResult(seq: Int, resultType: Int, content: String?) {
            Logger.i(TAG, "onMeetupGetListResult response, seq:$seq, rt:$resultType, txt:$content")
            val data = try {
                GsonUtils.fromJson<List<ContactGroup>>(
                    content,
                    object : TypeToken<List<ContactGroup>>() {}.type
                )
            } catch (e: Exception) {
                Logger.e(TAG, "failed to analysis meetup list content, seq:$seq")
                null
            }
            notifyObservers(
                MessageEvent(
                    AppEvent.MEETUP_QUERY_LIST_RESULT_EVENT.ordinal,
                    MeetupResponse(
                        AppEvent.MEETUP_QUERY_LIST_RESULT_EVENT.ordinal,
                        resultType,
                        seq,
                        data
                    )
                )
            )
        }

        override fun onMeetupGetDetailResult(seq: Int, resultType: Int, content: String?) {
            Logger.i(
                TAG,
                "onMeetupGetDetailResult response, seq:$seq, rt:$resultType, txt:$content"
            )
            val data = try {
                GsonUtils.fromJson(content, ContactGroup::class.java)
            } catch (e: Exception) {
                Logger.e(TAG, "failed to analysis meetup list content, seq:$seq")
                null
            }
            notifyObservers(
                MessageEvent(
                    AppEvent.MEETUP_QUERY_DETAIL_RESULT_EVENT.ordinal,
                    MeetupResponse(
                        AppEvent.MEETUP_QUERY_DETAIL_RESULT_EVENT.ordinal,
                        resultType,
                        seq,
                        data
                    )
                )
            )
        }

        /**
         * prompt by warning
         * kEchoDetected
         * kSystemSpeakerVolumeTooLow --android no need, do nothing
         * WIFI_DISTANCE_FAR_AWAY
         */
        override fun onWarning(warning: Int) {
            Logger.w(TAG, "onWarning response,warning:$warning")
            when (warning) {
                Constants.ValoranWarning.getValue(Constants.ValoranWarning.ECHO_DETECTED) -> {
                    noticeManager.notice(
                        NoticeEvent(
                            NoticeCode.CODE_ROOM_DEVICE_ECHO,
                            NoticeType.NOTIFICATION_A,
                            MeetingNotificationType.NO_2
                        )
                    )
                }
                Constants.ValoranWarning.getValue(Constants.ValoranWarning.WIFI_DISTANCE_FAR_AWAY) -> {
                    noticeManager.notice(
                        NoticeEvent(
                            NoticeCode.CODE_WIFI_AP,
                            NoticeType.NOTIFICATION_C,
                            MeetingNotificationType.WIFI_AP
                        )
                    )
                }
                else -> {
                }
            }
        }

        /**
         * prompt by error,maybe need switch status
         */
        override fun onError(error: Int) {
            Logger.e(TAG, "onError response, error:$error")

        }

        /********************* changed user data  start ***********************/

        /**
         */
        override fun onMajorMediaChange(user: RoomUser?, reason: Int) {
            if (isLog(reason)) {
                Logger.i(TAG, "onMajorMediaChange response, user:$user, reason:$reason")
            }
            if (user == null) {
                Logger.e(TAG, "onMajorMediaChange received room user is empty")
                return
            }
            val majorMediaUser = if (isMySelf(user)) {
                updateLocalUser(user)
            } else {
                user.toARoomUser()
            }
            if (majorMediaUser == null) {
                Logger.e(TAG, "onMajorMediaChange received major media user is empty")
                return
            }
            roomRepository.saveMajorMedia(majorMediaUser)
            val event = MessageEvent(AppEvent.MAJOR_USER_CHANGED.ordinal)
            notifyObservers(event)
        }

        /**
         */
        override fun onMediaListItemAdd(position: Int, user: RoomUser?) {
            Logger.i(TAG, "onMediaListItemAdd response,pos:$position, user:$user")
            if (user == null) {
                Logger.e(TAG, "onMediaListItemAdd received room user is empty")
                return
            }
            val mediaUser = if (isMySelf(user)) {
                updateLocalUser(user)
            } else {
                user.toARoomUser()
            }
            if (mediaUser == null) {
                Logger.e(TAG, "onMediaListItemAdd received media user is empty")
                return
            }
            roomRepository.addMediaItem(position, mediaUser)
            val event = MessageEvent(AppEvent.USER_LIST_CHANGED.ordinal)
            notifyObservers(event)
        }

        override fun onMediaListItemRemove(position: Int) {
            Logger.i(TAG, "onMediaListItemRemove response,pos:$position")
            roomRepository.removeMediaAt(position)
            val event = MessageEvent(AppEvent.USER_LIST_CHANGED.ordinal)
            notifyObservers(event)
        }

        /**
         */
        override fun onMediaListItemMove(from: Int, to: Int, user: RoomUser?, reason: Int) {
            if (isLog(reason)) {
                Logger.i(
                    TAG,
                    "onMediaListItemMove response, from:$from, to:$to, user:$user, reason:$reason"
                )
            }
            if (user == null) {
                Logger.e(TAG, "onMediaListItemMove received room user is empty")
                return
            }
            val mediaUser = if (isMySelf(user)) {
                updateLocalUser(user)
            } else {
                user.toARoomUser()
            }
            if (mediaUser == null) {
                Logger.e(TAG, "onMediaListItemMove received media user is empty")
                return
            }
            roomRepository.moveMediaItem(from, to, mediaUser)
            val event = MessageEvent(AppEvent.USER_LIST_CHANGED.ordinal)
            notifyObservers(event)
        }

        /**
         */
        override fun onMediaListItemChange(position: Int, user: RoomUser?, reason: Int) {
            if (isLog(reason)) {
                Logger.i(
                    TAG,
                    "onMediaListItemChange response, pos:$position, user:$user, reason:$reason"
                )
            }
            if (user == null) {
                Logger.e(TAG, "onMediaListItemChange received room user is empty")
                return
            }
            val mediaUser = if (isMySelf(user)) {
                updateLocalUser(user)
            } else {
                user.toARoomUser()
            }
            if (mediaUser == null) {
                Logger.e(TAG, "onMediaListItemChange received media user is empty")
                return
            }
            roomRepository.changeMediaListItem(position, mediaUser)
            val event = MessageEvent(AppEvent.USER_INFO_CHANGED.ordinal)
            notifyObservers(event)
        }

        override fun onMediaUsersCount(count: Int) {
            Logger.i(TAG, "onMediaUsersCount response, count:$count")

        }

        /**
         */
        override fun onUserListSetup(users: Array<RoomUser>?) {
            Logger.i(TAG, "onUserListSetup response,users:$users")
            val aRoomUserList = arrayListOf<ARoomUser>()
            users?.forEach { user ->
                val roomUser = if (isMySelf(user)) {
                    updateLocalUser(user)
                } else {
                    user.toARoomUser()
                }
                if (roomUser == null) {
                    Logger.e(TAG, "onUserListSetup can not get localUser")
                } else {
                    aRoomUserList.add(roomUser)
                }
            }
            roomRepository.setupRoomUserList(aRoomUserList)
            val event = MessageEvent(AppEvent.USER_LIST_CHANGED.ordinal)
            notifyObservers(event)
        }

        private fun isMySelf(user: RoomUser): Boolean {
            return getLocalUser()?.streamId != null && getLocalUser()?.streamId == user.streamId
        }

        private fun isMySelf(user: SimpleRoomUser): Boolean {
            return getLocalUser()?.streamId != null && getLocalUser()?.streamId == user.streamId
        }

        /**
         */
        override fun onUserListRangeAdd(position: Int, users: Array<RoomUser>?) {
            Logger.i(TAG, "onUserListRangeAdd request, pos:$position, users:${users?.size}")
            val aRoomUserList = arrayListOf<ARoomUser>()
            users?.forEach { user ->
                val roomUser = if (isMySelf(user)) {
                    updateLocalUser(user)
                } else {
                    user.toARoomUser()
                }
                if (roomUser == null) {
                    Logger.e(TAG, "onUserListRangeAdd can not get localUser")
                } else {
                    aRoomUserList.add(roomUser)
                }
            }
            roomRepository.addRoomUsers(position, aRoomUserList)
            val event = MessageEvent(AppEvent.USER_LIST_CHANGED.ordinal)
            notifyObservers(event)
        }

        /**
         */
        override fun onUserListItemAdd(position: Int, user: RoomUser?) {
            Logger.i(TAG, "onUserListItemAdd response, pos:$position, user:$user")
            if (user == null) {
                Logger.e(TAG, "onUserListItemAdd received room user is empty")
                return
            }
            val roomUser = if (isMySelf(user)) {
                updateLocalUser(user)
            } else {
                user.toARoomUser()
            }
            if (roomUser == null) {
                Logger.e(TAG, "onUserListItemAdd received media user is empty")
                return
            }
            roomRepository.addRoomUser(position, roomUser)
            val event = MessageEvent(AppEvent.USER_LIST_CHANGED.ordinal)
            notifyObservers(event)
        }

        override fun onUserListItemRemove(position: Int) {
            Logger.i(TAG, "onUserListItemRemove response, position:$position")
            roomRepository.removeRoomUserAt(position)
            val event = MessageEvent(AppEvent.USER_LIST_CHANGED.ordinal)
            notifyObservers(event)
        }

        /**
         */
        override fun onUserListItemMove(from: Int, to: Int, user: RoomUser?, reason: Int) {
            if (isLog(reason)) {
                Logger.i(TAG, "onUserListItemMove response, reason:$reason")
            }
            if (user == null) {
                Logger.e(TAG, "onUserListItemMove received room user is empty")
                return
            }
            val roomUser = if (isMySelf(user)) {
                updateLocalUser(user)
            } else {
                user.toARoomUser()
            }
            if (roomUser == null) {
                Logger.e(TAG, "onUserListItemMove received media user is empty")
                return
            }
            roomRepository.moveRoomUser(from, to, roomUser)
            val event = MessageEvent(AppEvent.USER_LIST_CHANGED.ordinal)
            notifyObservers(event)
        }

        /**
         */
        override fun onUserListItemChange(position: Int, user: RoomUser?, reason: Int) {
            if (isLog(reason)) {
                Logger.i(
                    TAG,
                    "onUserListItemChange response, pos:$position, user:$user, reason:$reason"
                )
            }
            if (user == null) {
                Logger.e(TAG, "onUserListItemChange received room user is empty")
                return
            }
            val roomUser = if (isMySelf(user)) {
                updateLocalUser(user)
            } else {
                user.toARoomUser()
            }
            if (roomUser == null) {
                Logger.e(TAG, "onUserListItemChange received media user is empty")
                return
            }
            roomRepository.changeRoomUserListItem(position, roomUser)
            val event = MessageEvent(AppEvent.USER_INFO_CHANGED.ordinal)
            notifyObservers(event)
        }

        override fun onUserListAttendeeCount(count: Int) {
            Logger.i(TAG, "onUserListAttendeeCount response, count:$count")
            val event = MessageEvent(AppEvent.USER_LIST_COUNT_CHANGED.ordinal, count)
            notifyObservers(event)
        }
    }

    private fun uploadLog() {
        getLocalUser()?.streamId?.apply {
            logUploader.doUploadLog(this)
        }
    }

    private fun isLog(reason: Int): Boolean {
        if (this.logType == LOG_VERBOSE) {
            return true
        }
        if (this.logType == LOG_IMPORTANT && isImportantReason(reason)) {
            return true
        }
        return false
    }

    private fun isImportantReason(reason: Int): Boolean {
        return (reason and (Constants.REASON_AUDIO or Constants.REASON_VIDEO or Constants.REASON_HOST or Constants.REASON_SHARE or Constants.REASON_INFO or Constants.REASON_INTERRUPT or Constants.REASON_INTERRUPT or Constants.REASON_DUMPING or Constants.REASON_CLOUD_RECORDING or Constants.REASON_STATUS)) != 0
    }

    private fun updateLocalUser(user: SimpleRoomUser): LocalUser? {
        return updateLocalUser(user.toARoomUser())
    }

    private fun updateLocalUser(user: RoomUser): LocalUser? {
        return updateLocalUser(user.toARoomUser())
    }

    override fun updateLocalUser(user: ARoomUser): LocalUser? {
        synchronized(this) {
            userRepository.updateLocalUserBy(user)
            notifyObservers(MessageEvent(AppEvent.LOCAL_USER_CHANGED.ordinal))
            return appContainer?.localUser
        }
    }

    override fun setLocalUser(user: LocalUser) {
        synchronized(this) {
            Logger.i(TAG, "set local user,user:${GsonUtils.toJson(user)}")
            appContainer?.localUser = user
            notifyObservers(MessageEvent(AppEvent.LOCAL_USER_CHANGED.ordinal))
        }
    }

    override fun acceptAssistantRequest(tag: String) {
        acceptRemoteRequest(Constants.RequestType.getValue(Constants.RequestType.ASSISTANT), tag)
    }

    override fun refuseAssistantRequest(tag: String) {
        refuseRemoteRequest(Constants.RequestType.getValue(Constants.RequestType.ASSISTANT), tag)
    }

    override fun acceptMediaRequest(type: Int) {
        acceptRemoteRequest(type, null)
    }

    override fun refuseMediaRequest(type: Int) {
        refuseRemoteRequest(type, null)
    }

    override fun changeRoom(newRoom: Room) {
        if (getRoomInfo() == null) {
            notifyObservers(MessageEvent(AppEvent.CHANGE_ROOM_EVENT.ordinal, newRoom))
        } else {
            this.pendingRoom = newRoom
            leaveRoom()
        }
    }

    override fun handleTokenBad() {
        appConfigRepository.clearToken()
        userRepository.getLocalUser()
            .retryWhen(RetryForever())
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(object : io.reactivex.Observer<LocalUser> {
                override fun onSubscribe(d: Disposable) {

                }

                override fun onNext(t: LocalUser) {
                    userConfigured.set(false)
                    setLocalUser(t)
                    configUser(t.streamId, t.uid, t.token)
                    handleLoginExpired()
                }

                override fun onError(e: Throwable) {
                    userConfigured.set(false)
                    handleLoginExpired()
                }

                override fun onComplete() {

                }
            })
    }

    private fun handleLoginExpired() {
        userRepository.removeInnerInfo()
        notifyObservers(MessageEvent(AppEvent.LOCAL_USER_CHANGED.ordinal, true))
        if (roomRepository.getRoom() == null) {
            cleanResource()
            noticeManager.notice(NoticeEvent(NoticeCode.CODE_JOIN_ROOM_LOGIN_EXPIRED))
        } else {
            leaveRoom()
            noticeManager.notice(
                NoticeEvent(
                    NoticeCode.CODE_JOIN_ROOM_LOGIN_EXPIRED,
                    NoticeType.DIALOG
                )
            )
        }
    }

    override fun getOperationSeqId(): Int {
        return if (operationSeqId.get() + 1 > Int.MAX_VALUE) {
            0
        } else {
            operationSeqId.incrementAndGet()
        }
    }

    override fun addMeetupMembers(currentSeq: Int, source: ArrayList<MemberNode>) {
        val sb = StringBuilder()
        source.forEach { node ->
            if (node == null) {
                return@forEach
            }
            sb.append(node.userid).append(",")
        }
        Logger.i(TAG, "add members request, seq:$currentSeq, add:$sb")
        val data = MeetupResponse(
            AppEvent.MEETUP_ADD_MEMBER_EVENT.ordinal,
            Constants.MeetupResultType.SUCCESS.value,
            currentSeq,
            source
        )
        notifyObservers(MessageEvent(AppEvent.MEETUP_ADD_MEMBER_EVENT.ordinal, data))
    }

    /********************* changed user data  end ***********************/

    private fun setRoomInfo(roomInfo: Room?) {
        appContainer?.room = roomInfo
    }

    private fun getLocalUser(): LocalUser? {
        return appContainer?.localUser
    }

    private fun getRoomInfo(): Room? {
        return appContainer?.room
    }

    private fun notifyObservers(event: MessageEvent) {
        appEventBus.notifyObservers(event)
    }

    companion object {
        private const val TAG = "[Comm][AppController]"
        private const val LOG_VERBOSE = 0
        private const val LOG_IMPORTANT = 1
    }
}