package io.agora.avc.manager.notice.notification

import android.app.Application
import android.os.SystemClock
import android.text.TextUtils
import io.agora.avc.MyApplication
import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.AppEventBus
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.bo.MeetingNotification
import io.agora.avc.bo.MeetingNotificationType
import io.agora.avc.bo.NoticeCode
import io.agora.avc.config.KEY_REQUEST_ID
import io.agora.avc.manager.notice.NoticeType
import io.agora.avc.repository.NotificationRepository
import io.agora.avc.repository.RoomRepository
import io.agora.logger.Logger
import io.reactivex.Observable
import io.reactivex.Observer
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import java.util.concurrent.TimeUnit
import javax.inject.Inject

class NotificationManagerImpl @Inject constructor(
    private val application: Application,
    private val notificationRepository: NotificationRepository,
    private val roomRepository: RoomRepository,
    private val appEventBus: AppEventBus,
) : NotificationManager {

    private var disappearDisposable: Disposable? = null

    override fun onEventReceived(arg: MessageEvent) {
        when (arg.type) {
            AppEvent.NOTIFY_EVENT.ordinal -> {
                val notification = arg.obj
                if (notification is MeetingNotification) {
                    Logger.i(
                        TAG,
                        "notification changed event, notification:$notification"
                    )
                    if (notification.show) {
                        put(notification)
                    } else {
                        val requestId = notification.data?.getString(KEY_REQUEST_ID)
                        when {
                            !TextUtils.isEmpty(requestId) -> {
                                remove(requestId!!)
                            }
                            notification.code != NoticeCode.NOT_CODE -> {
                                remove(notification.code)
                            }
                            notification.group != null -> {
                                notification.group.forEach {
                                    remove(it)
                                }
                            }
                            else -> {
                                remove(notification.notificationType)
                            }
                        }
                    }
                }
            }
            AppEvent.APPLY_ASSISTANT_OPERATION_RESULT.ordinal -> {
                (arg.obj as? Boolean)?.let {
                    if (it) {
                        remove(MeetingNotificationType.NO_8)
                    }
                }
            }
            AppEvent.ROOM_INFO_CHANGED.ordinal -> {
                //I'm give up host, need hide request apply assistant notification
                val roomInfo = roomRepository.getRoom()
                if (roomInfo?.hostUid == (application as MyApplication).appContainer.localUser?.uid
                    && roomInfo?.hasHost() != true
                ) {
                    remove(NoticeCode.CODE_AS_ROOM_ASK_HOST_CN)
                    remove(NoticeCode.CODE_AS_ROOM_ASK_HOST_EN)
                }
            }
            else -> {
            }
        }
    }

    override fun put(meetingNotification: MeetingNotification) {
        meetingNotification.showTime = SystemClock.elapsedRealtime()

        var hasTheSameType = false
        notificationRepository.getNotificationList().forEachIndexed { index, item ->
            if (item.notificationType == meetingNotification.notificationType) {
                notificationRepository.set(index, meetingNotification)
                hasTheSameType = true
                return@forEachIndexed
            }
        }
        if (!hasTheSameType) {
            notificationRepository.add(meetingNotification)
        }

        if (!disappearTaskIsRunning()) {
            runDisappearTask()
        }

        postNotification()
    }

    override fun remove(meetingNotificationType: Int, notify: Boolean) {
        notificationRepository.remove(meetingNotificationType)
        if (notify) {
            postNotification()
        }
    }

    override fun remove(code: NoticeCode, notify: Boolean) {
        notificationRepository.remove(code)
        if (notify) {
            postNotification()
        }
    }

    override fun remove(requestId: String, notify: Boolean) {
        notificationRepository.remove(requestId)
        if (notify) {
            postNotification()
        }
    }

    override fun removeAt(position: Int, notify: Boolean) {
        notificationRepository.removeAt(position)
        if (notify) {
            postNotification()
        }
    }

    private fun postNotification() {
        appEventBus.notifyObservers(MessageEvent(AppEvent.NOTIFICATION_CHANGED_EVENT.ordinal))
    }

    override fun onDestroy() {

    }

    private fun disappearTaskIsRunning() = disappearDisposable != null

    private fun runDisappearTask() {
        Observable
            .interval(0, 1, TimeUnit.SECONDS)
            .subscribeOn(Schedulers.io())
            .subscribe(
                object : Observer<Long> {
                    override fun onComplete() {

                    }

                    override fun onSubscribe(d: Disposable) {
                        disappearDisposable = d
                    }

                    override fun onNext(t: Long) {
                        notificationRepository.getNotificationList().forEach {
                            val timeOut = if (it.noticeType == NoticeType.NOTIFICATION_C) {
                                MeetingNotification.NOTICE_TYPE_C_TIME_OUT
                            } else {
                                MeetingNotification.NOTICE_TYPE_AB_TIME_OUT
                            }
                            if (SystemClock.elapsedRealtime() - it.showTime > timeOut) {
                                notificationRepository.remove(it)
                                postNotification()
                            }
                        }

                        if (notificationRepository.getNotificationList().isEmpty()) {
                            disappearDisposable?.dispose()
                            disappearDisposable = null
                        }
                    }

                    override fun onError(e: Throwable) {
                    }
                }
            )
    }

    companion object {
        private const val TAG = "[COMM][Notification]"
    }
}