package io.agora.avc.manager.rating.video

import android.app.Application
import io.agora.avc.biz.AppController
import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.AppEventBus
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.bo.NoticeCode
import io.agora.avc.extensions.getConferenceNickname
import io.agora.avc.manager.notice.NoticeEvent
import io.agora.avc.manager.notice.NoticeManager
import io.agora.avc.repository.RoomRepository
import io.agora.avc.utils.FileIOUtils
import io.agora.avc.utils.FileUtils2
import io.agora.avc.utils.TimeUtils
import io.agora.logger.LogToFile
import io.agora.logger.Logger
import io.reactivex.Observable
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import java.io.File
import java.io.IOException
import java.util.concurrent.TimeUnit
import javax.inject.Inject

class VideoRatingManagerImpl @Inject constructor(
    private val application: Application,
    private val roomRepository: RoomRepository,
    private val appEventBus: AppEventBus,
    private val appController: AppController,
    private val noticeManager: NoticeManager,
) : VideoRatingManager {

    private var videoRatingTimer: Disposable? = null
    override var videoScore: VideoScore? = null

    override fun toggleVideoRating() {
        if (videoScore == null) {
            val users = roomRepository.queryOtherPublishers()
            if (users == null || users.isEmpty()) {
                Logger.e(TAG, "failed to start video rating, users is empty")
                noticeManager.notice(NoticeEvent(NoticeCode.CODE_VIDEO_RATING_NO_REMOTE_USER))
                return
            }

            val names = arrayOfNulls<String>(users.size)
            users.forEachIndexed { index, user ->
                names[index] = "${user.getConferenceNickname()}(${user.streamId})"
            }
            appEventBus.notifyObservers(MessageEvent(AppEvent.VIDEO_RATING_START.ordinal, users))
        } else {
            stopVideoRating()
        }
    }

    override fun startVideoRating(streamId: Int) {
        val nowMills = TimeUtils.getNowMills()
        val scoreFile =
            LogToFile.getRtcVideoDumpPath() + File.separator + "$streamId-$nowMills.log"
        val dumpPath =
            LogToFile.getRtcVideoDumpPath() + File.separator + "$streamId-$nowMills.264"
        Logger.i(TAG, "start video Rating,it file path:$dumpPath")
        videoScore = VideoScore(
            scorePath = scoreFile,
            videoPath = dumpPath,
            createTime = nowMills
        )
        appController.startDumpVideoReceiveTrack(streamId, dumpPath)
        startVideoRatingCountDown()
    }

    override fun stopVideoRating() {
        Logger.i(TAG, "stop video rating")
        if (videoRatingTimer?.isDisposed == false) {
            videoRatingTimer?.dispose()
        }
        appController.stopDumpVideoReceiveTrack()
        appEventBus.notifyObservers(
            MessageEvent(AppEvent.VIDEO_RATING_FINISH.ordinal, videoScore)
        )
        videoScore = null
    }

    override fun recordVideoScore(videoScore: VideoScore) {
        Logger.i(TAG, "record video score: $videoScore")
        try {
            val scoreFile = File(videoScore.scorePath)
            if (FileUtils2.isFileExists(scoreFile)) {
                FileUtils2.delete(scoreFile)
            }
            if (!FileUtils2.createOrExistsFile(scoreFile)) {
                Logger.e(TAG, "video rating score file create failure!")
                return
            }
            Logger.i(TAG, "video rating score file create success: ${scoreFile.absolutePath}")
            FileIOUtils.writeFileFromString(scoreFile, "${videoScore.score?.value}", true)
        } catch (e: IOException) {
            Logger.e(TAG, "video rating score write failure", e)
        } catch (e: Exception) {
            Logger.e(TAG, "video rating score write exception", e)
        }
    }

    private fun startVideoRatingCountDown() {
        videoScore?.let { _videoScore ->
            _videoScore.createTime?.let { createTime ->
                Observable
                    .interval(0, 1, TimeUnit.SECONDS)
                    .take(VIDEO_RATING_TIME)
                    .subscribeOn(Schedulers.io())
                    .subscribe(object : io.reactivex.Observer<Long> {
                        override fun onComplete() {
                            stopVideoRating()
                        }

                        override fun onSubscribe(d: Disposable) {
                            Logger.i(TAG, "start video rating")
                            videoRatingTimer = d
                        }

                        override fun onNext(t: Long) {
                            val time = TimeUtils.getNowMills() - createTime
                            appEventBus.notifyObservers(
                                MessageEvent(AppEvent.VIDEO_RATING_TIME.ordinal, time)
                            )
                        }

                        override fun onError(e: Throwable) {

                        }
                    })
            }
        }
    }

    companion object {
        private const val TAG = "[COMM][VideoRatingManager]"
        private const val VIDEO_RATING_TIME: Long = 60L

    }
}