package io.agora.avc.manager.upgrade

import android.app.Application
import android.text.TextUtils
import io.agora.avc.BuildConfig
import io.agora.avc.bo.AppVersion
import io.agora.avc.bo.Latest
import io.agora.avc.manager.splite.SPLiteProxy
import io.agora.avc.net.download.Downloader
import io.agora.avc.repository.AppConfigRepository
import io.agora.frame.http.progress.ProgressListener
import io.agora.logger.Logger
import java.util.concurrent.atomic.AtomicBoolean
import javax.inject.Inject
import kotlin.math.min

class AppUpgradeManagerImpl @Inject constructor(
    private val application: Application,
    private val appConfigRepository: AppConfigRepository,
) : AppUpgradeManager {

    private var forceUpdate: AtomicBoolean = AtomicBoolean(false)
    private var downloader: Downloader? = null

    override fun hasNewVersion(
        appVersion: AppVersion?,
        localVersion: String,
        filter: Boolean
    ): Boolean {
        var result = false
        if (appVersion == null) {
            return result
        }

        val minVersion = appVersion?.minimum
        val latestVersion = appVersion?.latest

        if (minVersion != null && latestVersion != null) {
            if (getUpgradeUrl(latestVersion) == null) {
                return result
            }

            val remoteMinimum = minVersion.v
            val remoteLatest = latestVersion.v

            if (compare(remoteMinimum, localVersion)) {
                result = true
            } else if (compare(remoteLatest, localVersion)
                && (filter || remoteLatest != SPLiteProxy.getLastVersion())
            ) {
                remoteLatest?.let {
                    SPLiteProxy.setLastVersion(it)
                }
                result = true
            }
        }

        return result
    }

    override fun isForceUpgrade(appVersion: AppVersion?, localVersion: String): Boolean {
        var result = false
        if (appVersion == null) {
            return result
        }

        val minVersion = appVersion?.minimum
        val latestVersion = appVersion?.latest

        if (minVersion != null && latestVersion != null) {
            if (getUpgradeUrl(latestVersion) == null) {
                return result
            }

            val remoteMinimum = minVersion.v
            val remoteLatest = latestVersion.v
            if (compare(remoteMinimum, localVersion)) {
                result = true
            }
        }

        return result
    }

    override fun showNoticeIfNeed(msg: String): Boolean {
        if (!TextUtils.isEmpty(msg) && msg != SPLiteProxy.getNotice()) {
            SPLiteProxy.setNotice(msg)
            return true
        }
        return false
    }

    override fun getUpgradeUrl(latestVersion: Latest): String? {
        return ""
//        return when (BuildConfig.FLAVOR) {
//            "google" -> {
//                latestVersion.gp_url
//            }
//            else -> {
//                latestVersion.url
//            }
//        }
    }

    override fun downloadApk(
        url: String,
        version: String,
        forceUpdate: Boolean,
        downloadDelay: Long,
        progressListener: ProgressListener?,
        completeListener: Downloader.CompleteListener?,
        errorListener: Downloader.ErrorListener?
    ) {
        val prev = this.forceUpdate.get()
        if (prev != forceUpdate) {
            this.forceUpdate.compareAndSet(prev, forceUpdate)
        }
        val fileName = "avc-latest-$version.apk"
        //todo waiting to verify the validity of the apk
//        val fileDir = FileUtils.getFileDir(Downloader.DIR_NAME)
//        val absolutePath = File(fileDir.absolutePath + File.separator + fileName).absolutePath
//        val isApkExists = FileUtils2.isFileExists(absolutePath)
//        if (isApkExists) {
//            Logger.i(TAG, "apk-$version existed")
//            completeListener?.onComplete(absolutePath)
//            return
//        }
        if (downloader == null) {
            downloader = Downloader.build {
                this.ctx = application
                this.url = url
                this.fileName = fileName
                this.progressListener = ProgressListener { currentBytes, contentLength, done ->
                    progressListener?.onProgress(currentBytes, contentLength, done)
                }
                this.completeListener = object : Downloader.CompleteListener {
                    override fun onComplete(path: String) {
                        completeListener?.onComplete(path)
                    }
                }
                this.errorListener = errorListener
            }
        }

        if (downloader != null && downloader?.canDownload() == true) {
            downloader?.start(downloadDelay)
        } else {
            Logger.w(TAG, "downloading apk")
        }
    }

    override fun forcingUpgrade(): Boolean {
        val executor = downloader
        return executor != null && !executor.canDownload() && forceUpdate.get()
    }

    private fun compare(version1: String?, version2: String?): Boolean {
        try {
            if (version1?.isNotEmpty() != true || version2?.isNotEmpty() != true) {
                return false
            }

            val splitVersion1 =
                version1.split("\\.".toRegex()).dropLastWhile { it.isEmpty() }.toTypedArray()
            val splitVersion2 =
                version2.split("\\.".toRegex()).dropLastWhile { it.isEmpty() }.toTypedArray()

            val size = min(splitVersion1.size, splitVersion2.size)

            for (i in 0 until size) {
                val diff = splitVersion2[i].toInt() - splitVersion1[i].toInt()
                if (diff == 0) {
                    continue
                }

                if (diff > 0) {
                    return false
                } else if (diff < 0) {
                    return true
                }
            }

            if (splitVersion2.size >= splitVersion1.size) {
                return false
            }
        } catch (e: Exception) {
            Logger.e(
                TAG,
                "app upgrade error,because remote version format error",
                e
            )
            return false
        }

        return true
    }

    companion object {
        private const val TAG = "[COMM][AppUpgradeManager]"
        private const val UPGRADE_TYPE = 0
    }
}