package io.agora.avc.net.download

import android.content.Context
import io.agora.avc.net.api.ApiService
import io.agora.avc.net.converter.CustomConverter
import io.agora.avc.utils.FileIOUtils
import io.agora.avc.utils.FileUtils
import io.agora.avc.utils.FileUtils2
import io.agora.frame.http.interceptor.ProgressInterceptor
import io.agora.frame.http.progress.ProgressListener
import io.agora.logger.Logger
import io.reactivex.Observable
import io.reactivex.Observer
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import okhttp3.OkHttpClient
import okhttp3.ResponseBody
import retrofit2.Retrofit
import retrofit2.adapter.rxjava2.RxJava2CallAdapterFactory
import java.io.File
import java.util.concurrent.TimeUnit

class Downloader(
    private val ctx: Context,
    private val connectTimeout: Long,
    private val writeTimeout: Long,
    private val readTimeout: Long,
    private val url: String,
    private val fileName: String,
    private val progressListener: ProgressListener?,
    private val completeListener: CompleteListener?,
    private var errorListener: ErrorListener? = null,
) {

    private var executor: Disposable? = null

    private constructor(builder: Builder) : this(
        builder.ctx,
        builder.connectTimeout,
        builder.writeTimeout,
        builder.readTimeout,
        builder.url,
        builder.fileName,
        builder.progressListener,
        builder.completeListener,
        builder.errorListener,
    )

    fun start(delay: Long = 0) {
        Logger.i(TAG, "start downloading apk:$url")
        val build = OkHttpClient.Builder()
            .connectTimeout(connectTimeout, TimeUnit.SECONDS)
            .writeTimeout(writeTimeout, TimeUnit.SECONDS)
            .readTimeout(readTimeout, TimeUnit.SECONDS)
            .addInterceptor(ProgressInterceptor { currentBytes, contentLength, done ->
                progressListener?.onProgress(currentBytes, contentLength, done)
                val progress = (currentBytes.toFloat() / contentLength).times(100).toInt()
                DownloadNotificationService.startService(ctx, progress)
            })
            .build()

        val retrofit = Retrofit.Builder()
            .addCallAdapterFactory(RxJava2CallAdapterFactory.create())
            .addConverterFactory(CustomConverter.create())
            .baseUrl(BASE_URL)
            .client(build)
            .build()

        Observable.just(1)
            .delay(delay, TimeUnit.MILLISECONDS)
            .flatMap {
                retrofit.create(ApiService::class.java)
                    .downloadApk(url)
            }
            .doOnNext { body ->
                val fileDir = FileUtils.getFileDir(DIR_NAME)
                val file = File(fileDir.absolutePath + File.separator + fileName)
                FileUtils2.createOrExistsDir(fileDir)
                FileIOUtils.writeFileFromIS2(file, body.byteStream(), false, null)
            }
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(object : Observer<ResponseBody> {
                override fun onSubscribe(d: Disposable) {
                    executor = d
                }

                override fun onNext(t: ResponseBody) {

                }

                override fun onError(e: Throwable) {
                    Logger.e(TAG, "failed to download apk:$url", e)
                    DownloadNotificationService.stopService(ctx)
                    errorListener?.onError(e)
                }

                override fun onComplete() {
                    Logger.i(TAG, "successfully to download apk:$url")
                    DownloadNotificationService.stopService(ctx)
                    completeListener?.onComplete(getDownloadPath())
                }
            })
    }

    private fun getDownloadPath(): String {
        val fileDir = FileUtils.getFileDir(DIR_NAME)
        return File(fileDir.absolutePath + File.separator + fileName).absolutePath
    }

    fun canDownload(): Boolean {
        return executor == null || executor?.isDisposed == true
    }

    class Builder {
        lateinit var ctx: Context
        val connectTimeout: Long = 15L
        val writeTimeout: Long = 15L
        val readTimeout: Long = 15L
        lateinit var url: String
        lateinit var fileName: String
        var progressListener: ProgressListener? = null
        var completeListener: CompleteListener? = null
        var errorListener: ErrorListener? = null

        fun build() = Downloader(this)
    }

    interface CompleteListener {
        fun onComplete(path: String)
    }

    interface ErrorListener {
        fun onError(e: Throwable)
    }

    companion object {
        const val TAG: String = "Downloader"
        const val DIR_NAME: String = "download"
        const val BASE_URL: String = "https://download.agora.io/"
        inline fun build(block: Builder.() -> Unit) = Builder().apply(block).build()
    }
}