package io.agora.avc.net.interceptor

import android.content.Context
import io.agora.avc.MyApplication
import io.agora.avc.bo.ErrorCode
import io.agora.avc.net.bean.BaseNo
import io.agora.avc.utils.GsonUtils
import io.agora.logger.Logger
import okhttp3.Interceptor
import okhttp3.Response
import okio.Buffer
import java.io.EOFException
import java.nio.charset.Charset
import java.nio.charset.StandardCharsets
import java.nio.charset.UnsupportedCharsetException

/**
 * detected bad token
 */
class TokenBadInterceptor(val context: Context) : Interceptor {
    override fun intercept(chain: Interceptor.Chain): Response {
        val response: Response = try {
            chain.proceed(chain.request())
        } catch (e: Exception) {
            throw e
        }

        val responseBody = response.body
        val contentLength = responseBody?.contentLength() ?: 0L
        val source = responseBody?.source()
        source?.request(Long.MAX_VALUE) // Buffer the entire body.
        val buffer = source?.buffer() ?: return response
        var charset: Charset = StandardCharsets.UTF_8
        responseBody?.contentType()?.let { contentType ->
            try {
                charset = contentType.charset(StandardCharsets.UTF_8) ?: StandardCharsets.UTF_8
            } catch (e: UnsupportedCharsetException) {
                return response
            }
        }
        if (!isPlaintext(buffer)) {
            return response
        }
        if (contentLength != 0L) {
            var baseNo: BaseNo? = null
            try {
                val json = buffer.clone().readString(charset)
                baseNo = GsonUtils.fromJson(json, BaseNo::class.java)
            } catch (e: Exception) {
                e.printStackTrace()
            }
            if (baseNo != null && (baseNo.code == ErrorCode.SERVER_400 || baseNo.code == ErrorCode.SERVER_2048)) {
                if (context is MyApplication) {
                    context.appContainer.appController?.apply {
                        Logger.e(TAG, "bad token detected")
                        handleTokenBad()
                    }
                }
            }
        }
        return response
    }

    private fun isPlaintext(buffer: Buffer): Boolean {
        return try {
            val prefix = Buffer()
            val byteCount = if (buffer.size < 64) buffer.size else 64
            buffer.copyTo(prefix, 0, byteCount)
            for (i in 0..15) {
                if (prefix.exhausted()) {
                    break
                }
                val codePoint = prefix.readUtf8CodePoint()
                if (Character.isISOControl(codePoint) && !Character.isWhitespace(codePoint)) {
                    return false
                }
            }
            true
        } catch (e: EOFException) {
            false // Truncated UTF-8 sequence.
        }
    }

    companion object {
        private const val TAG = "TokenBad"
    }
}