package io.agora.avc.push

import android.content.Context
import com.agora.valoran.bean.IncomingData
import io.agora.avc.MyApplication
import io.agora.avc.R
import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.AppEventBus
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.manager.media.MediaPlayer
import io.agora.avc.manager.notice.NotifyResult
import io.agora.avc.manager.upgrade.AppUpgradeManager
import io.agora.avc.push.notification.CallingNotification
import io.agora.logger.Logger
import io.reactivex.Observable
import io.reactivex.Observer
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import java.util.*
import java.util.concurrent.ConcurrentHashMap
import java.util.concurrent.TimeUnit

abstract class BasePushManager(
    private val dataRepository: PushDataRepository,
    private val appEventBus: AppEventBus,
    private val mediaPlayer: MediaPlayer,
    private val appUpgradeManager: AppUpgradeManager,
) : PushManager {
    private var monitor: Disposable? = null
    private var incomingCall: IncomingData? = null
    private val notificationStore by lazy {
        ConcurrentHashMap<String, Int>()
    }

    private val nc by lazy {
        CallingNotification(getContext())
    }

    override fun onIncomingNew(incomingData: IncomingData) {
        Observable
            .create<Long> {
                val record = dataRepository.queryIncomingRecord(incomingData.requestId)
                val forcingUpgrade = appUpgradeManager.forcingUpgrade()
                if (record == null && !forcingUpgrade) {
                    it.onNext(dataRepository.saveIncomingRecord(incomingData))
                } else if (forcingUpgrade) {
                    Logger.w(TAG, "forcing upgrade, ignore request:${incomingData.requestId}")
                    it.onComplete()
                } else {
                    Logger.w(TAG, "ignore duplicate request:${incomingData.requestId}")
                    it.onComplete()
                }
            }
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(object : Observer<Long> {
                override fun onSubscribe(d: Disposable) {

                }

                override fun onNext(t: Long) {
                    Logger.i(TAG, "successfully to save request:${incomingData.requestId}")
                    val mnt = monitor
                    if (mnt != null && !mnt.isDisposed) {
                        Logger.w(TAG, "ignore latest request:${incomingData.requestId}")
                        return
                    }
                    val ctx = getContext()
                    if (ctx is MyApplication) {
                        val room = ctx.appContainer.room
                        if (room != null && room.name == incomingData.rid) {
                            Logger.w(TAG, "already in the room:${room.name}")
                            return
                        }
                    }
                    incomingCall = incomingData
                    startRing(incomingData)
                    val event = MessageEvent(AppEvent.INCOMING_CALL_EVENT.ordinal, incomingData)
                    appEventBus.notifyObservers(event)
                }

                override fun onError(e: Throwable) {
                    Logger.e(TAG, "failed to analysis incoming new", e)
                }

                override fun onComplete() {

                }
            })
    }

    private fun startRing(incomingData: IncomingData) {
        Observable
            .interval(0, TIME_BASE, TimeUnit.MILLISECONDS)
            .take(TIME_COUNT + 1)
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(object : Observer<Long> {
                override fun onComplete() {
                    incomingCall = null
                    mediaPlayer.stopMusic()
                    val event = MessageEvent(AppEvent.INCOMING_CALL_TIME_OUT.ordinal)
                    appEventBus.notifyObservers(event)
                }

                override fun onSubscribe(d: Disposable) {
                    monitor = d
                    playRingIfNeed()
                }

                override fun onNext(t: Long) {
                    val appForeground = MyApplication.isAppForeground()
                    if (appForeground) {
                        playRingIfNeed()
                        nc.cancel(getNotificationId(incomingData))
                    } else {
                        nc.notifyCalling(getNotificationId(incomingData), incomingData)
                    }
                }

                override fun onError(e: Throwable) {
                    Logger.e(TAG, "ring timer error", e)
                    incomingCall = null
                    mediaPlayer.stopMusic()
                }
            })
    }

    private fun getNotificationId(incomingData: IncomingData): Int {
        var notificationId = notificationStore[incomingData.requestId]
        if (notificationId == null) {
            val calendar = Calendar.getInstance()
            val day = calendar.get(Calendar.DAY_OF_MONTH)
            val hour = calendar.get(Calendar.HOUR_OF_DAY)
            val minute = calendar.get(Calendar.MINUTE)
            val second = calendar.get(Calendar.SECOND)
            notificationId = "$day$hour$minute$second".toInt()
            notificationStore[incomingData.requestId] = notificationId
        }
        return notificationId
    }

    override fun rejectIncoming() {
        unsubscribeMonitor()
        mediaPlayer.stopMusic()
        val event = MessageEvent(AppEvent.INCOMING_REJECT_EVENT.ordinal)
        appEventBus.notifyObservers(event)
    }

    override fun cancelIncoming() {
        unsubscribeMonitor()
        mediaPlayer.stopMusic()
        val event = MessageEvent(AppEvent.INCOMING_CANCEL_EVENT.ordinal)
        appEventBus.notifyObservers(event)
    }

    override fun answerIncoming(type: Int, success: Boolean, code: Int, seq: Int) {
        unsubscribeMonitor()
        mediaPlayer.stopMusic()
        val event = MessageEvent(
            AppEvent.INCOMING_ANSWER_EVENT.ordinal,
            NotifyResult(type, success, code, seq)
        )
        appEventBus.notifyObservers(event)
    }

    private fun unsubscribeMonitor() {
        monitor?.apply {
            if (!isDisposed) {
                dispose()
            }
        }
    }

    fun getContext(): Context {
        return MyApplication.appContext
    }

    override fun triggerIncomingCall() {
        Logger.i(TAG, "trigger call monitor:${monitor?.isDisposed} incomingCall:$incomingCall")
        monitor?.apply {
            if (!isDisposed && incomingCall != null) {
                val event = MessageEvent(AppEvent.INCOMING_CALL_EVENT.ordinal, incomingCall)
                appEventBus.notifyObservers(event)
            }
        }
    }

    override fun receiveIncomingCall(data: IncomingData) {
        Logger.i(TAG, "receive call:${data.requestId}")
        if (incomingCall != null && incomingCall?.requestId == data.requestId) {
            this.incomingCall = null
        }
    }

    private fun playRingIfNeed() {
        if (mediaPlayer.isPlaying(R.raw.ring)) {
            return
        }
        if (MyApplication.isAppForeground()
            || nc.isNeedPlaySound()
        ) {
            mediaPlayer.playMusic(R.raw.ring, true)
        }
    }

    abstract fun doIncomingNew(incomingData: IncomingData)

    companion object {
        private const val TAG = "BasePushManager"
        private const val TIME_COUNT = 60L
        private const val TIME_BASE = 500L
    }
}