package io.agora.avc.push.notification

import android.app.NotificationChannel
import android.app.NotificationManager
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.os.Build
import androidx.core.app.NotificationCompat
import androidx.core.content.ContextCompat
import androidx.core.os.bundleOf
import com.agora.valoran.bean.IncomingData
import io.agora.avc.R
import io.agora.avc.app.master.AgoraActivity
import io.agora.avc.notification.BaseNotification
import io.agora.avc.push.*

class CallingNotification(
    private val ctx: Context,
) : BaseNotification(CHANNEL_ID, ctx) {

    private val builder by lazy {
        getNotificationBuilder()
    }

    fun notifyCalling(notificationId: Int, data: IncomingData) {
        setNotificationContent(data)
        notificationManager.notify(notificationId, builder.build())
    }

    private fun createNotificationChannel() {
        // Create the NotificationChannel, but only on API 26+ because
        // the NotificationChannel class is new and not in the support library
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            val name = ctx.getString(R.string.notification_name_calling)
            val descriptionText = ctx.getString(R.string.notification_des_calling)
            val importance = NotificationManager.IMPORTANCE_HIGH
            val channel = NotificationChannel(CHANNEL_ID, name, importance).apply {
                description = descriptionText
                setSound(null, null)
                enableVibration(false)
                enableLights(false)
            }
            notificationManager.createNotificationChannel(channel)
        }
    }

    private fun setNotificationContent(data: IncomingData) {
        val title = data.inviterName
        val content = ctx.getString(R.string.invite_in_avc, data.rid)
        builder.setContentTitle(title)
        builder.setContentText(content)
        builder.setContentIntent(getContentIntent(ctx, data))
    }

    private fun getNotificationBuilder(): NotificationCompat.Builder {
        createNotificationChannel()

        return NotificationCompat
            .Builder(ctx, CHANNEL_ID)
            .setSmallIcon(createSmallIcon())
            .setColor(ContextCompat.getColor(ctx, R.color.color_primary))
            .setStyle(NotificationCompat.DecoratedCustomViewStyle())
            .setSound(null)
            .setPriority(NotificationCompat.PRIORITY_MAX)
            .setCategory(NotificationCompat.CATEGORY_CALL)
            .setVisibility(NotificationCompat.VISIBILITY_PUBLIC)// show details even when lock screen
            .setOnlyAlertOnce(false)// avoid multiple remote user request triggers multiple sounds
            .setAutoCancel(true)
            .setLocalOnly(true)
    }

    private fun getContentIntent(context: Context, data: IncomingData): PendingIntent? {
        return PendingIntent.getActivity(
            context,
            requestCode,
            Intent(context, AgoraActivity::class.java).apply {
                action = ACTION_PUSH
                putExtras(
                    bundleOf(
                        KEY_RID to data.rid,
                        KEY_PWD to data.pwd,
                        KEY_REQUEST_ID to data.requestId,
                        KEY_TICKET to data.ticket,
                        KEY_INVITER to data.inviterName,
                        KEY_AVATAR to data.inviterAvatar,
                    )
                )
            },
            PendingIntent.FLAG_UPDATE_CURRENT
        )
    }

    private fun getCancelIntent(context: Context): PendingIntent? {
        return PendingIntent.getActivity(
            context,
            requestCode,
            Intent(context, AgoraActivity::class.java),
            PendingIntent.FLAG_UPDATE_CURRENT
        )
    }

    private fun getConfirmIntent(context: Context): PendingIntent? {
        return PendingIntent.getActivity(
            context,
            requestCode,
            Intent(context, AgoraActivity::class.java),
            PendingIntent.FLAG_UPDATE_CURRENT
        )
    }

    fun cancel(notificationId: Int) {
        notificationManager.cancel(notificationId)
    }

    companion object {
        private const val TAG = "CallingNotification"
        const val CHANNEL_ID = "io.agora.avc.push.notification.CallingNotification"
        private const val PACKAGE_NAME = "io.agora.vcall"
        private var requestCode = 0x10000
        private const val TIMEOUT = 1000L
    }

}