package io.agora.avc.receiver

import android.app.NotificationManager
import android.app.PendingIntent
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import androidx.core.app.NotificationCompat
import androidx.core.app.NotificationManagerCompat
import androidx.core.content.ContextCompat
import io.agora.avc.R
import io.agora.avc.app.master.AgoraActivity
import io.agora.avc.bo.MeetingNotification
import io.agora.avc.bo.NoticeCode
import io.agora.avc.manager.notice.NotificationController
import io.agora.avc.service.RequestService
import io.agora.logger.Logger

class MediaRequestReceiver : BroadcastReceiver() {

    override fun onReceive(context: Context?, intent: Intent?) {
        val targetIntent = Intent(context, RequestService::class.java)
        targetIntent?.action = intent?.action
        intent?.let {
            targetIntent?.putExtras(it)
        }
        context?.startService(targetIntent)
    }

    companion object {
        private const val TAG = "[BroadCast][MediaRequest]"
        private const val KEY_USER_NAME = "USER_NAME"
        private const val KEY_USER_REQUEST = "USER_REQUEST"
        const val MEETING_EVENT_ACTION_AUDIO = "io.agora.avc.MEETING_EVENT_AUDIO"
        const val MEETING_EVENT_ACTION_VIDEO = "io.agora.avc.MEETING_EVENT_VIDEO"
        const val ACTION_CONFIRM = "ACTION_CONFIRM"
        private const val DATA_NAME = "DATA_NAME"
        private const val DATA_MESSAGE = "DATA_MESSAGE"
        private const val MEETING_EVENT_OPEN = "MEETING_EVENT_OPEN"
        private const val NOTIFICATION_ID_EVENT_AUDIO_REQUEST = 150
        private const val NOTIFICATION_ID_EVENT_VIDEO_REQUEST = 151
        private const val TAP_REQUEST = 0
        private const val REFUSE_REQUEST = 1
        private const val AGREE_REQUEST = 2
        private var DEFAULT_REQUEST_TIME: Long = 10 * 1000// unit seconds


        fun notifyRemoteRequest(context: Context, event: MeetingNotification) {
            val isAudio = event.code == NoticeCode.CODE_PEER_INVITE_MICROPHONE_RECEIVE
            val contentText = event.title
            Logger.i(TAG, "notify remote request,isAudio:$isAudio data:${contentText}")
            NotificationController.createNotificationChannel(
                context,
                NotificationController.MEETING_EVENT_CHANNEL_ID,
                R.string.meeting_event_notification_channel_name,
                R.string.meeting_event_notification_channel_des
            )
            val contentPendingIntent = getContentIntent(context)
            val refuseAction = getRefuseAction(context, isAudio)
            val agreeAction = getAgreeAction(context, isAudio)
            val builder =
                NotificationCompat
                    .Builder(context, NotificationController.MEETING_EVENT_CHANNEL_ID)
                    .setSmallIcon(NotificationController.createSmallIcon())
                    .setColor(ContextCompat.getColor(context, R.color.color_primary))
                    .setContentTitle(context.getText(R.string.remote_unmute_request))
                    .setContentText(contentText)
                    .setPriority(NotificationCompat.PRIORITY_HIGH)
                    .setVisibility(NotificationCompat.VISIBILITY_PUBLIC)// show details even when lock screen
                    .setCategory(NotificationCompat.CATEGORY_MESSAGE)
                    .setAutoCancel(true)// dismiss when user taps or click action
                    .setContentIntent(contentPendingIntent)
                    .setTimeoutAfter(getUnMuteRequestTimeout())
                    .setOnlyAlertOnce(true)// avoid multiple remote user request triggers multiple sounds
                    .addAction(refuseAction)
                    .addAction(agreeAction)

            val notificationId = getNotificationId(isAudio)
            with(NotificationManagerCompat.from(context)) {
                // notificationId is a unique int for each notification that you must define
                notify(notificationId, builder.build())
            }
        }

        private fun getContentIntent(context: Context): PendingIntent? {
            return PendingIntent.getActivity(
                context,
                TAP_REQUEST,
                Intent(context, AgoraActivity::class.java).apply {
                    action = MEETING_EVENT_OPEN
                },
                PendingIntent.FLAG_UPDATE_CURRENT
            )
        }

        private fun getRefuseAction(context: Context, isAudio: Boolean): NotificationCompat.Action {
            val refusePendingIntent = PendingIntent.getBroadcast(
                context,
                REFUSE_REQUEST, // difference from openPendingIntent/tapPendingIntent requestCode
                Intent(context, MediaRequestReceiver::class.java).apply {
                    action = if (isAudio) {
                        MEETING_EVENT_ACTION_AUDIO
                    } else {
                        MEETING_EVENT_ACTION_VIDEO
                    }
                    putExtra(ACTION_CONFIRM, false)
                },
                PendingIntent.FLAG_UPDATE_CURRENT
            )
            return NotificationCompat.Action(
                0,
                context.getString(R.string.pop_cancel_refuse),
                refusePendingIntent
            )
        }

        private fun getAgreeAction(context: Context, isAudio: Boolean): NotificationCompat.Action {
            val agreePendingIntent = if (isAudio) {
                val intent = Intent(context, MediaRequestReceiver::class.java).apply {
                    action = MEETING_EVENT_ACTION_AUDIO
                    putExtra(ACTION_CONFIRM, true)
                }
                PendingIntent.getBroadcast(
                    context,
                    AGREE_REQUEST, // difference from refusePendingIntent/tapPendingIntent requestCode
                    intent,
                    PendingIntent.FLAG_UPDATE_CURRENT
                )
            } else {
                val intent = Intent(context, AgoraActivity::class.java).apply {
                    action = MEETING_EVENT_ACTION_VIDEO
                }
                PendingIntent.getActivity(
                    context,
                    TAP_REQUEST,
                    intent,
                    PendingIntent.FLAG_UPDATE_CURRENT
                )
            }
            return NotificationCompat.Action(
                0,
                context.getString(R.string.pop_confirm_open),
                agreePendingIntent
            )
        }

        fun dismissRequestNotification(context: Context, isAudio: Boolean) {
            val notificationManager = getNotificationManager(context)
            val notificationId = getNotificationId(isAudio)
            notificationManager.cancel(notificationId)
        }

        private fun getNotificationManager(context: Context): NotificationManager {
            return ContextCompat.getSystemService(
                context,
                NotificationManager::class.java
            ) as NotificationManager
        }

        private fun getNotificationId(isAudio: Boolean): Int {
            return if (isAudio) {
                NOTIFICATION_ID_EVENT_AUDIO_REQUEST
            } else {
                NOTIFICATION_ID_EVENT_VIDEO_REQUEST
            }
        }

        private fun getUnMuteRequestTimeout(): Long {
            return DEFAULT_REQUEST_TIME
        }

        fun dismissRequestNotification(context: Context) {
            dismissRequestNotification(context, true)
            dismissRequestNotification(context, false)
        }
    }
}