package io.agora.avc.repository.impl

import android.util.ArrayMap
import io.agora.avc.app.address.AddressBookNode
import io.agora.avc.app.address.GroupNode
import io.agora.avc.app.address.MemberNode
import io.agora.avc.repository.AddressBookRepository
import io.agora.avc.utils.JsonUtils
import org.json.JSONArray
import org.json.JSONObject
import org.json.JSONTokener
import java.util.*
import javax.inject.Inject

class AddressBookRepositoryImpl @Inject constructor() : AddressBookRepository {
    private val addressBook = arrayListOf<AddressBookNode>()

    private val callUsers = arrayListOf<MemberNode>()
    private val addedUsers = arrayListOf<MemberNode>()

    private val addressBookMap = ArrayMap<String, GroupNode>()

    override fun queryInvitee(): ArrayList<MemberNode> {
        return callUsers
    }

    override fun queryAdded(): ArrayList<MemberNode> {
        return addedUsers
    }

    override fun queryAddressBook(deptId: String): AddressBookNode? {
        val rootNode = addressBook[0]
        if (rootNode is GroupNode && rootNode.deptId == deptId) {
            return rootNode
        }
        return searchDeptMembers(rootNode, deptId)
    }

    override fun analysisGetResponse(content: String?): AddressBookNode? {
        if (content == null || content.isEmpty()) {
            return null
        }
        val json = JSONTokener(content).nextValue()
        if (json !is JSONObject) {
            return null
        }
        return constructAddressBook(json)[0]
    }

    override fun analysisSearchResponse(content: String?): List<AddressBookNode>? {
        if (content == null || content.isEmpty()) {
            return null
        }
        val json = JSONTokener(content).nextValue()
        if (json !is JSONArray) {
            return null
        }
        return constructAddressBook(json)
    }

    override fun clear() {
        addressBook.clear()
        callUsers.clear()
        addressBookMap.clear()
        addedUsers.clear()
    }

    override fun clearInvitee() {
        callUsers.clear()
    }

    override fun clearAdded() {
        addedUsers.clear()
    }

    override fun getDeptInfo(seq: Int, dept: String): GroupNode? {
        val result = addressBookMap[dept]
        if (result != null) {
            return result
        }
        return null
    }

    override fun saveDeptInfo(dept: String, data: GroupNode) {
        addressBookMap[dept] = data
    }

    override fun saveAdded(data: ArrayList<MemberNode>) {
        addedUsers.addAll(data)
    }

    private fun constructAddressBook(data: JSONArray): List<AddressBookNode>? {
        if (data != null && data.length() > 0) {
            val memberList = arrayListOf<AddressBookNode>()
            for (index in 0 until data.length()) {
                memberList.add(createMemberNode(data.getJSONObject(index)))
            }
            return memberList
        }
        return null
    }

    private fun constructAddressBook(jsonRoot: JSONObject): List<AddressBookNode> {
        val result = arrayListOf<AddressBookNode>()
        val stack = Stack<JSONObject>()
        val stackNode = Stack<AddressBookNode>()
        val rootNode = createAddressBookNode(jsonRoot)
        val department = (rootNode as? GroupNode)?.name
        result.add(rootNode)
        stack.push(jsonRoot)
        stackNode.push(rootNode)
        while (stack.isNotEmpty()) {
            val item = stack.pop()
            val node = stackNode.pop()
            val members = JsonUtils.getJSONArray(item, "members", null)
            if (members != null && members.length() > 0) {
                val memberList = arrayListOf<AddressBookNode>()
                for (index in 0 until members.length()) {
                    memberList.add(createMemberNode(members.getJSONObject(index), department))
                }
                node.children.addAll(memberList)
            }
            val children = JsonUtils.getJSONArray(item, "children", null)
            if (children != null && children.length() > 0) {
                val childrenList = arrayListOf<AddressBookNode>()
                for (index in 0 until children.length()) {
                    val groupNode = createGroupNode(children.getJSONObject(index))
                    childrenList.add(groupNode)
                    stack.push(children.getJSONObject(index))
                    stackNode.push(groupNode)
                }
                node.children.addAll(childrenList)
            }
        }
        return result
    }

    private fun createAddressBookNode(item: JSONObject): AddressBookNode {
        val deptId = JsonUtils.getInt(item, "deptId", -1)
        return if (deptId != -1) {
            createGroupNode(item)
        } else {
            createMemberNode(item)
        }
    }

    private fun searchDeptMembers(tree: AddressBookNode, deptId: String): AddressBookNode? {
        val queue = ArrayDeque<AddressBookNode>()
        queue.offer(tree)
        while (queue.isNotEmpty()) {
            val item = queue.poll()
            if (item.children.isNotEmpty()) {
                for (child in item.children) {
                    if (child is AddressBookNode) {
                        if (child is GroupNode && child.deptId == deptId) {
                            return child
                        }
                        queue.offer(child)
                    }
                }
            }
        }
        return null
    }

    private fun createGroupNode(item: JSONObject?): AddressBookNode {
        val name = JsonUtils.getString(item, "name")
        val deptId = JsonUtils.getInt(item, "deptId")
        return GroupNode(name, deptId.toString())
    }

    private fun createMemberNode(dataRoot: JSONObject): MemberNode {
        val department = JsonUtils.getString(dataRoot, "department")
        return createMemberNode(dataRoot, department)
    }

    private fun createMemberNode(dataRoot: JSONObject, department: String?): MemberNode {
        val userid = JsonUtils.getString(dataRoot, "userid")
        val name = JsonUtils.getString(dataRoot, "name")
        val alias = JsonUtils.getString(dataRoot, "alias")
        val mainDepartment = JsonUtils.getInt(dataRoot, "main_department")
        val avatar = JsonUtils.getString(dataRoot, "avatar")
        return MemberNode(
            userid,
            name,
            alias,
            mainDepartment,
            avatar,
            department = department,
        )
    }
}
