package io.agora.avc.repository.impl

import io.agora.avc.bo.AppConfig
import io.agora.avc.bo.AppTips
import io.agora.avc.bo.AppVersion
import io.agora.avc.bo.DevConfig
import io.agora.avc.manager.splite.SPLite
import io.agora.avc.net.api.ApiService
import io.agora.avc.repository.AppConfigRepository
import io.agora.avc.utils.DeviceUtils
import io.agora.avc.utils.GsonUtils
import io.agora.frame.base.BaseModel
import io.agora.frame.data.IDataRepository
import io.agora.logger.LogConverter
import io.agora.logger.Logger
import io.reactivex.Observable
import javax.inject.Inject

class AppConfigRepositoryImpl @Inject constructor(
    dataRepository: IDataRepository?,
    private val spLite: SPLite
) : BaseModel(dataRepository), AppConfigRepository {

    private var appVer: AppVersion? = null

    private var appConfig: AppConfig? = null

    private var devConfig: DevConfig? = null

    override fun queryAppVer(
        appVersionName: String,
        type: Int,
        language: String
    ): Observable<AppVersion> {
        return getRetrofitService(ApiService::class.java)
            .getAppVer(DeviceUtils.getPlatform(), appVersionName, type, language)
            .doOnNext { _appConfig ->
                appVer = _appConfig
            }
    }

    override fun getAppVer(): AppVersion? {
        Logger.i(TAG, "get app ver, (result has value):${appVer != null}")
        return appVer
    }

    override fun queryAppConfig(
        appVersionName: String,
        type: Int,
        view: String
    ): Observable<AppConfig> {
        return getRetrofitService(ApiService::class.java)
            .getAppConfig("android", appVersionName, type, view)
            .doOnNext { _appConfig ->
                saveAppConfig(_appConfig)
            }
    }

    override fun saveAppConfig(_appConfig: AppConfig?) {
        spLite.saveAppConfig(GsonUtils.toJson(_appConfig))
    }

    override fun getAppConfig(): AppConfig? {
        if (appConfig != null) {
            return appConfig
        }
        return spLite.getAppConfig()
    }

    override fun getLastVersion(): String {
        return spLite.getLastVersion().apply {
            Logger.i(TAG, "get last version:${this}")
        }
    }

    override fun saveLastVersion(value: String) {
        Logger.i(TAG, "save last version:${value}")
        spLite.setLastVersion(value)
    }

    override fun getUpgradeNotice(): String {
        return spLite.getNotice().apply {
            Logger.i(TAG, "get upgrade notice:${this}")
        }
    }

    override fun saveUpgradeNotice(value: String) {
        Logger.i(TAG, "save upgrade notice:${value}")
        spLite.setNotice(value)
    }

    override fun getDarkTheme(): Int {
        return spLite.getDarkTheme().apply {
            Logger.i(TAG, "get dark theme:${if (this == 2) "yes" else "no"}")
        }
    }

    override fun saveDarkTheme(value: Int) {
        Logger.i(TAG, "save dark theme:${if (value == 2) "yes" else "no"}")
        spLite.setDarkTheme(value)
    }

    override fun getDevelopMode(): Boolean {
        return spLite.getDevelopMode().apply {
            Logger.i(TAG, "get developer mode:${LogConverter.enable(this)}")
        }
    }

    override fun saveDevelopMode(value: Boolean) {
        Logger.i(TAG, "save developer mode:${LogConverter.enable(value)}")
        spLite.setDevelopMode(value)
    }

    override fun getShowWelcome(): Boolean {
        return spLite.getShowWelcome().apply {
            Logger.i(TAG, "get show welcome:${LogConverter.visible(this)}")
        }
    }

    override fun saveShowWelcome(value: Boolean) {
        Logger.i(TAG, "save show welcome:${LogConverter.visible(value)}")
        spLite.setShowWelcome(value)
    }

    override fun privacyShowed(): Boolean {
        return spLite.privacyShowed().apply {
            Logger.i(TAG, "privacy showed:${this}")
        }
    }

    override fun saveShowPrivacy(value: Boolean) {
        Logger.i(TAG, "save show privacy:${value}")
        spLite.setShowPrivacy(value)
    }

    override fun getPictureMode(): Boolean {
        return spLite.getPictureMode().apply {
            Logger.i(TAG, "get picture mode:${LogConverter.enable(this)}")
        }
    }

    override fun savePictureMode(value: Boolean) {
        Logger.i(TAG, "save picture mode:${LogConverter.enable(value)}")
        spLite.setPictureMode(value)
    }

    override fun isTestServer(): Boolean {
        return spLite.isTestServer().apply {
            Logger.i(TAG, "get is test server:${this}")
        }
    }

    override fun saveTestServer(value: Boolean) {
        Logger.i(TAG, "save test server:${value}")
        spLite.setTestServer(value)
    }

    override fun getAvcGuid(): String {
        return spLite.getAvcGuid().apply {
            Logger.i(TAG, "get avc guid:${this}")
        }
    }

    override fun saveAvcGuid(value: String) {
        Logger.i(TAG, "save avc guid:${value}")
        spLite.setAvcGuid(value)
    }

    override fun clearToken() {
        Logger.i(TAG, "clear token")
        spLite.clearToken()
    }

    override fun getAppTips(): AppTips? {
        return spLite.getAppTips().apply {
            Logger.i(TAG, "get app tips:${this}")
        }
    }

    override fun saveAppTips(appTips: AppTips) {
        Logger.i(TAG, "save app tips:${appTips}")
        spLite.setAppTips(appTips)
    }

    override fun clearAppTips() {
        Logger.i(TAG, "clear app tips")
        spLite.clearAppTips()
    }

    override fun getRtcParameters(): Set<String> {
        return spLite.getRtcParameters().apply {
            Logger.i(TAG, "get rtc parameters:${this}")
        }
    }

    override fun saveRtcParameters(value: Set<String>) {
        Logger.i(TAG, "save rtc parameters:${value}")
        spLite.setRtcParameters(value)
    }

    override fun saveRoomMode(mode: Int) {
        Logger.i(TAG, "save room mode:$mode")
        spLite.setRoomMode(mode)
    }

    override fun getRoomMode(): Int {
        return spLite.getRoomMode()
    }

    override fun saveDevConfig(config: DevConfig) {
        return spLite.saveDevConfig(GsonUtils.toJson(config)).apply {
            Logger.i(TAG, "save dev config:$config")
        }
    }

    override fun getDevConfig(): DevConfig {
        return spLite.getDevConfig() ?: DevConfig()
    }

    override fun saveOpenNotificationGuideTime(time: Long) {
        spLite.saveOpenNotificationGuideTime(time)
    }

    override fun getOpenNotificationGuideTime() =
        spLite.getOpenNotificationGuideTime()

    override fun saveBannerNotificationGuideTime(time: Long) {
        spLite.saveBannerNotificationGuideTime(time)
    }

    override fun getBannerNotificationGuideTime(): Long? {
        return spLite.getBannerNotificationGuideTime()
    }

    override var needShowFraudTips: Boolean = true

    companion object {
        private const val TAG = "[Repository][appConfig]"
    }
}