package io.agora.avc.repository.impl

import io.agora.avc.bo.Attachment
import io.agora.avc.bo.Issue
import io.agora.avc.dao.AppDatabase
import io.agora.avc.po.Attach
import io.agora.avc.po.Problem
import io.agora.avc.po.ProblemWithAttach
import io.agora.avc.repository.ProblemRepository
import io.agora.avc.utils.GsonUtils
import io.agora.avc.utils.TimeUtils
import io.agora.frame.base.BaseModel
import io.agora.frame.data.IDataRepository
import io.agora.logger.Logger
import javax.inject.Inject

class ProblemRepositoryImpl @Inject constructor(
    dataRepository: IDataRepository
) : BaseModel(dataRepository), ProblemRepository {

    private var issue: Issue? = null

    private val roomDatabase by lazy {
        getRoomDatabase(AppDatabase::class.java, AppDatabase.DB_NAME)
    }

    override fun getProblemWithAttach(problemId: Long): ProblemWithAttach {
        return roomDatabase.problemDao().getProblemWithAttach(problemId).apply {
            Logger.i(
                TAG, "get problem with attach, problemId:$problemId" +
                        ", result:{problem=${this.problem.id},attachSize=${this.attach.size}}"
            )
        }
    }

    /**
     * add problem
     * @return problemId. add failed will be return -1
     */
    override fun addProblem(
        issue: Issue,
        reporterStreamId: Int?,
        reporterUid: String?
    ): Long {
        if (reporterStreamId == null && reporterUid == null) {
            Logger.e(TAG, "The unique ID of the reported user cannot be empty")
            return -1
        }
        val issueName: String = issue.issueName
        val issueStreamId: Int = issue.issueStreamId
        val oppositeStreamId: Int? = issue.oppositeStreamId
        val description: String? = issue.description
        val rid: String = issue.rid
        val sdkVersion: String = issue.sdkVersion
        val appVersion: String = issue.appVersion
        val tags = GsonUtils.toJson(issue.tags)
        val status = issue.status
        val zipFilePath = issue.path ?: ""
        val createTime = TimeUtils.getNowMills()
        val platform = issue.platform
        val problem = Problem(
            null,
            issueName = issueName,
            issueStreamId = issueStreamId,
            oppositeStreamId = oppositeStreamId,
            description = description,
            rid = rid,
            sdkVersion = sdkVersion,
            appVersion = appVersion,
            tags = tags,
            status = status,
            zipFilePath = zipFilePath,
            reporterStreamId = reporterStreamId,
            reporterUid = reporterUid,
            createTime = createTime,
            platform = platform,
            issueId = null
        )

        val problemId = roomDatabase.problemDao().insert(problem)

        problemId?.let {
            issue.attachmentList?.let { _attachmentList ->
                val attachIds = arrayListOf<Long>()
                _attachmentList.forEach { attachment ->
                    attachment.id?.let { id ->
                        attachIds.add(id)
                    }
                }
                val attachList = roomDatabase.attachDao().loadAllByIds(attachIds.toLongArray())
                if (attachList.isNotEmpty()) {
                    attachList.forEach {
                        it.problemId = problemId
                    }
                    roomDatabase.attachDao().insertAll(attachList)
                }
            }
        }

        return problemId
    }

    override fun updateProblem(problem: Problem) {
        Logger.i(TAG, "update problem, problemId:${problem.id}")
        roomDatabase.problemDao().update(problem)
    }

    override fun addAttach(attachment: Attachment): Long {
        Logger.i(TAG, "add attach, id:${attachment.id}")
        val opts = if (attachment.opts != null) {
            GsonUtils.toJson(attachment.opts)
        } else {
            null
        }

        val attach = Attach(
            id = null,
            name = attachment.name,
            path = attachment.path,
            mimeType = attachment.mimeType,
            size = attachment.size ?: 0,
            problemId = 0,
            opts = opts
        )

        return roomDatabase.attachDao().insert(attach)
    }

    override fun getAll(): List<Issue> {
        val problemWithAttach = roomDatabase.problemDao().getAll()
        val ret = arrayListOf<Issue>()
        problemWithAttach.forEach { data ->
            val attachment = arrayListOf<Attachment>()
            data.attach.forEach { attach ->
                attachment.add(
                    Attachment(
                        name = attach.name,
                        path = attach.path,
                        mimeType = attach.mimeType,
                        size = attach.size,
                        opts = GsonUtils.fromJson(attach.opts, Attachment.Opts::class.java)
                    )
                )
            }
            val tagArray = GsonUtils.fromJson(data.problem.tags, Array<Int>::class.java)
            val issue = Issue(
                issueName = data.problem.issueName,
                issueStreamId = data.problem.issueStreamId,
                oppositeStreamId = data.problem.oppositeStreamId,
                description = data.problem.description,
                rid = data.problem.rid,
                sdkVersion = data.problem.sdkVersion,
                appVersion = data.problem.appVersion,
                tags = tagArray.toMutableList(),
                status = data.problem.status,
                path = data.problem.zipFilePath,
                attachmentList = attachment,
                createTime = data.problem.createTime,
                platform = data.problem.platform
            )
            ret.add(issue)
        }
        return ret.apply {
            Logger.i(TAG, "get all problem, size:${this.size}")
        }
    }

    override fun getIssue(): Issue? {
        return this.issue
    }

    override fun updateIssue(issue: Issue) {
        Logger.i(TAG, "update issue, issue:$issue")
        this.issue = issue
    }

    override fun clearIssue() {
        Logger.i(TAG, "clear issue")
        this.issue = null
    }

    companion object {
        private const val TAG = "[Repository][Problem]"
    }
}