package io.agora.avc.screenshare

import android.content.Context
import io.agora.avc.config.APP_ID
import io.agora.avc.utils.GsonUtils
import io.agora.logger.Logger
import io.agora.rtc.Constants
import io.agora.rtc.IMetadataObserver
import io.agora.rtc.IRtcEngineEventHandler
import io.agora.rtc.RtcEngine
import io.agora.rtc.internal.EncryptionConfig
import io.agora.rtc.mediaio.IVideoSource
import io.agora.rtc.models.ChannelMediaOptions
import io.agora.rtc.models.DataStreamConfig
import io.agora.rtc.video.VideoEncoderConfiguration
import io.reactivex.Observable
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import protobuf.RTCMetadata
import java.util.concurrent.TimeUnit


class ScreenShareClient {
    private var rtcEngine: RtcEngine? = null

    private var observer: IMetadataObserver? = null

    private var metadata: ByteArray? = null

    private var dataStreamId: Int? = null

    private var dataStreaming: Disposable? = null

    fun prepare(application: Context, videoSource: IVideoSource, logPath: String) {
        rtcEngine = RtcEngine.create(application, APP_ID, object : IRtcEngineEventHandler() {
            override fun onConnectionStateChanged(state: Int, reason: Int) {
                super.onConnectionStateChanged(state, reason)
                Logger.i(TAG, "connection state changed, state:$state, reason:$reason")
            }
        })
        observer = object : IMetadataObserver {
            override fun getMaxMetadataSize(): Int {
                return metadata?.size ?: 0
            }

            override fun onReadyToSendMetadata(timeStampMs: Long): ByteArray {
                return metadata ?: byteArrayOf()
            }

            override fun onMetadataReceived(buffer: ByteArray?, uid: Int, timeStampMs: Long) {

            }

        }
        //log config
        rtcEngine?.setLogFileSize(LOG_FILE_SIZE)
        rtcEngine?.setLogFile(logPath)
        rtcEngine?.setLogFilter(Constants.LOG_FILTER_INFO)
        //disable dual stream mode
        rtcEngine?.enableDualStreamMode(false)
        rtcEngine?.enableVideo()
        rtcEngine?.enableAudioVolumeIndication(1000, 3, false)
        rtcEngine?.setChannelProfile(Constants.CHANNEL_PROFILE_LIVE_BROADCASTING)
        rtcEngine?.setVideoSource(videoSource)
        rtcEngine?.disableAudio()
        rtcEngine?.muteAllRemoteAudioStreams(true)
        rtcEngine?.muteAllRemoteVideoStreams(true)
        rtcEngine?.registerMediaMetadataObserver(observer, IMetadataObserver.VIDEO_METADATA)
//        startDataStreaming()
    }

    private fun setAudioProfile() {
        val profile = Constants.AudioScenario.getValue(Constants.AudioScenario.DEFAULT)
        val scenario = Constants.AudioScenario.getValue(Constants.AudioScenario.GAME_STREAMING)
        Logger.i(TAG, "set audio profile:$profile scenario:$scenario")
        rtcEngine?.setAudioProfile(profile, scenario)
    }

    fun joinChannel(
        channelId: String,
        streamId: Int,
        parentName: String,
        parentStreamId: Int,
        token: String,
        encryptionMode: Int,
        encryptionKey: String,
        encryptionSalt: String,
        cname: String,
        ips: ArrayList<String>?,
        domain: String,
        hasWatermark: Boolean
    ) {
        setEncryption(encryptionMode, encryptionSalt, encryptionKey)
        if (channelId.isEmpty()) {
            throw NullPointerException("can not join channel, channelId is empty")
        }
        rtcEngine?.setClientRole(Constants.CLIENT_ROLE_BROADCASTER)
        //set dynamic Parameter
        rtcEngine?.setParameters("{\"che.video.render.texture.mode\":0}")
        setAudioProfile()
        setMetadata(parentName, parentStreamId, hasWatermark)
        setLocalAccessPoint(ips, domain)
        val returnCode = rtcEngine?.joinChannel(token,
            cname,
            null,
            streamId,
            ChannelMediaOptions().apply {
                autoSubscribeAudio = false
                autoSubscribeVideo = false
                publishLocalAudio = false
                publishLocalVideo = true
            })
        Logger.i(
            TAG, "join channelId:$channelId, " +
                    "streamId:$streamId, " +
                    "parentName:$parentName, " +
                    "parentStreamId:$parentStreamId, " +
                    "token:$token, " +
                    "encryptionMode:$encryptionMode, " +
                    "encryptionKey:$encryptionKey, " +
                    "encryptionSalt:$encryptionSalt, " +
                    "cname:$cname, " +
                    "hasWatermark:$hasWatermark, " +
                    "return code:$returnCode"
        )
    }

    private fun setLocalAccessPoint(ips: ArrayList<String>?, domain: String) {
        Logger.i(TAG, "set local access point, ips:${GsonUtils.toJson(ips)} domain:$domain")
        ips?.let {
            rtcEngine?.setLocalAccessPoint(ips, domain)
        }
    }

    private fun setEncryption(
        encryptionMode: Int,
        encryptionSalt: String,
        encryptionKey: String
    ) {
        val config = EncryptionConfig()
        when (encryptionMode) {
            1 -> config.encryptionMode = EncryptionConfig.EncryptionMode.AES_128_XTS
            2 -> config.encryptionMode = EncryptionConfig.EncryptionMode.AES_128_ECB
            3 -> config.encryptionMode = EncryptionConfig.EncryptionMode.AES_256_XTS
            4 -> config.encryptionMode = EncryptionConfig.EncryptionMode.SM4_128_ECB
            5 -> config.encryptionMode = EncryptionConfig.EncryptionMode.AES_128_GCM
            6 -> config.encryptionMode = EncryptionConfig.EncryptionMode.AES_256_GCM
            7 -> config.encryptionMode = EncryptionConfig.EncryptionMode.AES_128_GCM2
            8 -> config.encryptionMode = EncryptionConfig.EncryptionMode.AES_256_GCM2
            else -> {

            }
        }
        if (encryptionMode == 7 || encryptionMode == 8) {
            val len = kotlin.math.min(config.encryptionKdfSalt.size, encryptionSalt.length)
            if (len == 0) {
                Logger.e(TAG, "setEncryption length invalid")
            }
            for (i in 0 until len) {
                config.encryptionKdfSalt[i] = encryptionSalt.toByteArray()[i]
            }
        }

        config.encryptionKey = encryptionKey
        rtcEngine?.enableEncryption(encryptionMode > 0, config)
    }

    /**
     * Increase the status field in Metadata, set it to 0x01|0x80, 0x80 means to turn on the watermark, 0x01 means to disable audio
     */
    private fun setMetadata(parentName: String, parentStreamId: Int, hasWatermark: Boolean) {
        val userBuilder = RTCMetadata.User.newBuilder()
        userBuilder.name = parentName
        userBuilder.parentStreamId = parentStreamId
        val status = if (hasWatermark) {
            FLAG_AUDIO or FLAG_WATERMARK
        } else {
            FLAG_AUDIO and FLAG_WATERMARK.inv()
        }
        userBuilder.status = status
        val userJson = convertUserBuilder2Json(userBuilder)
        Logger.i(TAG, "create share metadata,user:$userJson")
        metadata = RTCMetadata.Metadata.newBuilder().setUser(userBuilder).build().toByteArray()
    }

    private fun convertUserBuilder2Json(builder: RTCMetadata.User.Builder?): String {
        return "name:${builder?.name},parentStreamId:${builder?.parentStreamId},status:${builder?.status}"
    }

    fun setVideoEncoderConfiguration(config: VideoEncoderConfiguration) {
        Logger.i(
            TAG,
            "set video encoder config, width:${config.dimensions.width}, height:${config.dimensions.height}, orientation:${config.orientationMode}"
        )
        rtcEngine?.setVideoEncoderConfiguration(config)
    }

    fun leaveRoom() {
//        stopDataStreaming()
        rtcEngine?.leaveChannel()
    }

    /**
     * Send metadata at 3S intervals to synchronize information with WEB
     */
    private fun startDataStreaming() {
        Logger.i(TAG, "start data streaming")
        Observable
            .interval(0, 3, TimeUnit.SECONDS)
            .subscribeOn(Schedulers.io())
            .subscribe(object : io.reactivex.Observer<Long> {
                override fun onComplete() {

                }

                override fun onSubscribe(d: Disposable) {
                    dataStreaming = d
                }

                override fun onNext(t: Long) {
                    sendStreamMessage()
                }

                override fun onError(e: Throwable) {

                }
            })

    }

    private fun sendStreamMessage() {
        if (dataStreamId == null) {
            val streamId = rtcEngine?.createDataStream(DataStreamConfig())
            if (streamId != null && streamId >= 0) {
                dataStreamId = streamId
            }
        }
        dataStreamId?.let {
            rtcEngine?.sendStreamMessage(it, metadata)
        }
    }

    private fun stopDataStreaming() {
        Logger.i(TAG, "stop data streaming")
        dataStreaming?.dispose()
    }


    companion object {
        private const val TAG = "[Comm][ScreenShareClient]"
        private const val LOG_FILE_SIZE = 5120
        private const val FLAG_WATERMARK = 0x80
        private const val FLAG_AUDIO = 0x01

    }

}