package io.agora.avc.screenshare

import android.app.Dialog
import android.content.ActivityNotFoundException
import android.content.Context
import android.content.Intent
import android.media.projection.MediaProjectionManager
import android.os.Build
import androidx.annotation.RequiresApi
import androidx.appcompat.app.AppCompatActivity
import androidx.fragment.app.Fragment
import io.agora.avc.R
import io.agora.avc.bo.ShareInfo
import io.agora.avc.utils.ToastUtils
import io.agora.avc.video.ExternalVideoInputService
import io.agora.avc.widget.WatermarkRequestDialog
import io.agora.logger.Logger

class ScreenShareManager(
    private val fragment: Fragment,
    private val shareAction: ScreenShareAction?
) {

    private var alertDialog: Dialog? = null

    private val screenShareIntent by lazy {
        Intent(fragment.context, ExternalVideoInputService::class.java)
    }

    private val serviceConnection = object : ScreenShareServiceConnection() {
        override fun disConnected() {
            shareAction?.notifyServiceDisConnection()
            stop()
        }

        override fun reconnected() {
            startScreenService()
            bindShareScreenService()
        }

        override fun onScreenStatusChanged(status: Int) {
            shareAction?.notifyScreenStatusChanged(status)
        }
    }

    private fun stopScreenService() {
        Logger.i(TAG, "Stop shared screen service")
        fragment.context?.stopService(screenShareIntent)
    }

    private fun bindShareScreenService() {
        fragment.context?.bindService(
            screenShareIntent,
            serviceConnection,
            Context.BIND_AUTO_CREATE
        )
    }

    private fun startScreenService() {
        Logger.i(TAG, "Start shared screen service")
        fragment.context?.startService(screenShareIntent)
    }


    fun unbindShareScreenService() {
        try {
            Logger.i(TAG, "Unbind shared screen service")
            serviceConnection.getService()?.let { _ ->
                shareAction?.stopScreenShare()
                serviceConnection.setService(null)
                fragment.context?.unbindService(serviceConnection)
            }
        } catch (e: Exception) {
            Logger.e(TAG, "Unbind shared screen service exception", e)
        }
    }

    @RequiresApi(Build.VERSION_CODES.LOLLIPOP)
    fun applyScreenShare(open: Boolean, isAgRoom: Boolean, showWaterMarkDialog: Boolean = false) {
        try {
            if (!open) {
                shareAction?.stopScreenShare()
                stop()
                return
            }
            if (!isAgRoom) {
                doScreenShare(false)
                return
            }
            checkWatermarkPermission(showWaterMarkDialog) { useWatermark ->
                doScreenShare(useWatermark)
            }
        } catch (e: ActivityNotFoundException) {
            ToastUtils.showShort(fragment.getString(R.string.screen_share_not_support))
            Logger.e(TAG, "Didn't find MediaProjectionPermissionActivity")
        }
    }

    private fun doScreenShare(useWatermark: Boolean) {
        val mediaProjectionManager =
            fragment.context?.getSystemService(Context.MEDIA_PROJECTION_SERVICE) as? MediaProjectionManager

        mediaProjectionManager?.apply {
            val intent = this.createScreenCaptureIntent()
            fragment.startActivityForResult(
                intent,
                if (useWatermark) CODE_SCREEN_CAPTURE_WATERMARK_REQUEST else CODE_SCREEN_CAPTURE_NO_WATERMARK_REQUEST,
            )
        }
    }

    private fun checkWatermarkPermission(
        showWaterMarkDialog: Boolean,
        action: (useWaterMark: Boolean) -> Unit
    ) {
        if (!showWaterMarkDialog) {
            action.invoke(false)
            return
        }

        fragment.context?.apply {
            alertDialog = WatermarkRequestDialog(this)
                .setOnItemClickListener(object : WatermarkRequestDialog.OnItemClickListener {
                    override fun onCancelClick() {
                        action.invoke(false)
                        alertDialog?.dismiss()
                    }

                    override fun onConfirmClick() {
                        action.invoke(true)
                        alertDialog?.dismiss()
                    }
                })
                .showDialog()
        }
    }

    fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
        Logger.i(TAG, "request media projection result:$requestCode, requestCode:$requestCode")
        if (resultCode == AppCompatActivity.RESULT_OK) {
            serviceConnection.setRequestResult(resultCode, data)
            if (requestCode == CODE_SCREEN_CAPTURE_WATERMARK_REQUEST) {
                shareAction?.requestScreenShare(true)
            } else if (requestCode == CODE_SCREEN_CAPTURE_NO_WATERMARK_REQUEST) {
                shareAction?.requestScreenShare(false)
            }
        }
    }

    fun start(shareInfo: ShareInfo?) {
        serviceConnection.setShareInfo(shareInfo)
        startScreenService()
        bindShareScreenService()
    }

    fun stop() {
        serviceConnection.stopShareScreen()
        unbindShareScreenService()
        stopScreenService()
    }

    fun isSharingScreen(): Boolean {
        return serviceConnection.isRunning()
    }

    fun isRequesting(): Boolean {
        return alertDialog?.isShowing ?: false
    }

    fun dismissDialog() {
        alertDialog?.dismiss()
    }

    companion object {
        const val TAG = "[Comm][ScreenShare]"
        const val CODE_SCREEN_CAPTURE_NO_WATERMARK_REQUEST = 3
        const val CODE_SCREEN_CAPTURE_WATERMARK_REQUEST = 4
    }
}