package io.agora.avc.screenshare

import android.content.ComponentName
import android.content.Intent
import android.content.ServiceConnection
import android.os.Build
import android.os.DeadObjectException
import android.os.IBinder
import android.os.IBinder.DeathRecipient
import androidx.annotation.RequiresApi
import io.agora.avc.bo.ShareInfo
import io.agora.avc.utils.ScreenUtils
import io.agora.avc.video.ExternalVideoInputManager
import io.agora.logger.Logger

/**
 * need not reBind service,when IBind disconnected.
 * because share screen service has high level
 */
abstract class ScreenShareServiceConnection : ServiceConnection {
    private var shareData: Intent? = null

    private var shareResultCode: Int? = null

    private var service: IExternalVideoInputService? = null

    private var shareInfo: ShareInfo? = null

    private var deathRecipient: DeathRecipient = object : DeathRecipient {
        override fun binderDied() {
            Logger.e(TAG, "share screen service binder Died callback")
            service?.asBinder()?.unlinkToDeath(this, 0)
            reconnected()
        }
    }

    override fun onServiceDisconnected(name: ComponentName?) {
        Logger.e(TAG, "share screen service disconnected")
        this.service = null
        disConnected()
    }

    override fun onServiceConnected(name: ComponentName?, service: IBinder?) {
        Logger.i(TAG, "share screen service connected")
        this.service = IExternalVideoInputService.Stub.asInterface(service)
        try {
            service?.linkToDeath(deathRecipient, 0)
        } catch (e: Exception) {
            Logger.e(TAG, "Failed to linkToDeath to share screen service", e)
        }
        try {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP &&
                this.service?.isRunning == false
            ) {
                startScreenShare()
            }
        } catch (e: Exception) {
            Logger.e(TAG, "Failed to share screen", e)
        }
    }

    private var listener: IShareScreenListener = object : IShareScreenListener.Stub() {
        override fun onStatusChanged(status: Int) {
            Logger.i(TAG, "screen share status changed:$status")
            onScreenStatusChanged(status)
        }
    }

    @RequiresApi(Build.VERSION_CODES.LOLLIPOP)
    private fun startScreenShare() {
        val channelId = shareInfo?.channelId
        if (channelId == null) {
            disConnected()
            Logger.e(TAG, "Failed to start screen share,channelId is null")
            return
        }
        if (shareResultCode != null && shareData != null) {
            shareData?.apply {
                putExtra(ScreenConstants.INTENT_RID, channelId)
                putExtra(ScreenConstants.INTENT_MEDIA_ID, shareInfo?.shareId)
                putExtra(ScreenConstants.INTENT_MEDIA_TOKEN, shareInfo?.token)
                putExtra(ScreenConstants.INTENT_PARENT_MEDIA_ID, shareInfo?.parentStreamId)
                putExtra(ScreenConstants.INTENT_NAME, shareInfo?.parentUserName)
                putExtra(ScreenConstants.INTENT_ENCRYPTION_MODE, shareInfo?.encryptionMode)
                putExtra(ScreenConstants.INTENT_ENCRYPTION_KEY, shareInfo?.encryptionKey)
                putExtra(ScreenConstants.INTENT_ENCRYPTION_SALT, shareInfo?.encryptionSalt)
                putExtra(ScreenConstants.INTENT_HAS_WATERMARK, shareInfo?.hasWatermark)
                putExtra(ScreenConstants.INTENT_CNAME, shareInfo?.cname)
                putExtra(ScreenConstants.INTENT_IPS, shareInfo?.ips)
                putExtra(ScreenConstants.INTENT_DOMAIN, shareInfo?.domain)
                putExtra(
                    ExternalVideoInputManager.FLAG_SCREEN_WIDTH,
                    ScreenUtils.getScreenWidth()
                )
                putExtra(
                    ExternalVideoInputManager.FLAG_SCREEN_HEIGHT,
                    ScreenUtils.getScreenHeight()
                )
                putExtra(
                    ExternalVideoInputManager.FLAG_SCREEN_DPI,
                    ScreenUtils.getScreenDensityDpi()
                )
                putExtra(ExternalVideoInputManager.FLAG_FRAME_RATE, 7)
            }

            try {
                service?.setExternalVideoInput(
                    ExternalVideoInputManager.TYPE_SCREEN_SHARE,
                    shareData
                )
                service?.registerListener(listener)
                service?.start()
            } catch (e: DeadObjectException) {
                Logger.e(TAG, "Received DeadObjectException,when execute start method")
            } catch (e: SecurityException) {
                Logger.e(TAG, "Received SecurityException,when execute start method")
            }
        }
    }

    override fun onBindingDied(name: ComponentName?) {
        Logger.e(TAG, "share screen service binding died")
        disConnected()
    }

    fun isRunning(): Boolean {
        try {
            return service?.isRunning ?: false
        } catch (e: DeadObjectException) {
            Logger.e(TAG, "Received DeadObjectException,when execute isRunning method")
        } catch (e: SecurityException) {
            Logger.e(TAG, "Received SecurityException,when execute isRunning method")
        }
        return false
    }

    fun getService(): IExternalVideoInputService? {
        return service
    }

    fun setService(externalVideoInputService: IExternalVideoInputService?) {
        this.service = externalVideoInputService
    }

    fun setRequestResult(resultCode: Int, data: Intent?) {
        this.shareResultCode = resultCode
        this.shareData = data
    }

    fun setShareInfo(shareInfo: ShareInfo?) {
        this.shareInfo = shareInfo
    }

    abstract fun disConnected()

    abstract fun reconnected()

    abstract fun onScreenStatusChanged(status: Int)

    fun stopShareScreen() {
        service?.stop()
        service?.unregisterListener(listener)
    }

    companion object {
        private const val TAG = "[Comm][ScreenShareServiceConnection]"
    }
}