package io.agora.avc.service

import android.app.ActivityManager
import android.app.PendingIntent
import android.app.Service
import android.content.Context
import android.content.Intent
import android.os.Bundle
import android.os.IBinder
import androidx.core.app.NotificationCompat
import androidx.core.content.ContextCompat
import dagger.android.AndroidInjection
import io.agora.avc.R
import io.agora.avc.app.master.AgoraActivity
import io.agora.avc.biz.AppController
import io.agora.avc.bo.Room
import io.agora.avc.manager.notice.NotificationController
import io.agora.logger.Logger
import javax.inject.Inject

class DaemonService : Service() {

    @Inject
    lateinit var appController: AppController

    private var roomName: String? = null

    override fun onCreate() {
        super.onCreate()
        AndroidInjection.inject(this)
    }

    override fun onStartCommand(intent: Intent?, flags: Int, startId: Int): Int {
        roomName = intent?.extras?.getString(KEY_ROOM_NAME)

        return if (intent?.action == ACTION_START_FOREGROUND_SERVICE) {
            startForegroundService()
            // Try to solve https://stackoverflow.com/questions/14054588/what-is-start-sticky-start-not-sticky-and-start-redeliver-intent-service
            START_REDELIVER_INTENT
        } else {
            stopForegroundService()
            START_NOT_STICKY
        }
    }

    override fun onTaskRemoved(rootIntent: Intent?) {
        super.onTaskRemoved(rootIntent)
        Logger.i(TAG, "Received onTaskRemoved callback")
        appController.leaveRoom()
        appController.destroy()
        killChildProcess()
        stopForegroundService()
    }

    private fun killChildProcess() {
        Logger.i(TAG, "call kill child process")
        val systemService =
            application.getSystemService(Context.ACTIVITY_SERVICE) as ActivityManager
        val runningAppProcesses = systemService.runningAppProcesses
        if (runningAppProcesses != null) {
            for (runningAppProcess in runningAppProcesses) {
                if (runningAppProcess.pid != android.os.Process.myPid()) {
                    Logger.i(TAG, "kill process:${runningAppProcess.processName}")
                    android.os.Process.killProcess(runningAppProcess.pid)
                }
            }
        }
    }

    override fun onBind(intent: Intent): IBinder? {
        return null
    }

    private fun startForegroundService() {
        Logger.i(TAG, "start daemon service")
        NotificationController.createNotificationChannel(
            applicationContext,
            NotificationController.DAEMON_CHANNEL_ID,
            R.string.daemon_notification_channel_name,
            R.string.daemon_notification_channel_des
        )

        val notificationBuilder =
            NotificationCompat
                .Builder(this, NotificationController.DAEMON_CHANNEL_ID)
                .setSmallIcon(NotificationController.createSmallIcon())
                .setColor(ContextCompat.getColor(this, R.color.color_primary))
                .setContentTitle(getString(R.string.daemon_notification_title))
                .setContentText(getString(R.string.daemon_notification_content, roomName))
                .setWhen(System.currentTimeMillis())
                .setPriority(NotificationCompat.PRIORITY_LOW)
                .setContentIntent(getContentIntent())
                .setShowWhen(false)

        startForeground(NOTIFICATION_ID_DAEMON_SERVICE, notificationBuilder.build())
    }

    private fun getContentIntent(): PendingIntent? {
        val intent = Intent(this, AgoraActivity::class.java).apply {
            action = "NOTIFICATION_OPEN"
        }
        return PendingIntent.getActivity(
            this,
            0,
            intent,
            PendingIntent.FLAG_UPDATE_CURRENT
        )
    }

    private fun stopForegroundService() {
        Logger.i(TAG, "stop daemon service")
        stopForeground(true)
        stopSelf()
    }

    companion object {

        private const val TAG = "[Service][Daemon]"

        const val ACTION_START_FOREGROUND_SERVICE = "ACTION_START_FOREGROUND_SERVICE"
        private const val ACTION_STOP_FOREGROUND_SERVICE = "ACTION_STOP_FOREGROUND_SERVICE"

        const val KEY_ROOM_NAME = "key_room_name"

        const val NOTIFICATION_ID_DAEMON_SERVICE = 123

        private var isServiceRunning = false

        fun startDaemonService(context: Context, roomInfo: Room?) {
            if (isServiceRunning) {
                return
            }
            try {
                val intent = Intent(context, DaemonService::class.java)
                intent.action = ACTION_START_FOREGROUND_SERVICE


                val bundle = Bundle()
                bundle.putString(KEY_ROOM_NAME, roomInfo?.name)
                intent.putExtras(bundle)

                // Crash: startForegroundService must call start startForeground
                // This crash usually happens when background global configuration changes(like: Debug DisplayCut configuration changes)
                context.startService(intent)
                isServiceRunning = true
            } catch (e: Exception) {
                e.printStackTrace()
            }
        }

        fun stopDaemonService(context: Context) {
            if (!isServiceRunning) {
                return
            }
            try {
                val intent = Intent(context, DaemonService::class.java)
                intent.action = ACTION_STOP_FOREGROUND_SERVICE

                // Crash: startForegroundService must call start startForeground
                // This crash usually happens when background global configuration changes(like: Debug DisplayCut configuration changes)
                context.startService(intent)
                isServiceRunning = false
            } catch (e: Exception) {
                e.printStackTrace()
            }
        }
    }
}
