package io.agora.avc.video;

import android.content.Intent;
import android.graphics.SurfaceTexture;
import android.opengl.EGLSurface;
import android.opengl.GLES11Ext;
import android.opengl.GLES20;
import android.os.Build;
import android.os.RemoteCallbackList;
import android.os.RemoteException;
import android.util.Size;
import android.view.Surface;

import androidx.annotation.RequiresApi;

import io.agora.avc.MyApplication;
import io.agora.avc.screenshare.IShareScreenListener;
import io.agora.avc.screenshare.ScreenShareClient;
import io.agora.avc.screenshare.ScreenShareEngine;
import io.agora.avc.screenshare.ScreenShareInput;
import io.agora.avc.video.gles.ProgramTextureOES;
import io.agora.avc.video.gles.core.EglCore;
import io.agora.avc.video.gles.core.GlUtil;
import io.agora.logger.Logger;
import io.agora.rtc.mediaio.IVideoFrameConsumer;
import io.agora.rtc.mediaio.IVideoSource;
import io.agora.rtc.mediaio.MediaIO;

@RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
public class ExternalVideoInputManager implements IVideoSource {
    private static final String TAG = "ExternalVideoInputManager";

    public static final int TYPE_LOCAL_VIDEO = 1;
    public static final int TYPE_SCREEN_SHARE = 2;
    public static final int TYPE_AR_CORE = 3;

    public static final String FLAG_VIDEO_PATH = "flag-local-video";
    public static final String FLAG_SCREEN_WIDTH = "screen-width";
    public static final String FLAG_SCREEN_HEIGHT = "screen-height";
    public static final String FLAG_SCREEN_DPI = "screen-dpi";
    public static final String FLAG_FRAME_RATE = "screen-frame-rate";

    private static final int DEFAULT_SCREEN_WIDTH = 640;
    private static final int DEFAULT_SCREEN_HEIGHT = 480;
    private static final int DEFAULT_SCREEN_DPI = 3;
    private static final int DEFAULT_FRAME_RATE = 15;

    private MyApplication mApplication;
    private ExternalVideoInputThread mThread;
    private int mCurInputType;
    private IExternalVideoInput mCurVideoInput;
    private IExternalVideoInput mNewVideoInput;

    // RTC video interface to send video
    private IVideoFrameConsumer mConsumer;
    private IEngine mEngine;
    private RemoteCallbackList<IShareScreenListener> mListener = new RemoteCallbackList<>();

    public ExternalVideoInputManager(MyApplication application) {
        mApplication = application;
    }

    void start() {
        Logger.INSTANCE.i(TAG, "Prepared to start external video input manager");
        mThread = new ExternalVideoInputThread();
        mThread.start();
        notifyStatusChanged(ShareStatus.RUNNING.getValue());
    }

    private void notifyStatusChanged(int status) {
        int num = mListener.beginBroadcast();
        for (int i = 0; i < num; i++) {
            try {
                mListener.getBroadcastItem(i).onStatusChanged(status);
            } catch (RemoteException e) {
                Logger.INSTANCE.e(TAG, "notify error, status:" + status, e);
                e.printStackTrace();
            }
        }
        mListener.finishBroadcast();
    }

    boolean setExternalVideoInput(int type, Intent intent) {
        // Do not reset current input if the target type is
        // the same as the current which is still running.
        if (mCurInputType == type && mCurVideoInput != null
                && mCurVideoInput.isRunning()) {
            return false;
        }

        IExternalVideoInput input = null;
        switch (type) {
            case TYPE_LOCAL_VIDEO:
            case TYPE_AR_CORE:
                break;
            case TYPE_SCREEN_SHARE:
                mEngine = new ScreenShareEngine(mApplication, intent, new ScreenShareClient());
                int width = intent.getIntExtra(FLAG_SCREEN_WIDTH, DEFAULT_SCREEN_WIDTH);
                int height = intent.getIntExtra(FLAG_SCREEN_HEIGHT, DEFAULT_SCREEN_HEIGHT);
                int dpi = intent.getIntExtra(FLAG_SCREEN_DPI, DEFAULT_SCREEN_DPI);
                int fps = intent.getIntExtra(FLAG_FRAME_RATE, DEFAULT_FRAME_RATE);
                Logger.INSTANCE.i(TAG, "ScreenShare:" + width + "|" + height + "|" + dpi + "|" + fps);
                input = new ScreenShareInput(mApplication.getApplicationContext(),
                        width, height, dpi, fps, intent);
                break;
        }

        setExternalVideoInput(input);
        mCurInputType = type;
        return true;
    }

    private void setExternalVideoInput(IExternalVideoInput source) {
        if (mThread != null && mThread.isAlive()) mThread.pauseThread();
        mNewVideoInput = source;
    }

    void stop() {
        if (mThread != null) {
            mThread.setThreadStopped();
            notifyStatusChanged(ShareStatus.IDLE.getValue());
        }
    }

    @Override
    public boolean onInitialize(IVideoFrameConsumer consumer) {
        mConsumer = consumer;
        return true;
    }

    @Override
    public boolean onStart() {
        return true;
    }

    @Override
    public void onStop() {
    }

    public boolean isRunning() {
        return mThread != null && mThread.isRunning();
    }

    @Override
    public void onDispose() {
        mConsumer = null;
    }

    @Override
    public int getBufferType() {
        return MediaIO.BufferType.TEXTURE.intValue();
    }

    @Override
    public int getCaptureType() {
        return MediaIO.CaptureType.SCREEN.intValue();
    }

    @Override
    public int getContentHint() {
        return MediaIO.ContentHint.DETAIL.intValue();
    }

    public void onConfigurationChanged(int orientationMode) {
        if (mEngine != null) {
            mEngine.onConfigurationChanged(orientationMode);
        }
    }

    public void registerListener(IShareScreenListener listener) {
        mListener.register(listener);
    }

    public void unregisterListener(IShareScreenListener listener) {
        mListener.unregister(listener);
    }

    private class ExternalVideoInputThread extends Thread {
        private final int DEFAULT_WAIT_TIME = 1;

        private EglCore mEglCore;
        private EGLSurface mEglSurface;
        private int mTextureId;
        private SurfaceTexture mSurfaceTexture;
        private Surface mSurface;
        private float[] mTransform = new float[16];
        private GLThreadContext mThreadContext;
        int mVideoWidth;
        int mVideoHeight;
        private volatile boolean mStopped;
        private volatile boolean mPaused;

        private void prepare() {
            mEglCore = new EglCore();
            mEglSurface = mEglCore.createOffscreenSurface(1, 1);
            mEglCore.makeCurrent(mEglSurface);
            mTextureId = GlUtil.createTextureObject(GLES11Ext.GL_TEXTURE_EXTERNAL_OES);
            mSurfaceTexture = new SurfaceTexture(mTextureId);
            mSurface = new Surface(mSurfaceTexture);
            mThreadContext = new GLThreadContext();
            mThreadContext.eglCore = mEglCore;
            mThreadContext.context = mEglCore.getEGLContext();
            mThreadContext.program = new ProgramTextureOES();
            mEngine.prepare(ExternalVideoInputManager.this);
        }

        private void release() {
            mEngine.release();
            mSurface.release();
            mEglCore.makeNothingCurrent();
            mEglCore.releaseSurface(mEglSurface);
            mSurfaceTexture.release();
            GlUtil.deleteTextureObject(mTextureId);
            mTextureId = 0;
            mEglCore.release();
        }

        @Override
        public void run() {
            prepare();

            while (!mStopped) {
                if (mCurVideoInput != mNewVideoInput) {
                    Logger.INSTANCE.i(TAG, "New video input selected");
                    // Current video input is running, but we now
                    // introducing a new video type.
                    // The new video input type may be null, referring
                    // that we are not using any video.
                    if (mCurVideoInput != null) {
                        mCurVideoInput.onVideoStopped(mThreadContext);
                        Logger.INSTANCE.i(TAG, "recycle stopped input");
                    }

                    mCurVideoInput = mNewVideoInput;
                    if (mCurVideoInput != null) {
                        mCurVideoInput.onVideoInitialized(mSurface);
                        Logger.INSTANCE.i(TAG, "initialize new input");
                    }

                    if (mCurVideoInput == null) {
                        continue;
                    }

                    Size size = mCurVideoInput.onGetFrameSize();
                    mVideoWidth = size.getWidth();
                    mVideoHeight = size.getHeight();
                    mSurfaceTexture.setDefaultBufferSize(mVideoWidth, mVideoHeight);

                    if (mPaused) {
                        // If current thread is in pause state, it must be paused
                        // because of switching external video sources.
                        mPaused = false;
                    }
                } else if (mCurVideoInput != null && !mCurVideoInput.isRunning()) {
                    // Current video source has been stopped by other
                    // mechanisms (video playing has completed, etc).
                    // A callback method is invoked to do some collect
                    // or release work.
                    // Note that we also set the new video source null,
                    // meaning at meantime, we are not introducing new
                    // video types.
                    Logger.INSTANCE.i(TAG, "current video input is not running");
                    mCurVideoInput.onVideoStopped(mThreadContext);
                    mCurVideoInput = null;
                    mNewVideoInput = null;
                }

                if (mPaused || mCurVideoInput == null) {
                    waitForTime(DEFAULT_WAIT_TIME);
                    continue;
                }

                try {
                    mSurfaceTexture.updateTexImage();
                    mSurfaceTexture.getTransformMatrix(mTransform);
                } catch (Exception e) {
                    e.printStackTrace();
                }

                if (mCurVideoInput != null) {
                    mCurVideoInput.onFrameAvailable(mThreadContext, mTextureId, mTransform);
                }

                mEglCore.makeCurrent(mEglSurface);
                GLES20.glViewport(0, 0, mVideoWidth, mVideoHeight);

                if (mConsumer != null) {
                    mConsumer.consumeTextureFrame(mTextureId,
                            MediaIO.PixelFormat.TEXTURE_OES.intValue(),
                            mVideoWidth, mVideoHeight, 0,
                            System.currentTimeMillis(), mTransform);
                }

                // The pace at which the output Surface is sampled
                // for video frames is controlled by the waiting
                // time returned from the external video source.
                waitForNextFrame();
            }

            if (mCurVideoInput != null) {
                // The manager will cause the current
                // video source to be stopped.
                mCurVideoInput.onVideoStopped(mThreadContext);
            }
            release();
        }

        void pauseThread() {
            mPaused = true;
        }

        void setThreadStopped() {
            if (mCurVideoInput != null) {
                mCurVideoInput.onVideoStopped(mThreadContext);
                Logger.INSTANCE.i(TAG, "stop playing video, due to thread stopped");
            }
            mStopped = true;
        }

        private void waitForNextFrame() {
            int wait = mCurVideoInput != null
                    ? mCurVideoInput.timeToWait()
                    : DEFAULT_WAIT_TIME;
            waitForTime(wait);
        }

        public boolean isRunning() {
            return !mStopped;
        }

        private void waitForTime(int time) {
            try {
                Thread.sleep(time);
            } catch (InterruptedException e) {
                e.printStackTrace();
                Thread.currentThread().interrupt();
            }
        }
    }
}
