package io.agora.avc.video;

import android.app.Service;
import android.content.Intent;
import android.content.pm.ServiceInfo;
import android.os.Build;
import android.os.IBinder;
import android.os.RemoteException;

import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;
import androidx.core.app.NotificationCompat;

import io.agora.avc.MyApplication;
import io.agora.avc.screenshare.IExternalVideoInputService;
import io.agora.avc.screenshare.IShareScreenListener;
import io.agora.avc.screenshare.ScreenShareNotification;
import io.agora.logger.Logger;

@RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
public class ExternalVideoInputService extends Service {
    private static final String TAG = "ExternalVideoInputService";
    private static final int NOTIFICATION_ID = 124;

    private ExternalVideoInputManager mSourceManager;
    private IExternalVideoInputService mService;
    private OrientationListener mOrientationListener;

    @Override
    public void onCreate() {
        super.onCreate();
        Logger.INSTANCE.i(TAG, "external video input service onCreate");
        startForeground();
        mOrientationListener = new OrientationListener(this);
        mOrientationListener.registerOrientationChangedListener(orientation -> {
            mSourceManager.onConfigurationChanged(orientation);
        });
        mSourceManager = new ExternalVideoInputManager((MyApplication) getApplication());
        mService = new IExternalVideoInputService.Stub() {
            @Override
            public boolean setExternalVideoInput(int type, Intent intent) throws RemoteException {
                synchronized (this) {
                    return mSourceManager.setExternalVideoInput(type, intent);
                }
            }

            @Override
            public void start() throws RemoteException {
                mOrientationListener.enable();
                startSourceManager();
            }

            @Override
            public void stop() throws RemoteException {
                mOrientationListener.disable();
                stopSourceManager();
            }

            @Override
            public boolean isRunning() throws RemoteException {
                synchronized (this) {
                    return mSourceManager.isRunning();
                }
            }

            @Override
            public void registerListener(IShareScreenListener listener) throws RemoteException {
                synchronized (this) {
                    mSourceManager.registerListener(listener);
                }
            }

            @Override
            public void unregisterListener(IShareScreenListener listener) throws RemoteException {
                synchronized (this) {
                    mSourceManager.unregisterListener(listener);
                }
            }
        };
    }

    @Nullable
    @Override
    public IBinder onBind(Intent intent) {
        Logger.INSTANCE.i(TAG, "external video input service onBind");
        return mService.asBinder();
    }

    @Override
    public void onRebind(Intent intent) {
        super.onRebind(intent);
        Logger.INSTANCE.i(TAG, "external video input service onRebind");
    }

    private void startForeground() {
        NotificationCompat.Builder builder = ScreenShareNotification.INSTANCE
                .getBuilder(getApplicationContext());

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            startForeground(NOTIFICATION_ID, builder.build(),
                    ServiceInfo.FOREGROUND_SERVICE_TYPE_MEDIA_PROJECTION);
        } else {
            startForeground(NOTIFICATION_ID, builder.build());
        }
    }

    private void startSourceManager() {
        synchronized (this) {
            mSourceManager.start();
        }
    }

    @Override
    public boolean onUnbind(Intent intent) {
        Logger.INSTANCE.e(TAG, "external video input service onUnBind");
        return true;
    }

    private void stopSourceManager() {
        synchronized (this) {
            if (mSourceManager != null) {
                mSourceManager.stop();
            }
        }
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        return START_NOT_STICKY;
    }

    @Override
    public void onDestroy() {
        Logger.INSTANCE.e(TAG, "external video input service onDestroy");
        stopSourceManager();
        stopForeground(true);
        mOrientationListener.unregisterOrientationChangedListener();
        super.onDestroy();
    }
}
