package io.agora.avc.widget

import android.app.Dialog
import android.content.Context
import android.content.Intent
import android.graphics.Color
import android.graphics.drawable.ColorDrawable
import android.net.Uri
import android.os.Bundle
import android.text.method.ScrollingMovementMethod
import android.view.LayoutInflater
import android.view.View
import android.view.Window
import androidx.core.view.isVisible
import androidx.databinding.DataBindingUtil
import io.agora.avc.BuildConfig
import io.agora.avc.R
import io.agora.avc.bo.AppVersion
import io.agora.avc.databinding.DialogAppUpgradeBinding
import io.agora.avc.utils.MetaDataUtils

private const val CHANNEL_KEY: String = "CHANNEL_ID"
private const val CHANNEL_GOOGLE: String = "google"
private const val CHANNEL_CHINA: String = "china"

class AppUpgradeDialog(
    private var ctx: Context,
    private val appVersion: AppVersion?
) : Dialog(ctx) {

    private var upgradeCallback: UpgradeCallback? = null
    private var dialogBinding: DialogAppUpgradeBinding? = null

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        requestWindowFeature(Window.FEATURE_NO_TITLE)

        window?.setBackgroundDrawable(ColorDrawable(Color.TRANSPARENT))

        setCancelable(false)
        dialogBinding = DataBindingUtil.inflate(
            LayoutInflater.from(context),
            R.layout.dialog_app_upgrade,
            null,
            false
        )

        val forceUpdate = appVersion?.forceUpdate ?: false
        val releaseNote = appVersion?.releaseNote
        val notice = appVersion?.notice
        val showNotice = appVersion?.showNotice ?: false

        dialogBinding?.tvTitle?.text = context.getString(R.string.app_upgrade)
        dialogBinding?.tvContent?.movementMethod = ScrollingMovementMethod.getInstance()
        if (releaseNote?.isNotEmpty() == true) {
            dialogBinding?.tvContent?.text = releaseNote
        } else {
            dialogBinding?.tvContent?.text = context.getString(R.string.upgrade_optional)
        }

        if (forceUpdate) {
            dialogBinding?.btnCancel?.isVisible = false
            dialogBinding?.btnConfirm?.text = context.getString(R.string.upgrade_confirm)
            dialogBinding?.btnConfirm?.setOnClickListener {
                upgrade(appVersion)
            }
        } else {
            dialogBinding?.btnCancel?.isVisible = true
            dialogBinding?.btnCancel?.text = context.getString(R.string.cancel)
            dialogBinding?.btnCancel?.setOnClickListener {
                if (showNotice && notice != null) {
                    upgradeCallback?.onUpgradeDismiss(notice)
                }
                dismiss()
            }
            dialogBinding?.btnConfirm?.text = context.getString(R.string.upgrade_confirm)
            dialogBinding?.btnConfirm?.setOnClickListener {
                upgrade(appVersion)
                dismiss()
                if (showNotice && notice != null) {
                    upgradeCallback?.onUpgradeDismiss(notice)
                }
            }
        }

        dialogBinding?.root?.apply {
            setContentView(this)
        }
    }

    fun showDialog(): Dialog {
        super.show()
        return this
    }

    fun setUpgradeCallback(callback: UpgradeCallback): AppUpgradeDialog {
        this.upgradeCallback = callback
        return this
    }

    interface UpgradeCallback {
        fun onUpgradeDismiss(notice: String)
        fun onUpgradeClick(appVersion: AppVersion?)
    }

    private fun upgrade(appVersion: AppVersion?) {
        val updateUrl = appVersion?.upgradeUrl
        if (MetaDataUtils.getString(CHANNEL_KEY) == CHANNEL_GOOGLE) {
            val uri = Uri.parse(updateUrl)
            val intent = Intent(Intent.ACTION_VIEW, uri)
            ctx?.startActivity(intent)
        } else if (MetaDataUtils.getString(CHANNEL_KEY) == CHANNEL_CHINA) {
            setProgress(0)
            upgradeCallback?.onUpgradeClick(appVersion)
        }
    }

    fun setProgress(process: Int) {
        if (process > 0 && dialogBinding?.progressBar?.isIndeterminate == true) {
            //Cannot switch to indeterminate mode while the progress indicator is visible.
            dialogBinding?.progressBar?.isVisible = false
            dialogBinding?.tvProgress?.isVisible = false
            dialogBinding?.progressBar?.isIndeterminate = false
        }
        if (process in 0 until 100) {
            dialogBinding?.progressBar?.isVisible = true
            dialogBinding?.tvProgress?.isVisible = true
            dialogBinding?.btnCancel?.visibility = View.INVISIBLE
            dialogBinding?.btnConfirm?.visibility = View.INVISIBLE
        } else {
            dialogBinding?.progressBar?.isVisible = false
            dialogBinding?.tvProgress?.isVisible = false
            dialogBinding?.btnCancel?.isVisible = (appVersion?.forceUpdate ?: false).not()
            dialogBinding?.btnConfirm?.isVisible = true
        }
        dialogBinding?.tvProgress?.text = "$process%"
        dialogBinding?.progressBar?.progress = process

    }

}