package io.agora.avc.widget

import android.content.Context
import android.os.CountDownTimer
import android.util.AttributeSet
import android.view.LayoutInflater
import android.widget.FrameLayout
import androidx.annotation.IntDef
import androidx.core.content.ContextCompat
import androidx.core.view.isVisible
import io.agora.avc.R
import io.agora.avc.databinding.LayoutUploadLogButtonBinding
import io.agora.avc.utils.ThemeUtils
import io.agora.logger.Logger

open class LoadingStatusTextButton @JvmOverloads constructor(
    context: Context, attrs: AttributeSet? = null, defStyleAttr: Int = 0
) : FrameLayout(context, attrs, defStyleAttr) {

    protected val binding =
        LayoutUploadLogButtonBinding.inflate(LayoutInflater.from(context), this, true)

    var normalText = context.getString(R.string.upload_log_button)
    var successText = context.getString(R.string.upload_log_fail_button)
    var failText = context.getString(R.string.upload_log_success_button)

    protected var normalTextColor: Int = 0
    protected var successTextColor: Int = 0
    protected var failTextColor: Int = 0

    private val countDownTimer by lazy {
        object : CountDownTimer(RESET_TIME, RESET_TIME) {
            override fun onTick(millisUntilFinished: Long) {}
            override fun onFinish() {
                isCountDownRunning = false
                state = State.NORMAL
            }
        }
    }
    private var isCountDownRunning = false

    /**
     * If not set, the setText method will be called,
     * otherwise the set listener will be called
     * param1:state;
     * param2:preState;
     */
    var onStatusChangedListener: ((@State Int, @State Int?) -> Unit)? = null

    init {
        context.obtainStyledAttributes(attrs, R.styleable.LoadingStatusTextButton).apply {
            getString(R.styleable.LoadingStatusTextButton_normalText)?.let {
                normalText = it
                binding.btnAction.text = it
            }
            getString(R.styleable.LoadingStatusTextButton_successText)?.let { successText = it }
            getString(R.styleable.LoadingStatusTextButton_failText)?.let { failText = it }

            getColor(
                R.styleable.LoadingStatusTextButton_normalTextColor,
                ThemeUtils.getAttrValue(context, R.attr.colorPrimary)
            ).let { normalTextColor = it }
            getColor(
                R.styleable.LoadingStatusTextButton_successTextColor,
                ContextCompat.getColor(context, R.color.green_300_2)
            ).let { successTextColor = it }
            getColor(
                R.styleable.LoadingStatusTextButton_failTextColor,
                ThemeUtils.getAttrValue(context, R.attr.colorError)
            ).let { failTextColor = it }

            recycle()
        }
    }

    var preState: Int? = null

    @State
    var state = State.NORMAL
        set(value) {
            if (isCountDownRunning) {
                Logger.i(TAG, "status can't be set, because count down running")
                return
            }
            preState = field
            field = value
            Logger.i(TAG, "set state, state=${State.toLog(value)}")
            binding.btnAction.isVisible = value != State.LOADING
            binding.loadingView.isVisible = value == State.LOADING
            setText(value)
            onStatusChangedListener?.invoke(value, preState)
        }

    fun setText(status: Int) {
        when (status) {
            State.SUCCESS -> {
                binding.btnAction.text = successText
                binding.btnAction.setTextColor(successTextColor)
            }
            State.FAIL -> {
                binding.btnAction.text = failText
                binding.btnAction.setTextColor(failTextColor)
            }
            else -> {
                binding.btnAction.text = normalText
                binding.btnAction.setTextColor(normalTextColor)
            }
        }
    }

    fun setText(text: String) {
        binding.btnAction.text = text
    }

    fun getText() = binding.btnAction.text?.toString()

    fun resetCountdown() {
        isCountDownRunning = true
        countDownTimer.cancel()
        countDownTimer.start()
    }

    fun resetPreState() {
        if (preState == null) {
            Logger.e(TAG, "reset preState error, preState is null")
            return
        }
        state = preState!!
    }

    fun isLoading() = state == State.LOADING

    override fun onDetachedFromWindow() {
        super.onDetachedFromWindow()
        countDownTimer.cancel()
    }

    override fun setEnabled(enabled: Boolean) {
        super.setEnabled(enabled)
        if (enabled) {
            state = state
            binding.btnAction.alpha = 1f
        } else {
            binding.btnAction.setTextColor(ThemeUtils.getAttrValue(context, R.attr.colorOnSurface))
            binding.btnAction.alpha = 0.38f
        }
    }

    @Target(
        AnnotationTarget.CLASS,
        AnnotationTarget.PROPERTY,
        AnnotationTarget.TYPE,
        AnnotationTarget.VALUE_PARAMETER
    )
    @IntDef(
        State.NORMAL,
        State.LOADING,
        State.SUCCESS,
        State.FAIL
    )
    annotation class State {
        companion object {
            const val NORMAL = 0
            const val LOADING = 1
            const val SUCCESS = 2
            const val FAIL = 3

            fun toLog(@State state: Int) =
                when (state) {
                    LOADING -> {
                        "loading"
                    }
                    SUCCESS -> {
                        "success"
                    }
                    FAIL -> {
                        "fail"
                    }
                    else -> {
                        "normal"
                    }
                }
        }
    }

    companion object {
        const val TAG = "[UI][LoadingStatusTextButton]"
        const val NO_TEXT = ""
        private const val RESET_TIME = 3000L
    }
}
