package io.agora.avc.widget

import android.content.Context
import android.util.AttributeSet
import android.view.LayoutInflater
import android.view.MotionEvent
import android.view.View
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.core.view.isVisible
import io.agora.avc.BR
import io.agora.avc.bo.valoran.ARoomUser
import io.agora.avc.databinding.LayoutMeetingControllerBinding
import io.agora.avc.databinding.LayoutMeetingControllerLandBinding
import io.agora.avc.utils.ScreenUtils
import io.agora.logger.Logger

class MeetingControllerView @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0
) : ConstraintLayout(context, attrs, defStyleAttr), View.OnClickListener {

    var touchEnable = true

    private var onClickListener: OnClickListener? = null

    private val bindingPortrait by lazy {
        LayoutMeetingControllerBinding.inflate(LayoutInflater.from(context), this, true)
            .apply {
                setVariable(BR.onClickListener, this@MeetingControllerView)
            }
    }

    private val bindingLand by lazy {
        LayoutMeetingControllerLandBinding.inflate(LayoutInflater.from(context), this, true)
            .apply {
                setVariable(BR.onClickListener, this@MeetingControllerView)
            }
    }

    init {
        initView(ScreenUtils.isPortrait())
    }

    fun recreate(isPortrait: Boolean) {
        initView(isPortrait)
    }

    private fun initView(isPortrait: Boolean) {
        Logger.i(TAG, "init view, screen orientation is portrait:${isPortrait}")
        bindingPortrait.root.isVisible = isPortrait
        bindingLand.root.isVisible = isPortrait.not()
    }

    override fun dispatchTouchEvent(ev: MotionEvent?): Boolean {
        return if (touchEnable) {
            super.dispatchTouchEvent(ev)
        } else {
            false
        }
    }

    override fun onClick(v: View?) {
        onClickListener?.onClick(v)
    }

    override fun setOnClickListener(l: OnClickListener?) {
        super.setOnClickListener(l)
        onClickListener = l
    }

    fun setAttendeesNumber(number: Int) {
        Logger.i(TAG, "set attendees number, number=$number")
        bindingPortrait.ivAttendees.setAttendeesNumber(number)
        bindingLand.ivAttendees.setAttendeesNumber(number)
    }

    fun setHasUnreadMessage(hasUnreadMessage: Boolean) {
        Logger.i(TAG, "set has unread message:$hasUnreadMessage")
        bindingLand.ivChat.hasUnread = hasUnreadMessage
        bindingPortrait.ivDot.isVisible = hasUnreadMessage
    }

    fun setScreenShare(isSharingScreen: Boolean) {
        if (bindingLand.ivScreenShare.opened != isSharingScreen) {
            Logger.i(TAG, "set screen share:$isSharingScreen")
            bindingLand.ivScreenShare.opened = isSharingScreen
            if (isSharingScreen) {
                bindingLand.ivScreenShare.startAnimator()
            } else {
                bindingLand.ivScreenShare.stopAnimator()
            }
        }
    }

    fun refreshData(user: ARoomUser? = null, isAgoraRoom: Boolean, isSharingScreen: Boolean) {
        user?.let {
            bindingPortrait.ivLocalAudio.mute = !it.audioState
            bindingLand.ivLocalAudio.mute = !it.audioState

            if (it.audioState) {
                bindingPortrait.ivLocalAudio.volume = it.volume
                bindingLand.ivLocalAudio.volume = it.volume
            }

            bindingPortrait.ivLocalVideo.opened = it.videoState
            bindingLand.ivLocalVideo.opened = it.videoState

            setScreenShare(isSharingScreen)
            bindingLand.ivCloudRecording.opened = it.isCloudRecording
            bindingLand.flCloudRecording.isVisible =
                isAgoraRoom && user.isThirdPartyLoggedIn == true
        }
    }

    fun isLocalAudioOn() = !bindingLand.ivLocalAudio.mute
    fun isLocalVideoOn() = bindingLand.ivLocalVideo.opened
    fun isScreenSharing() = bindingLand.ivScreenShare.opened
    fun isCloudRecording() = bindingLand.ivCloudRecording.opened

    fun getLocalVideo() = bindingPortrait.ivLocalVideo
    fun getLocalAudio() = bindingPortrait.ivLocalAudio

    fun getLocalVideoL() = bindingLand.ivLocalVideo
    fun getLocalAudioL() = bindingLand.ivLocalAudio
    fun getLocalScreenShareL() = bindingLand.ivScreenShare
    fun getMenu() = bindingPortrait.flMenu

    companion object {
        private const val TAG = "[UI][MeetingControllerView]"
    }
}
