package io.agora.avc.widget

import android.content.Context
import android.util.AttributeSet
import android.view.LayoutInflater
import android.widget.ArrayAdapter
import android.widget.Filter
import androidx.constraintlayout.widget.ConstraintLayout
import io.agora.avc.R
import io.agora.avc.config.RESOLUTION_MEDIUM
import io.agora.avc.config.resolutionDetail
import io.agora.avc.databinding.LayoutSettingsItemResolutionBinding

class ResolutionSetting @JvmOverloads constructor(
    context: Context,
    attributeSet: AttributeSet? = null,
    defStyleAttr: Int = 0
) : ConstraintLayout(context, attributeSet, defStyleAttr) {

    private val resolutionValues by lazy {
        listOf("360P", "480P", "720P")
    }

    private val binding = LayoutSettingsItemResolutionBinding.inflate(
        LayoutInflater.from(context),
        this,
        true
    )

    private var mResolutionChangedListener: OnResolutionChangedListener? = null

    private var isScopeRoom = false

    init {
        val a = context.obtainStyledAttributes(attributeSet, R.styleable.ResolutionSetting)
        isScopeRoom = a.getInt(R.styleable.ResolutionSetting_scope, 0) != 0
        a.recycle()

        val current = RESOLUTION_MEDIUM

        setResolution(current)

        binding.tvResolution.setAdapter(
            object : ArrayAdapter<String>(
                context,
                R.layout.item_simple_dropdown_item_1line,
                resolutionValues
            ) {
                override fun getFilter(): Filter {
                    return object : Filter() {
                        override fun performFiltering(constraint: CharSequence?): FilterResults {
                            val result = FilterResults()
                            result.values = resolutionValues
                            result.count = resolutionValues.size
                            return result
                        }

                        override fun publishResults(constraint: CharSequence?, results: FilterResults) {
                            if (results.count > 0) {
                                notifyDataSetChanged()
                            } else {
                                notifyDataSetInvalidated()
                            }
                        }
                    }
                }
            }
        )

        binding.tvResolution.setOnItemClickListener { _, _, position, _ ->
            setCurrentResolution(position)
        }

        binding.tvResolution.setOnDismissListener {
            binding.tvResolution.clearFocus()
        }
    }

    fun setResolution(resolution: Int) {
        binding.tvResolution.setText(resolutionDetail(resolution), false)
    }

    private fun setCurrentResolution(resolution: Int) =
        mResolutionChangedListener?.onResolutionChanged(resolution)

    fun setOnResolutionChangedListener(listener: OnResolutionChangedListener) {
        this.mResolutionChangedListener = listener
    }

    interface OnResolutionChangedListener {
        fun onResolutionChanged(position: Int)
    }
}