package io.agora.avc.widget

import android.content.Context
import android.util.AttributeSet
import android.view.GestureDetector
import android.view.MotionEvent
import android.view.ScaleGestureDetector
import android.view.View
import android.widget.FrameLayout
import androidx.core.view.GestureDetectorCompat

class ScaleLayout @JvmOverloads constructor(
    context: Context, attrs: AttributeSet? = null, defStyleAttr: Int = 0
) : FrameLayout(context, attrs, defStyleAttr) {

    private var gestureDetector: GestureDetectorCompat? = null
    private var gestureDetectorListener: GestureDetector.OnGestureListener? = null
    private var scaleGestureDetector: ScaleGestureDetector? = null
    private var scaleGestureListener: ScaleGestureDetector.SimpleOnScaleGestureListener? = null
    private var renderAdapter: RenderAdapter? = null
    private val minScale = 1f
    private val maxScale = 3f
    private var currentScale = minScale
    private var oldScaleValue = minScale
    private var currentDistanceX = 0f
    private var currentDistanceY = 0f

    init {
        gestureDetectorListener = object : GestureDetector.OnGestureListener {
            override fun onDown(e: MotionEvent): Boolean {
                return false
            }

            override fun onShowPress(e: MotionEvent) {}
            override fun onSingleTapUp(e: MotionEvent): Boolean {
                callOnClick()
                return true
            }

            override fun onScroll(
                e1: MotionEvent?,
                e2: MotionEvent?,
                distanceX: Float,
                distanceY: Float
            ): Boolean {
                onScroll(distanceX, distanceY)
                return false
            }

            override fun onLongPress(e: MotionEvent?) {
                e?.let {
                    performLongClick()
                }
            }

            override fun onFling(
                e1: MotionEvent?,
                e2: MotionEvent?,
                velocityX: Float,
                velocityY: Float
            ): Boolean {
                return false
            }
        }
        scaleGestureListener = object : ScaleGestureDetector.SimpleOnScaleGestureListener() {
            override fun onScale(detector: ScaleGestureDetector): Boolean {
                renderAdapter?.getVideoView()?.let { renderView ->
                    val scale = detector.scaleFactor
                    var targetScale = if (scale < 1) {
                        currentScale + (scale - oldScaleValue) * 2
                    } else {
                        currentScale + (scale - oldScaleValue) / 2
                    }
                    oldScaleValue = scale
                    if (targetScale <= minScale) {
                        targetScale = minScale
                    } else if (targetScale >= maxScale) {
                        targetScale = maxScale
                    }

                    if (targetScale in minScale..maxScale) {
                        currentScale = targetScale
                        renderView.scaleX = currentScale
                        renderView.scaleY = currentScale

                        renderAdapter?.getWatermarkView()?.apply {
                            scaleX = currentScale
                            scaleY = currentScale
                        }
                    }
                }
                return super.onScale(detector)
            }

            override fun onScaleBegin(detector: ScaleGestureDetector): Boolean {
                oldScaleValue = 1f
                return super.onScaleBegin(detector)
            }
        }
        gestureDetector = GestureDetectorCompat(context, gestureDetectorListener)
        scaleGestureDetector = ScaleGestureDetector(context, scaleGestureListener)
    }

    fun onScroll(distanceX: Float, distanceY: Float) {
        renderAdapter?.getVideoView()?.let { renderView ->
            val width = renderView.width
            val height = renderView.height
            var targetX = currentDistanceX - distanceX
            var targetY = currentDistanceY - distanceY
            val leftBound = -width * (currentScale - 1f) / 2f
            val rightBound = width * (currentScale - 1f) / 2f
            val topBound = height * (currentScale - 1f) / 2f
            val bottomBound = -height * (currentScale - 1f) / 2f
            if (targetX < leftBound) {
                targetX = leftBound
            } else if (targetX > rightBound) {
                targetX = rightBound
            }
            if (targetY > topBound) {
                targetY = topBound
            } else if (targetY < bottomBound) {
                targetY = bottomBound
            }
            currentDistanceX = targetX
            currentDistanceY = targetY
            renderView.translationX = currentDistanceX
            renderView.translationY = currentDistanceY

            renderAdapter?.getWatermarkView()?.apply {
                translationX = currentDistanceX
                translationY = currentDistanceY
            }
        }
    }

    override fun onTouchEvent(event: MotionEvent?): Boolean {
        return gestureDetector?.onTouchEvent(event) == true ||
                scaleGestureDetector?.onTouchEvent(event) == true
    }

    fun scaleToOrigin() {
        renderAdapter?.getVideoView()?.let { renderView ->
            scrollToCenter()
            currentDistanceX = 0f
            currentDistanceY = 0f
            renderView.scaleX = minScale
            renderView.scaleY = minScale

            renderAdapter?.getWatermarkView()?.apply {
                scaleX = minScale
                scaleY = minScale
            }

            currentScale = minScale
            oldScaleValue = minScale
        }
    }

    private fun scrollToCenter() {
        onScroll(currentDistanceX, currentDistanceY)
    }

    interface RenderAdapter {
        fun getVideoView(): View?
        fun getWatermarkView(): View?
    }

    fun setRenderAdapter(adapter: RenderAdapter) {
        this.renderAdapter = adapter
    }
}