package io.agora.avc.widget

import android.content.Context
import android.content.res.ColorStateList
import android.util.AttributeSet
import android.view.LayoutInflater
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.core.view.isVisible
import io.agora.avc.R
import io.agora.avc.databinding.LayoutSpeakingImageView2Binding
import io.agora.avc.utils.ThemeUtils

class SpeakingImageView2 @JvmOverloads constructor(
    context: Context, attrs: AttributeSet? = null, defStyleAttr: Int = 0
) : ConstraintLayout(context, attrs, defStyleAttr) {

    private val binding by lazy {
        LayoutSpeakingImageView2Binding.inflate(LayoutInflater.from(context), this)
    }

    var mute = true
        set(value) {
            field = value
            binding.ivMute.isVisible = value
            binding.progressBar.isVisible = !value
            if (binding.ivBg.isVisible) {
                binding.ivBg.isChecked = !value
            }
        }

    var volume = VOLUME_MAX
        set(value) {
            field = value
            if (!mute) {
                binding.progressBar.progress =
                    (VOLUME_MIN + (value.toFloat()) * (VOLUME_MAX - VOLUME_MIN).toFloat() / VOLUME_MAX).toInt()
            }
        }

    var loading = false
        set(value) {
            field = value
            binding.loadingView.isVisible = value
            if (value) {
                binding.ivMute.isVisible = false
                binding.progressBar.isVisible = false
            } else {
                mute = mute
            }
        }

    var speaking = false
        set(value) {
            field = value
            binding.ivSpeaking.isVisible = value
            if (value) {
                binding.loadingView.isVisible = false
                binding.ivMute.isVisible = false
                binding.progressBar.isVisible = false
            } else {
                mute = mute
            }
        }

    var isMySelf = false
        set(value) {
            field = value
            speaking = !value && speaking
            volume = if (value) VOLUME_MIN else VOLUME_MAX
        }

    init {
        context.obtainStyledAttributes(attrs, R.styleable.SpeakingImageView2).apply {
            getDimension(
                R.styleable.SpeakingImageView2_imageWidth,
                0f
            ).apply {
                binding.ivMute.layoutParams.width = this.toInt()
                binding.loadingView.layoutParams.width = this.toInt()
                binding.ivSpeaking.layoutParams.width = this.toInt()
            }

            getDimension(
                R.styleable.SpeakingImageView2_imageHeight,
                0f
            ).apply {
                binding.ivMute.layoutParams.height = this.toInt()
                binding.progressBar.layoutParams.height = this.toInt()
                binding.ivSpeaking.layoutParams.height = this.toInt()
            }

            getDrawable(R.styleable.SpeakingImageView2_progressDrawable).apply {
                binding.progressBar.progressDrawable = this
            }

            getColor(
                R.styleable.SpeakingImageView2_muteImageTint,
                ThemeUtils.getAttrValue(context, R.attr.colorError)
            ).apply {
                binding.ivMute.imageTintList = ColorStateList.valueOf(this)
            }

            getDrawable(R.styleable.SpeakingImageView2_muteImage).apply {
                binding.ivMute.setImageDrawable(this)
            }

            getDrawable(R.styleable.SpeakingImageView2_speakingImage).apply {
                binding.ivSpeaking.setImageDrawable(this)
            }

            getColor(
                R.styleable.SpeakingImageView2_speakingImageTint,
                ThemeUtils.getAttrValue(context, R.attr.colorError)
            ).apply {
                binding.ivSpeaking.imageTintList = ColorStateList.valueOf(this)
            }

            getDrawable(R.styleable.SpeakingImageView2_backgroundImage).apply {
                binding.ivBg.isVisible = this != null
                if (this != null) {
                    binding.ivBg.setImageDrawable(this)
                }
            }
            recycle()
        }
    }

    companion object {
        const val VOLUME_MAX = 255
        const val VOLUME_MIN = 110
    }
}