package io.agora.avc.widget

import android.content.Context
import android.content.res.ColorStateList
import android.util.TypedValue
import android.view.LayoutInflater
import androidx.annotation.IntDef
import androidx.appcompat.app.AlertDialog
import androidx.core.content.ContextCompat
import androidx.core.view.isVisible
import androidx.core.view.updateLayoutParams
import androidx.databinding.DataBindingUtil
import com.google.android.material.dialog.MaterialAlertDialogBuilder
import io.agora.avc.R
import io.agora.avc.databinding.DialogUploadingBinding
import io.agora.avc.utils.ConvertUtils
import io.agora.avc.utils.ThemeUtils

class UploadingDialog(
    private val context: Context,
    private var status: Int,
    private var checkAgain: Boolean,
    private var content: CharSequence? = null,
    private val cancelClickListener: OnCancelClickListener?,
    private val retryClickListener: OnRetryClickListener?,
) {

    private var dialog: AlertDialog? = null
    private var cancelConfirmDialog: AlertDialog? = null

    private val binding by lazy {
        DataBindingUtil.inflate<DialogUploadingBinding>(
            LayoutInflater.from(context),
            R.layout.dialog_uploading,
            null,
            false
        ).apply {
            ivClose.setOnClickListener {
                dismiss()
            }
            button.setOnClickListener {
                if (this@UploadingDialog.status == UploadingStatus.ERROR) {
                    retryClickListener?.onRetryClick()
                } else if (this@UploadingDialog.status == UploadingStatus.STARTING && checkAgain) {
                    showCloseConfirmDialog()
                } else {
                    cancelClickListener?.onCancelClick()
                    dismiss()
                }
            }
        }
    }

    private constructor(builder: Builder) : this(
        builder.context,
        builder.status,
        builder.checkAgain,
        builder.content,
        builder.cancelClickListener,
        builder.retryClickListener,
    )

    fun show(status: Int, content: CharSequence? = null) {
        content?.let {
            this.content = it
        }
        if (!isShowing()) {
            val margin = context.resources.getDimension(R.dimen.inset_start).toInt()
            this.dialog = MaterialAlertDialogBuilder(context, R.style.UploadStatusAlertDialog)
                .setView(binding.root)
                .setBackgroundInsetStart(margin)
                .setBackgroundInsetEnd(margin)
                .setBackgroundInsetTop(margin)
                .setBackgroundInsetBottom(margin)
                .setCancelable(false)
                .create()
            this.dialog?.show()
        }
        changeStatus(status)
    }

    fun changeStatus(status: Int) {
        this.status = status
        val showButton = cancelClickListener != null
        when (status) {
            UploadingStatus.COMPLETED -> {
                dismissCancelConfirmDialog()
                binding.icon.setImageResource(R.drawable.ic_upload_status_success)
                binding.icon.imageTintList = ColorStateList.valueOf(getPrimaryColor())
                binding.status.text = context.getString(R.string.dump_uploaded)
                binding.progressBar.isIndeterminate = false
                binding.ivClose.isVisible = false
                binding.progressBar.progress = 100
                binding.button.isVisible = showButton
                binding.button.setBackgroundColor(
                    ContextCompat.getColor(
                        context,
                        R.color.transparent
                    )
                )
                binding.button.text = context.getString(R.string.ok_button)
            }
            UploadingStatus.STARTING -> {
                binding.progressBar.setIndicatorColor(
                    ThemeUtils.getAttrValue(context, R.attr.colorPrimary)
                )
                binding.icon.setImageResource(R.drawable.ic_upload_status_upload)
                binding.icon.imageTintList = ColorStateList.valueOf(getPrimaryColor())
                binding.status.text = context.getString(R.string.dump_uploading)
                binding.progressBar.isVisible = false
                binding.progressBar.isIndeterminate = true
                binding.progressBar.isVisible = true
                binding.ivClose.isVisible = false
                binding.button.updateLayoutParams {
                    width = ConvertUtils.dp2px(128f)
                }
                binding.button.apply {
                    text = context.getString(R.string.quit_button)
                    setBackgroundColor(ContextCompat.getColor(context, R.color.transparent))
                    setTextColor(ThemeUtils.getAttrValue(context, R.attr.chipTextColor2))
                    cornerRadius = ConvertUtils.dp2px(8f)
                    minWidth = ConvertUtils.dp2px(128f)
                    isVisible = showButton
                }
            }
            UploadingStatus.ERROR -> {
                dismissCancelConfirmDialog()
                binding.progressBar.setIndicatorColor(
                    ContextCompat.getColor(context, R.color.red_400_3)
                )
                binding.progressBar.trackColor =
                    ContextCompat.getColor(context, android.R.color.transparent)
                binding.progressBar.isIndeterminate = false
                binding.ivClose.isVisible = true
                binding.progressBar.progress = 100
                binding.icon.setImageResource(R.drawable.ic_upload_status_fail)
                binding.icon.imageTintList =
                    ColorStateList.valueOf(ContextCompat.getColor(context, R.color.red_400_3))
                binding.status.text = context.getString(R.string.dump_uploaded_failed)
                binding.button.apply {
                    text = context.getString(R.string.audio_mos_button_retry)
                    setBackgroundColor(
                        ThemeUtils.getAttrValue(
                            context,
                            R.attr.colorError
                        )
                    )
                    minWidth = ConvertUtils.dp2px(92f)
                    setTextColor(ContextCompat.getColor(context, R.color.white))
                    cornerRadius = ConvertUtils.dp2px(4f)
                    isVisible = true
                }
            }
            else -> {
            }
        }
        binding.content.isVisible = content != null
        content?.let {
            binding.content.text = it
        }
        binding.content.text = content
    }

    private fun getPrimaryColor(): Int {
        val typedValue = TypedValue()
        binding.root.context.theme.resolveAttribute(R.attr.colorPrimary, typedValue, true)
        return typedValue.data
    }

    private fun showCloseConfirmDialog() {
        cancelConfirmDialog =
            MaterialAlertDialogBuilder(context, R.style.CustomMaterialAlertDialog)
                .setCancelable(false)
                .setMessage(context.getString(R.string.notice_bug_report_cancel_upload_message))
                .setNegativeButton(R.string.notice_bug_report_cancel_upload_cancel) { _, _ ->

                }
                .setPositiveButton(R.string.notice_bug_report_cancel_upload_confirm) { _, _ ->
                    cancelClickListener?.onCancelClick()
                    dismiss()
                }
                .show()
    }

    fun isShowing(): Boolean {
        dialog?.let { alert ->
            return alert.isShowing
        }
        return false
    }

    private fun dismissCancelConfirmDialog() {
        cancelConfirmDialog?.apply {
            if (isShowing) {
                dismiss()
            }
        }
    }

    fun dismiss() {
        dismissCancelConfirmDialog()
        dialog?.let { alert ->
            if (alert.isShowing) {
                alert.dismiss()
            }
        }
    }

    class Builder {
        lateinit var context: Context
        var status: Int = UploadingStatus.IDLE
        var checkAgain: Boolean = false
        var content: CharSequence? = null
        var cancelClickListener: OnCancelClickListener? = null
        var retryClickListener: OnRetryClickListener? = null

        fun build() = UploadingDialog(this)
    }

    interface OnCancelClickListener {
        fun onCancelClick()
    }

    interface OnRetryClickListener {
        fun onRetryClick()
    }

    companion object {
        const val TAG: String = "UploadingDialog"
        inline fun build(block: Builder.() -> Unit) = Builder().apply(block).build()
    }
}

@IntDef(
    UploadingStatus.IDLE,
    UploadingStatus.STARTING,
    UploadingStatus.STOPPED,
    UploadingStatus.ERROR,
    UploadingStatus.COMPLETED,
)
@kotlin.annotation.Retention(AnnotationRetention.SOURCE)
annotation class UploadingStatus {
    companion object {
        const val IDLE = 0
        const val STARTING = 1
        const val STOPPED = 2
        const val ERROR = 3
        const val COMPLETED = 4
    }
}
