package io.agora.avc.widget

import android.content.Context
import android.graphics.Canvas
import android.graphics.Color
import android.graphics.Paint
import android.graphics.Rect
import android.util.AttributeSet
import android.view.View
import androidx.annotation.ColorInt
import io.agora.avc.R
import kotlin.math.max

/**
 * use case
<io.agora.avc.widget.Watermark
android:id="@+id/watermark"
android:layout_width="match_parent"
android:layout_height="0dp"
android:paddingTop="10dp"
app:degrees="-20"
app:horGap="100dp"
app:layout_constraintEnd_toEndOf="parent"
app:layout_constraintStart_toStartOf="parent"
app:layout_constraintTop_toBottomOf="@+id/actionBar"
app:markColor="@color/gray_b3300"
app:markSize="14sp"
app:markText="Watermark"
app:verticalGap="50dp" />
 */
class Watermark @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0
) : View(context, attrs, defStyleAttr) {

    private var paint: Paint = Paint()
    private var showText: String = "Watermark"
    private var markColor: Int = Color.WHITE
    private var markSize: Float = 36f
    private var verticalGap: Float = 100f
    private var horGap: Float = 50f
    private var degrees: Float = -20f
    private var markWidth = 0
    private var markHeight = 0
    private var minWidth = 100
    private var minHeight = 100
    private var data: ArrayList<Point> = arrayListOf()

    init {
        context.theme.obtainStyledAttributes(
            attrs,
            R.styleable.Watermark,
            0, 0
        ).apply {
            try {
                showText = getString(R.styleable.Watermark_markText) ?: "Watermark"
                markSize = getDimension(R.styleable.Watermark_markSize, 36f)
                markColor = getColor(R.styleable.Watermark_markColor, Color.WHITE)
                verticalGap = getDimension(R.styleable.Watermark_verticalGap, 100f)
                horGap = getDimension(R.styleable.Watermark_horGap, 50f)
                degrees = getFloat(R.styleable.Watermark_degrees, -20f)
            } finally {
                recycle()
            }
        }
        paint.apply {
            isAntiAlias = true
            textSize = markSize
            color = markColor
        }
        setMarkRect()
    }

    private fun setMarkRect() {
        val rect = Rect()
        paint.getTextBounds(showText, 0, showText.length, rect)
        markWidth = rect.width()
        markHeight = rect.height()
        if (markWidth == 0) {
            markHeight = minWidth
        }
        if (markHeight == 0) {
            markHeight = minHeight
        }
    }

    override fun onMeasure(widthMeasureSpec: Int, heightMeasureSpec: Int) {
        setMeasuredDimension(
            getDefaultSize(markWidth, widthMeasureSpec),
            getDefaultSize(markHeight, heightMeasureSpec)
        )
    }

    override fun onSizeChanged(w: Int, h: Int, oldw: Int, oldh: Int) {
        designData(w, h)
    }

    private fun designData(w: Int, h: Int) {
        val startOffset = max(paddingStart, paddingLeft)
        val endOffset = max(paddingEnd, paddingRight)
        val topOffset = paddingTop.toFloat()
        val bottomOffset = paddingBottom.toFloat()
        data.clear()
        var startX: Float = startOffset.toFloat()
        var startY: Float = markHeight + topOffset
        var lines = 1
        do {
            do {
                data.add(Point(startX, startY))
                startX += markWidth + horGap
            } while (startX < 1.5 * w - endOffset - markWidth)
            lines++
            startX = if (lines % 2 == 0) {
                startOffset.toFloat() + markWidth
            } else {
                startOffset.toFloat()
            }
            startY += markHeight + verticalGap
        } while (startY < 1.5 * h - bottomOffset)
    }

    private fun reDraw() {
        designData(measuredWidth, measuredHeight)
        invalidate()
    }

    override fun onDraw(canvas: Canvas) {
        canvas.save()
        canvas.translate(-0.25f * measuredWidth, -0.25f * measuredHeight)
        canvas.rotate(degrees, measuredWidth / 2f, measuredHeight / 2f)
        for (point in data) {
            canvas.drawText(showText, point.x, point.y, paint)
        }
        canvas.restore()
    }

    fun setDegree(degrees: Float) {
        this.degrees = degrees
        invalidate()
    }

    fun setHorGap(gap: Float) {
        this.horGap = gap
        reDraw()
    }

    fun setVerticalGap(gap: Float) {
        this.verticalGap = gap
        reDraw()
    }

    fun setMarkSize(size: Float) {
        this.markSize = size
        this.paint.textSize = size
        setMarkRect()
        reDraw()
    }

    fun setMarkColor(@ColorInt color: Int) {
        this.markColor = color
        this.paint.color = color
        invalidate()
    }

    fun setMarkText(showText: String) {
        this.showText = showText
        setMarkRect()
        reDraw()
    }

    data class Point(val x: Float, val y: Float)

    companion object {
        private const val TAG = "Watermark"
    }
}