package io.agora.avc.widget.helper

import android.content.Context
import android.util.AttributeSet
import android.view.View
import android.view.ViewTreeObserver.OnGlobalLayoutListener
import androidx.coordinatorlayout.widget.CoordinatorLayout
import com.google.android.material.bottomsheet.BottomSheetBehavior
import com.google.android.material.bottomsheet.BottomSheetBehavior.STATE_COLLAPSED
import com.google.android.material.bottomsheet.BottomSheetBehavior.STATE_HIDDEN
import io.agora.avc.R
import io.agora.logger.Logger

/**
 *  There can only be one child View
 */
class SmallStreamCoordinatorLayout @JvmOverloads constructor(
    context: Context, attrs: AttributeSet? = null, defStyleAttr: Int = 0
) : CoordinatorLayout(context, attrs, defStyleAttr) {

    private val bottomSheet by lazy {
        getChildAt(0)
    }

    private var bottomSheetBehavior: BottomSheetBehavior<View>? = null

    var peekHeight = 0
        set(value) {
            field = value
            bottomSheetBehavior?.peekHeight = value
        }

    var halfExpandedRatio = 0f
        set(value) {
            field = value
            bottomSheetBehavior?.halfExpandedRatio = value
        }

    var bottomSheetCallback: BottomSheetBehavior.BottomSheetCallback? = null

    var isMediaListVisible = false

    init {

        context.obtainStyledAttributes(attrs, R.styleable.SmallStreamCoordinatorLayout).apply {
            peekHeight =
                getDimension(R.styleable.SmallStreamCoordinatorLayout_peekHeight, 30f).toInt()
            halfExpandedRatio =
                getFloat(R.styleable.SmallStreamCoordinatorLayout_halfExpandedRatio, 0.5f)
            recycle()
        }

        viewTreeObserver.addOnGlobalLayoutListener(object : OnGlobalLayoutListener {
            override fun onGlobalLayout() {
                viewTreeObserver.removeOnGlobalLayoutListener(this)
                bottomSheetBehavior = BottomSheetBehavior.from(bottomSheet).apply {
                    halfExpandedRatio = this@SmallStreamCoordinatorLayout.halfExpandedRatio
                    peekHeight = this@SmallStreamCoordinatorLayout.peekHeight
                    isFitToContents = false
                    isGestureInsetBottomIgnored = true
                    addBottomSheetCallback(object : BottomSheetBehavior.BottomSheetCallback() {
                        override fun onStateChanged(bottomSheet: View, newState: Int) {
                            Logger.i(TAG, "onStateChanged: newState:$newState")
                            bottomSheetCallback?.onStateChanged(bottomSheet, newState)
                            isMediaListVisible =
                                newState != STATE_COLLAPSED && newState != STATE_HIDDEN
                        }

                        override fun onSlide(bottomSheet: View, slideOffset: Float) {
                            isMediaListVisible = slideOffset >= 0.1f
                            if (slideOffset > halfExpandedRatio) {
                                bottomSheet.layoutParams.apply {
                                    height =
                                        (this@SmallStreamCoordinatorLayout.height * slideOffset).toInt()
                                    bottomSheet.layoutParams = this
                                }
                            } else {
                                bottomSheet.layoutParams.apply {
                                    height =
                                        (this@SmallStreamCoordinatorLayout.height * halfExpandedRatio).toInt()
                                    bottomSheet.layoutParams = this
                                }
                            }

                            bottomSheetCallback?.onSlide(bottomSheet, slideOffset)
                        }
                    })
                }
            }
        })
    }

    fun turnToHalfExpanded() {
        bottomSheetBehavior?.state = BottomSheetBehavior.STATE_HALF_EXPANDED
    }

    fun isCollapsed() = bottomSheetBehavior?.state == BottomSheetBehavior.STATE_COLLAPSED
    fun isHalfExpanded() = bottomSheetBehavior?.state == BottomSheetBehavior.STATE_HALF_EXPANDED
    fun isExpanded() = bottomSheetBehavior?.state == BottomSheetBehavior.STATE_EXPANDED

    companion object {
        const val TAG = "[UI][SmallStreamCoordinatorLayout]"
        const val MEDIA_LIST_VISIBLE_RATIO = 0.1f
    }
}

