package io.agora.avc

import android.app.Activity
import android.app.Application
import android.content.Context
import android.os.Bundle
import android.os.StrictMode
import androidx.appcompat.app.AppCompatDelegate
import androidx.multidex.MultiDex
import com.google.gson.JsonSyntaxException
import io.agora.avc.base.AppContainer
import io.agora.avc.bo.RoomMode
import io.agora.avc.bo.UserInfo
import io.agora.avc.di.component.ApplicationComponent
import io.agora.avc.di.component.DaggerApplicationComponent
import io.agora.avc.manager.network.NetWorkManager
import io.agora.avc.manager.splite.SPLiteImp
import io.agora.avc.manager.splite.SPLiteProxy
import io.agora.avc.utils.AppUtils
import io.agora.avc.utils.DeviceUtils
import io.agora.avc.utils.GsonUtils
import io.agora.avc.utils.ProcessUtils
import io.agora.frame.base.BaseApplication
import io.agora.logger.AppInfo
import io.agora.logger.Log
import io.agora.logger.Logger
import io.reactivex.plugins.RxJavaPlugins
import javax.inject.Inject

open class MyApplication : BaseApplication() {

    val appContainer: AppContainer = AppContainer()

    @Inject
    lateinit var netWorkManager: NetWorkManager

    override fun attachBaseContext(base: Context?) {
        super.attachBaseContext(base)
        app = this
        MultiDex.install(this)
        initSPLite()
        initApplicationDelegate(base)
    }

    override fun onCreate() {
        useStrictMode()
        super.onCreate()
        appContext = applicationContext
        //judgement process by pid
        isMainProcess = ProcessUtils.isMainProcess()
        initLogger()
        initDagger()
        setAppTheme()
        initRxJavaErrorHandler()
        initActivityLifecycleCallbacks()
        installNetWorkManager()
    }

    private fun useStrictMode() {
        if (BuildConfig.DEBUG) {
            StrictMode.setThreadPolicy(
                StrictMode.ThreadPolicy.Builder()
                    .detectAll()
                    .penaltyLog()
                    .build()
            )
            StrictMode.setVmPolicy(
                StrictMode.VmPolicy.Builder()
                    .detectLeakedSqlLiteObjects()
                    .detectLeakedClosableObjects()
                    .penaltyLog()
                    .build()
            )
        }
    }

    private fun initSPLite() {
        SPLiteProxy.setSpLite(SPLiteImp().apply {
            migrate()
        })
    }

    private fun initRxJavaErrorHandler() {
        RxJavaPlugins.setErrorHandler { t -> Logger.e(TAG, "RxJava error handler", t) }
    }

    private fun initLogger() {
        val appInfo = AppInfo()
        appInfo.appVersionName = AppUtils.getAppVersionName()
        appInfo.appVersionCode = AppUtils.getAppVersionCode()
        appInfo.processName = ProcessUtils.getCurrentProcessName()
        appInfo.brand = DeviceUtils.getBrand()
        appInfo.model = DeviceUtils.getModel()
        appInfo.sdkVersionName = DeviceUtils.getSDKVersionName()
        Logger.init(this, isMainProcess, BuildConfig.DEBUG, appInfo)
        Logger.printer(Log())
    }

    private fun initDagger() {
        val appComponent: ApplicationComponent = DaggerApplicationComponent.builder()
            .appComponent(appComponent)
            .build()
        appComponent.inject(this)
    }

    private fun setAppTheme() {
        if (isMainProcess) {
            Logger.i(TAG, "set app theme")
            val theme = SPLiteProxy.getDarkTheme()
            if (theme == AppCompatDelegate.MODE_NIGHT_YES) {
                AppCompatDelegate.setDefaultNightMode(AppCompatDelegate.MODE_NIGHT_YES)
            } else if (theme == AppCompatDelegate.MODE_NIGHT_NO) {
                AppCompatDelegate.setDefaultNightMode(AppCompatDelegate.MODE_NIGHT_NO)
            }

            if (SPLiteProxy.getRoomMode() == RoomMode.AGORA.value) {
                val tokenJson = SPLiteProxy.getUserInfo()
                val userInfo = try {
                    GsonUtils.fromJson(tokenJson, UserInfo::class.java)
                } catch (e: JsonSyntaxException) {
                    Logger.e(TAG, "get user info error, error:", e)
                    null
                }
                if (userInfo?.innerInfo == null) {
                    Logger.w(TAG, "reset room mode")
                    SPLiteProxy.setRoomMode(RoomMode.NORMAL.value)
                }
            }
        }
    }

    private fun initActivityLifecycleCallbacks() {
        registerActivityLifecycleCallbacks(object : ActivityLifecycleCallbacks {
            override fun onActivityCreated(activity: Activity, savedInstanceState: Bundle?) {
                Logger.i(TAG, "${activity.javaClass.canonicalName} created")
            }

            override fun onActivityStarted(activity: Activity) {
                activityCount++
                if (activityCount == 1) {
                    Logger.i(TAG, "Application back to foreground")
                }
            }

            override fun onActivityResumed(activity: Activity) {
            }

            override fun onActivityPaused(activity: Activity) {
            }

            override fun onActivityStopped(activity: Activity) {
                activityCount--
                if (activityCount == 0) {
                    Logger.i(TAG, "Application back to background")
                }
            }

            override fun onActivitySaveInstanceState(activity: Activity, outState: Bundle) {
            }

            override fun onActivityDestroyed(activity: Activity) {
                Logger.i(TAG, "${activity.javaClass.canonicalName} destroyed")
            }
        })
    }

    private fun installNetWorkManager() {
        netWorkManager.install()
    }

    companion object {
        const val TAG = "[Comm][App]"

        var activityCount: Int = 0

        fun isAppForeground(): Boolean = activityCount > 0

        var isMainProcess: Boolean = false

        lateinit var app: Application
            private set

        lateinit var appContext: Context
            private set
    }
}
