package io.agora.avc.app.address

import android.content.res.Configuration
import android.os.Bundle
import androidx.appcompat.app.AlertDialog
import androidx.appcompat.content.res.AppCompatResources
import androidx.core.os.bundleOf
import androidx.core.view.isVisible
import androidx.navigation.fragment.NavHostFragment
import androidx.recyclerview.widget.DividerItemDecoration
import androidx.recyclerview.widget.RecyclerView
import com.google.android.material.dialog.MaterialAlertDialogBuilder
import io.agora.avc.R
import io.agora.avc.bo.AppConfig
import io.agora.avc.bo.LocalUser
import io.agora.avc.databinding.AvcFragmentAddressBookContainerBinding
import io.agora.avc.extensions.safeNavigate
import io.agora.avc.widget.OnActionTxtClickListener
import io.agora.frame.base.NovelFragment

const val OVER_VISIBLE_NUMBER_TEXT = "999+"
const val MAX_VISIBLE_NUMBER = 999

class AddressBookContainerFragment :
    NovelFragment<AddressBookContainerViewModel, AvcFragmentAddressBookContainerBinding>() {

    private var localUser: LocalUser? = null
    private var dept: GroupNode? = null
    private var appConfig: AppConfig? = null
    private var callConfirmDialog: AlertDialog? = null
    private var operationType: Int = OPERATION_INVITE
    private var dataType: Int = DATA_ORG
    private var deptId: String = ROOT_DEPT_NO
    private var groupInfo: GroupInfo? = null

    private var maxSizeLayoutManager: MaxSizeLayoutManager? = null
    private var navHostFragment: NavHostFragment? = null

    private val selectedAdapter by lazy {
        UserSelectedAdapter().apply {
            setOnUserSelectedListener(object : UserSelectedAdapter.OnUserSelectedListener {
                override fun onItemClicked(node: MemberNode, position: Int) {
                    mViewModel?.uncheckMember(node)
                    notifySelectedChanged()
                }
            })
        }
    }

    private val dividerItemDecoration by lazy {
        DividerItemDecoration(context, DividerItemDecoration.HORIZONTAL).also { decoration ->
            AppCompatResources.getDrawable(requireContext(), R.drawable.avc_shape_user_selected_divider)
                ?.let {
                    decoration.setDrawable(it)
                }
        }
    }

    override fun getLayoutId(): Int {
        return R.layout.avc_fragment_address_book_container
    }

    override fun allocObserver() {
        mViewModel?.appConfigLiveData?.observe(this) { config ->
            this.appConfig = config
        }
        mViewModel?.deptLiveData?.observe(this) { dept ->
            this.dept = dept
            mBinding.actionBar.setTitle(dept.name)
        }
        mViewModel?.localUserLiveData?.observe(this) { user ->
            localUser = user
        }
        mViewModel?.selectedLiveData?.observe(this) { list ->
            selectedAdapter.setNewData(list)
            scrollToEnd(false)
            refreshSelectedNumber(list.size)
            invalidateCallable(list)
        }
        mViewModel?.callResultEvent?.observe(this) { success ->
            if (success) {
                onBackPressed()
            } else {
                mBinding.btnCall.isEnabled = true
            }
        }
        mViewModel?.addMemberEvent?.observe(this) { success ->
            // local addition must succeed
            if (success) {
                onBackPressed()
            }
        }
        mViewModel?.groupInfoChangedEvent?.observe(this) { info ->
            this.groupInfo = info
            setActionTxt(info)
        }
    }

    private fun refreshSelectedNumber(size: Int) {
        if (size == 0) {
            mBinding.tvCount.isVisible = false
            return
        }
        mBinding.tvCount.isVisible = true
        if (size > MAX_VISIBLE_NUMBER) {
            mBinding.tvCount.text = OVER_VISIBLE_NUMBER_TEXT
        } else {
            mBinding.tvCount.text = "$size"
        }
    }

    private fun notifySelectedChanged() {
        navHostFragment?.childFragmentManager?.fragments?.apply {
            for (fragment in this) {
                if (fragment is AddressBookFragment) {
                    fragment.onSelectedChanged()
                }
            }
        }
    }

    private fun setActionTxt(info: GroupInfo?) {
        if (info == null) {
            mBinding.actionBar.setActionTxt(null)
            return
        }
        if (info.isGroupList) {
            mBinding.actionBar.setActionTxt(getString(R.string.avc_group_create_mobile))
        } else if (info.isLeaf && info.isAllSelected) {
            mBinding.actionBar.setActionTxt(getString(R.string.avc_group_deselect_all))
        } else if (info.isLeaf && !info.isAllSelected) {
            mBinding.actionBar.setActionTxt(getString(R.string.avc_group_select_all))
        } else {
            mBinding.actionBar.setActionTxt(null)
        }
    }

    private fun invalidateCallable(list: List<MemberNode>?) {
        val size = list?.size ?: 0
        mBinding.btnCall.isEnabled = size > 0
        val describe = if (isOperationAdd()) {
            getString(R.string.avc_group_add_member)
        } else {
            getString(R.string.avc_invite_button)
        }
        val callText = if (size > 0) {
            "$describe($size)"
        } else {
            describe
        }
        mBinding.btnCall.text = callText
        adjustEditStyle(size)
    }

    private fun adjustEditStyle(size: Int) {
        mBinding.ivSearch.isVisible = size <= 0
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        val operationType = arguments?.getInt(KEY_OPERATION_TYPE) ?: OPERATION_INVITE
        this.operationType = operationType
        val dataType = arguments?.getInt(KEY_DATA_TYPE) ?: DATA_ORG
        this.dataType = dataType
        val deptId = arguments?.getString(KEY_DEPT_ID) ?: ROOT_DEPT_NO
        this.deptId = deptId
        if (operationType == 0 || dataType == 0) {
            throw IllegalArgumentException("params is illegal,ot:$operationType,dt:$dataType")
        }
        mViewModel?.initParams(deptId, operationType, dataType)
        val fragment = NavHostFragment.create(
            R.navigation.avc_nav_address_book_graph, bundleOf(
                KEY_DEPT_ID to deptId,
                KEY_OPERATION_TYPE to operationType,
                KEY_DATA_TYPE to dataType
            )
        )
        childFragmentManager.beginTransaction().replace(R.id.fragment_container, fragment).commit()
        this.navHostFragment = fragment
    }

    override fun initialize(savedInstanceState: Bundle?) {
        mBinding.rvSelected.adapter = selectedAdapter
        maxSizeLayoutManager = createLayoutManager()
        mBinding.rvSelected.layoutManager = maxSizeLayoutManager
        mBinding.rvSelected.addItemDecoration(dividerItemDecoration)
        dept?.name?.let {
            mBinding.actionBar.setTitle(it)
        }
        mBinding.tvSearch.setOnClickListener {
            safeNavigate(
                R.id.action_address_book_container_to_search,
                bundleOf(KEY_OPERATION_TYPE to operationType)
            )
        }
        if (isOperationAdd()) {
            mBinding.btnCall.text = getString(R.string.avc_group_add_member)
        } else {
            mBinding.btnCall.text = getString(R.string.avc_invite_button)
        }
        mBinding.btnCall.setOnClickListener {
            if (isOperationAdd()) {
                mViewModel?.addMembers()
            } else {
                showCallConfirmDialog()
            }
        }
        mBinding.tvCount.setOnClickListener {
            safeNavigate(
                R.id.action_address_book_container_to_member_selected,
                bundleOf(KEY_OPERATION_TYPE to operationType)
            )
        }
        mBinding.actionBar.setOnActionTxtClickListener(object : OnActionTxtClickListener {
            override fun onActionTxtClicked() {
                navHostFragment?.childFragmentManager?.fragments?.apply {
                    for (fragment in this) {
                        if (fragment is AddressBookFragment) {
                            fragment.onActionTxtClicked(groupInfo)
                            return@apply
                        }
                    }
                }
            }
        })
        setActionTxt(groupInfo)
    }

    private fun isOperationAdd() = operationType == OPERATION_ADD

    override fun onConfigurationChanged(newConfig: Configuration) {
        super.onConfigurationChanged(newConfig)
        if (newConfig.orientation == Configuration.ORIENTATION_LANDSCAPE) {
            maxSizeLayoutManager?.setMaxWidth(LANDSCAPE_MAX_WIDTH)
        } else {
            maxSizeLayoutManager?.setMaxWidth(PORTRAIT_MAX_WIDTH)
        }
    }

    private fun scrollToEnd(animate: Boolean = true) {
        mBinding.rvSelected.post {
            if (mBinding == null) {
                return@post
            }
            (selectedAdapter.itemCount - 1).let {
                if (it > 0) {
                    if (animate) {
                        mBinding.rvSelected.smoothScrollToPosition(it)
                    } else {
                        mBinding.rvSelected.scrollToPosition(it)
                    }
                }
            }
        }
    }

    private fun createLayoutManager(): MaxSizeLayoutManager {
        val maxWidth =
            if (resources.configuration.orientation == Configuration.ORIENTATION_PORTRAIT) {
                PORTRAIT_MAX_WIDTH
            } else {
                LANDSCAPE_MAX_WIDTH
            }
        return MaxSizeLayoutManager(context, RecyclerView.HORIZONTAL, false, maxWidth)
    }

    private fun showCallConfirmDialog() {
        callConfirmDialog =
            MaterialAlertDialogBuilder(requireContext(), R.style.avc_CustomMaterialAlertDialog)
                .setTitle(R.string.avc_invite_ready_title)
                .setCancelable(true)
                .setNegativeButton(R.string.avc_invite_confirm_no) { _, _ ->

                }
                .setPositiveButton(R.string.avc_invite_confirm_yes) { _, _ ->
                    mBinding.btnCall.isEnabled = false
                    mViewModel?.startCall()
                }
                .show()
    }

    override fun onDestroyView() {
        callConfirmDialog?.let {
            if (it.isShowing) {
                it.dismiss()
            }
        }
        super.onDestroyView()
    }

    companion object {
        private const val TAG = "[UI][AddressBookContainer]"
        private const val LANDSCAPE_MAX_WIDTH = 540f
        private const val PORTRAIT_MAX_WIDTH = 220f
    }

}