package io.agora.avc.app.address

import android.app.Application
import androidx.lifecycle.MutableLiveData
import com.agora.valoran.bean.SimpleRoomUser
import io.agora.avc.app.group.MeetupResponse
import io.agora.avc.base.AppViewModel
import io.agora.avc.biz.AppController
import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.bo.AppConfig
import io.agora.avc.bo.LocalUser
import io.agora.avc.bo.MeetingNotificationType
import io.agora.avc.bo.NoticeCode
import io.agora.avc.manager.notice.NoticeEvent
import io.agora.avc.manager.notice.NoticeManager
import io.agora.avc.manager.notice.NoticeType
import io.agora.avc.manager.notice.NotifyResult
import io.agora.avc.repository.AddressBookRepository
import io.agora.avc.repository.AppConfigRepository
import io.agora.frame.base.livedata.EventLiveData
import io.agora.logger.Logger
import javax.inject.Inject

class AddressBookContainerViewModel @Inject constructor(
    application: Application,
    appController: AppController,
    private val addressBookRepository: AddressBookRepository,
    private val appConfigRepository: AppConfigRepository,
    private val noticeManager: NoticeManager,
) : AppViewModel(application, appController) {

    val appConfigLiveData = MutableLiveData<AppConfig?>()
    val deptLiveData = MutableLiveData<GroupNode>()
    val selectedLiveData = MutableLiveData<List<MemberNode>>()
    val localUserLiveData = MutableLiveData<LocalUser>()
    val callResultEvent = EventLiveData<Boolean>()
    val addMemberEvent = EventLiveData<Boolean>()
    val groupInfoChangedEvent = MutableLiveData<GroupInfo>()
    private var selectedData = arrayListOf<MemberNode>()
    private var currentSeq = -1
    private var currentDept = "-1"
    private var operationType = OPERATION_INVITE
    private var dataType = DATA_ORG
    private var appConfig: AppConfig? = null

    override fun onResume() {
        super.onResume()
        queryLocalUser()
        queryAppConfig()
        notifyUserSelectedChanged(selectedData)
    }

    private fun queryAppConfig() {
        this.appConfig = appConfigRepository.getAppConfig()
        appConfigLiveData.postValue(appConfig)
    }

    private fun notifyUserSelectedChanged(list: ArrayList<MemberNode>) {
        selectedLiveData.postValue(list)
    }

    private fun queryLocalUser() {
        getLocalUser()?.let { user ->
            localUserLiveData.postValue(user)
        }
    }

    fun initParams(deptId: String, operationType: Int, dataType: Int) {
        Logger.i(TAG, "init params, type:$operationType, dt:$dataType")
        this.currentSeq = appController.getOperationSeqId()
        this.currentDept = deptId
        this.dataType = dataType
        setOperationType(operationType)
    }

    /**
     * deselected user
     */
    fun uncheckMember(memberNode: MemberNode) {
        deselectUser(memberNode)
        notifyUserSelectedChanged(selectedData)
    }

    fun startCall() {
        val source = arrayListOf<SimpleRoomUser>()
        for (node in selectedData) {
            source.add(SimpleRoomUser().apply {
                thirdPartyId = node.userid
                thirdPartyDepartment = node.department ?: ""
                thirdPartyName = node.name
                thirdPartyAlias = node.alias
            })
        }
        if (source.isEmpty()) {
            Logger.i(TAG, "failed to request call, reason:none user")
            return
        }
        Logger.i(TAG, "start call request")
        currentSeq = appController.getOperationSeqId()
        appController.inviteContacts(currentSeq, source)
    }

    override fun onEventReceived(arg: MessageEvent) {
        when (arg.type) {
            AppEvent.CALL_RESULT_EVENT.ordinal -> {
                handleCallResult(arg.obj)
            }
            AppEvent.MEETUP_ADD_MEMBER_EVENT.ordinal -> {
                handleAddMemberResult(arg.obj)
            }
            AppEvent.ADDRESS_BOOK_SELECT_DATA_CHANGED_EVENT.ordinal -> {
                notifyUserSelectedChanged(selectedData)
            }
            AppEvent.ADDRESS_BOOK_GROUP_STATUS_CHANGED_EVENT.ordinal -> {
                arg.obj?.let { obj ->
                    if (obj is GroupInfo) {
                        groupInfoChangedEvent.postValue(obj)
                    }
                }
            }
            AppEvent.ADDRESS_BOOK_DEPT_INFO_CHANGED_EVENT.ordinal -> {
                arg.obj?.let { obj ->
                    if (obj is GroupNode) {
                        deptLiveData.postValue(obj)
                    }
                }
            }
        }
    }

    private fun handleAddMemberResult(data: Any?) {
        if (operationType == OPERATION_INVITE) {
            return
        }
        if (data !is MeetupResponse) {
            return
        }
        //local addition must succeed
        addMemberEvent.postValue(true)
    }

    private fun handleCallResult(data: Any?) {
        if (isOperationAdd()) {
            return
        }
        if (data !is NotifyResult) {
            return
        }
        noticeManager.notice(
            NoticeEvent(
                NoticeCode.CODE_INVITE_FEEDBACK,
                NoticeType.NOTIFICATION_C,
                MeetingNotificationType.NO_9
            )
        )
        callResultEvent.postValue(data.success)
    }

    override fun getUIEvents(): Array<AppEvent>? {
        return arrayOf(
            AppEvent.CALL_RESULT_EVENT,
            AppEvent.MEETUP_ADD_MEMBER_EVENT,
            AppEvent.ADDRESS_BOOK_SELECT_DATA_CHANGED_EVENT,
            AppEvent.ADDRESS_BOOK_GROUP_STATUS_CHANGED_EVENT,
            AppEvent.ADDRESS_BOOK_DEPT_INFO_CHANGED_EVENT
        )
    }

    override fun onCleared() {
        super.onCleared()
        if (currentDept == ROOT_DEPT_NO) {
            if (isOperationAdd()) {
                addressBookRepository.clearAdded()
            } else {
                addressBookRepository.clearInvitee()
            }
        }
    }

    fun addMembers() {
        val source = arrayListOf<MemberNode>()
        for (node in selectedData) {
            source.add(node)
        }
        if (source.isNotEmpty()) {
            currentSeq = appController.getOperationSeqId()
            appController.addMeetupMembers(currentSeq, source)
        }
    }

    private fun setOperationType(operationType: Int) {
        this.operationType = operationType
        selectedData = if (isOperationAdd()) {
            addressBookRepository.queryAdded()
        } else {
            addressBookRepository.queryInvitee()
        }
    }

    private fun isOperationAdd() = operationType == OPERATION_ADD

    private fun deselectUser(node: AddressBookNode) {
        if (node is MemberNode && !node.isJoined) {
            val position = selectedData.indexOf(node)
            if (position in 0 until selectedData.size) {
                selectedData.removeAt(position)
            }
        }
    }

    companion object {
        private const val TAG = "[VM][AddressBookContainer]"
    }
}