package io.agora.avc.app.address

import android.os.Bundle
import androidx.core.os.bundleOf
import androidx.core.view.isVisible
import androidx.navigation.Navigation
import androidx.navigation.fragment.NavHostFragment
import androidx.recyclerview.widget.LinearLayoutManager
import io.agora.avc.R
import io.agora.avc.bo.AppConfig
import io.agora.avc.bo.LocalUser
import io.agora.avc.databinding.AvcFragmentAddressBookBinding
import io.agora.avc.extensions.getConferenceNicknameMaybeAssistant
import io.agora.avc.extensions.safeNavigate
import io.agora.frame.base.NovelFragment

const val ROOT_DEPT_NO = "1"
const val KEY_DEPT_ID = "deptId"
const val KEY_DEPT_NAME = "deptName"
const val KEY_OPERATION_TYPE = "key_operation_type"
const val KEY_DATA_TYPE = "key_data_type"
const val OPERATION_INVITE = 1
const val OPERATION_ADD = 2
const val DATA_ORG = 11
const val DATA_CUSTOM = 12

class AddressBookFragment : NovelFragment<AddressBookViewModel, AvcFragmentAddressBookBinding>() {

    private var localUser: LocalUser? = null
    private var dept: GroupNode? = null
    private var appConfig: AppConfig? = null
    private var operationType: Int = OPERATION_INVITE
    private var dataType: Int = DATA_ORG
    private var deptId: String = ROOT_DEPT_NO
    private var deptName: String? = null

    private val addressBookListener: AddressBookListener = object : AddressBookListener {
        override fun onItemClicked(node: AddressBookNode, position: Int) {
            if (node is GroupNode) {
                safeNavigate(
                    R.id.action_address_book_to_self,
                    bundleOf(
                        KEY_DEPT_ID to node.deptId,
                        KEY_DEPT_NAME to node.name,
                        KEY_OPERATION_TYPE to operationType,
                        KEY_DATA_TYPE to dataType
                    )
                )
            } else if (node is MemberNode && !node.isJoined) {
                mViewModel?.toggleMember(node, position)
            } else if (node is MyGroupNode) {
                safeNavigate(
                    R.id.action_address_book_to_self,
                    bundleOf(
                        KEY_DEPT_ID to node.groupId,
                        KEY_DEPT_NAME to node.name,
                        KEY_OPERATION_TYPE to operationType,
                        KEY_DATA_TYPE to DATA_CUSTOM
                    )
                )
            }
        }

        override fun onToggleGroup(node: AddressBookNode, position: Int) {
            //todo select all/deselect all
        }

        override fun onRefreshClicked() {
            mViewModel?.queryDept(deptName, deptId, operationType, dataType)
        }

        override fun onLongClicked(node: AddressBookNode, position: Int) {
            if (node is MyGroupNode && deptId != ROOT_DEPT_NO) {
                GroupOperationFragment.navigateTo(this@AddressBookFragment, node.groupId, node.name)
            }
        }

        override fun onCreateGroupClicked() {
            goCreateGroup()
        }
    }

    private val pageAdapter by lazy {
        AddressBookAdapter(addressBookListener)
    }

    override fun getLayoutId(): Int {
        return R.layout.avc_fragment_address_book
    }

    override fun allocObserver() {
        mViewModel?.appConfigLiveData?.observe(this) { config ->
            this.appConfig = config
        }
        mViewModel?.localUserLiveData?.observe(this) { user ->
            localUser = user
            mBinding.watermark.setMarkText(user.getConferenceNicknameMaybeAssistant())
        }
        mViewModel?.membersLiveData?.observe(this) { list ->
            pageAdapter.setNewData(list)
        }
        mViewModel?.membersItemChangedEvent?.observe(this) { position ->
            pageAdapter.notifyItemChanged(position)
        }
        mViewModel?.addGroupEvent?.observe(this) { permit ->
            if (permit) {
                goCreateGroup()
            }
        }
    }

    private fun goCreateGroup() {
        Navigation
            .findNavController(requireActivity(), R.id.nav_fragment)
            .navigate(R.id.action_global_add_group)
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        val operationType = arguments?.getInt(KEY_OPERATION_TYPE) ?: OPERATION_INVITE
        this.operationType = operationType
        val dataType = arguments?.getInt(KEY_DATA_TYPE) ?: DATA_ORG
        this.dataType = dataType
        val deptId = arguments?.getString(KEY_DEPT_ID) ?: ROOT_DEPT_NO
        this.deptId = deptId
        this.deptName = arguments?.getString(KEY_DEPT_NAME)
        if (operationType == 0 || dataType == 0) {
            throw IllegalArgumentException("params is illegal,ot:$operationType,dt:$dataType")
        }
        mViewModel?.queryDept(this.deptName, deptId, operationType, dataType)
    }

    override fun initialize(savedInstanceState: Bundle?) {
        mBinding.rvUserList.adapter = pageAdapter
        mBinding.rvUserList.layoutManager = LinearLayoutManager(context)
        localUser?.let { user ->
            mBinding.watermark.setMarkText(user.getConferenceNicknameMaybeAssistant())
        }
    }

    override fun onBackPressed() {
        val success = NavHostFragment.findNavController(this).navigateUp()
        if (!success) {
            Navigation.findNavController(requireActivity(), R.id.nav_fragment).navigateUp()
        }
    }

    override fun showLoading() {
        mBinding.progressBar.isVisible = true
    }

    override fun hideLoading() {
        mBinding.progressBar.isVisible = false
    }

    fun onSelectedChanged() {
        mViewModel.doSelectedChanged()
    }

    fun onActionTxtClicked(groupInfo: GroupInfo?) {
        mViewModel?.doActionTxtClicked(groupInfo)
    }

    companion object {
        private const val TAG = "[UI][AddressBook]"
    }

}