package io.agora.avc.app.address

import android.app.Application
import androidx.lifecycle.MutableLiveData
import com.agora.valoran.Constants
import io.agora.avc.MyApplication
import io.agora.avc.R
import io.agora.avc.app.group.ContactGroup
import io.agora.avc.app.group.MeetupResponse
import io.agora.avc.base.AppViewModel
import io.agora.avc.biz.AppController
import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.AppEventBus
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.bo.AppConfig
import io.agora.avc.bo.LocalUser
import io.agora.avc.bo.NoticeCode
import io.agora.avc.bo.valoran.ARoomUser
import io.agora.avc.manager.notice.NoticeEvent
import io.agora.avc.manager.notice.NoticeManager
import io.agora.avc.repository.AddressBookRepository
import io.agora.avc.repository.AppConfigRepository
import io.agora.avc.repository.RoomRepository
import io.agora.frame.base.livedata.EventLiveData
import io.agora.logger.Logger
import java.util.concurrent.CopyOnWriteArrayList
import javax.inject.Inject

const val MY_GROUP_DEPT = Int.MIN_VALUE.toString()

class AddressBookViewModel @Inject constructor(
    application: Application,
    appController: AppController,
    private val roomRepository: RoomRepository,
    private val addressBookRepository: AddressBookRepository,
    private val appConfigRepository: AppConfigRepository,
    private val noticeManager: NoticeManager,
    private val appEventBus: AppEventBus,
) : AppViewModel(application, appController) {

    val membersLiveData = MutableLiveData<List<AddressBookNode>>()
    val appConfigLiveData = MutableLiveData<AppConfig?>()
    val membersItemChangedEvent = EventLiveData<Int>()
    val deptLiveData = MutableLiveData<GroupNode>()
    val selectedLiveData = MutableLiveData<List<MemberNode>>()
    val localUserLiveData = MutableLiveData<LocalUser>()
    val addGroupEvent = EventLiveData<Boolean>()
    private val dataTree = arrayListOf<AddressBookNode>()
    private var selectedData = arrayListOf<MemberNode>()
    private var joinedData = roomRepository.getRoomUserList()
    private var currentSeq = -1
    private var currentDept = "-1"
    private var operationType = OPERATION_INVITE
    private var dataType = DATA_ORG
    private var appConfig: AppConfig? = null
    private val emptySearcher = EmptyNode()
    private val emptyGroupMember = EmptyGroupNode()
    private val emptyMember = EmptyMemberNode()
    private val errorMember = ErrorNode()

    override fun onResume() {
        super.onResume()
        joinedData = roomRepository.getRoomUserList()
        queryLocalUser()
        queryAppConfig()
        selectedLiveData.postValue(selectedData)
        calibrateUserSelected(
            getLocalUser(),
            dataTree,
            selectedData,
            joinedData,
            operationType,
            membersLiveData
        )
        calibrateGroupInfo(dataTree, currentDept)
        deptLiveData.value?.let {
            notifyDeptInfoChanged(it)
        }
    }

    private fun notifyDeptInfoChanged(groupNode: GroupNode) {
        deptLiveData.postValue(groupNode)
        appEventBus.notifyObservers(
            MessageEvent(
                AppEvent.ADDRESS_BOOK_DEPT_INFO_CHANGED_EVENT.ordinal,
                groupNode
            )
        )
    }

    private fun queryAppConfig() {
        this.appConfig = appConfigRepository.getAppConfig()
        appConfigLiveData.postValue(appConfig)
    }

    private fun notifyUserSelectedChanged(list: ArrayList<MemberNode>) {
        selectedLiveData.postValue(list)
        appEventBus.notifyObservers(MessageEvent(AppEvent.ADDRESS_BOOK_SELECT_DATA_CHANGED_EVENT.ordinal))
    }

    private fun calibrateGroupInfo(
        members: ArrayList<AddressBookNode>,
        deptId: String
    ) {
        var isLeaf = members.isNotEmpty()
        var allSelected = members.isNotEmpty()
        members.forEach { node ->
            if (isLeaf && node !is MemberNode) {
                isLeaf = false
            }
            if (allSelected && !node.isChecked) {
                allSelected = false
            }
        }
        val groupInfo = GroupInfo(isLeaf, allSelected, deptId == MY_GROUP_DEPT)
        appEventBus.notifyObservers(
            MessageEvent(
                AppEvent.ADDRESS_BOOK_GROUP_STATUS_CHANGED_EVENT.ordinal,
                groupInfo
            )
        )
    }

    /**
     * calibrate member selected
     */
    private fun calibrateUserSelected(
        localUser: LocalUser?,
        members: ArrayList<AddressBookNode>,
        selectedData: ArrayList<MemberNode>,
        joinedData: CopyOnWriteArrayList<ARoomUser>,
        operationType: Int,
        liveData: MutableLiveData<List<AddressBookNode>>
    ) {
        members.forEach { node ->
            if (node is MemberNode) {
                //check isChecked
                if (node.isChecked && !selectedData.contains(node)) {
                    node.isChecked = false
                } else if (!node.isChecked && selectedData.contains(node)) {
                    node.isChecked = true
                }
                //check isEnable
                if (localUser != null && localUser.thirdPartyUid == node.userid) {
                    //local user always can not check, node can not click
                    node.isJoined = true
                    node.isChecked = true
                } else if (operationInvite(operationType) && isJoined(joinedData, node.userid)) {
                    // if ot is invite & user is joined, node can not click
                    node.isJoined = true
                    node.isChecked = true
                } else {
                    node.isJoined = false
                }
            }
        }
        liveData.postValue(members)
    }

    private fun isJoined(joinedData: CopyOnWriteArrayList<ARoomUser>, userid: String): Boolean {
        for (user in joinedData) {
            if (user.thirdPartyUid == userid) {
                return true
            }
        }
        return false
    }

    private fun queryLocalUser() {
        getLocalUser()?.let { user ->
            localUserLiveData.postValue(user)
        }
    }

    fun queryDept(deptName: String?, deptId: String, operationType: Int, dataType: Int) {
        Logger.i(TAG, "query deptId:$deptId, type:$operationType, dt:$dataType")
        this.currentSeq = appController.getOperationSeqId()
        this.currentDept = deptId
        this.dataType = dataType
        deptName?.let {
            notifyDeptInfoChanged(GroupNode(it, deptId))
        }
        setOperationType(operationType)
        if (isCustomGroup(dataType)) {
            queryMyGroup(deptId)
        } else {
            queryDeptInfo(deptId)
        }
    }

    private fun operationInvite(operationType: Int): Boolean = operationType == OPERATION_INVITE

    private fun isCustomGroup(dataType: Int): Boolean = dataType == DATA_CUSTOM

    private fun isMyGroupList(groupId: String): Boolean = groupId == MY_GROUP_DEPT

    /**
     * query dept data
     */
    private fun queryDeptInfo(deptId: String) {
        showLoading()
        val data = if (dataType == DATA_ORG) {
            addressBookRepository.getDeptInfo(currentSeq, deptId)
        } else {
            null
        }
        //use cached data
        if (data != null) {
            handleAddressBookResult(
                AddressBookResponse(
                    Constants.AddressBookOperationType.GET.value,
                    Constants.AddressBookResultType.SUCCESS.value,
                    currentSeq,
                    data
                )
            )
            return
        }
        appController.getAddressBook(currentSeq, deptId)
    }

    /**
     * query group data
     */
    private fun queryMyGroup(groupId: String) {
        showLoading()
        currentSeq = appController.getOperationSeqId()
        if (isMyGroupList(groupId)) {
            appController.getMeetupList(currentSeq, PAGE, PAGE_SIZE)
        } else {
            appController.getMeetupDetail(currentSeq, groupId)
        }
    }

    /**
     * deselected user
     */
    fun uncheckMember(memberNode: MemberNode) {
        deselectUser(memberNode)
        notifyUserSelectedChanged(selectedData)
        calibrateUserSelected(
            getLocalUser(),
            dataTree,
            selectedData,
            joinedData,
            operationType,
            membersLiveData
        )
        calibrateGroupInfo(dataTree, currentDept)
    }

    fun toggleMember(node: MemberNode, position: Int) {
        if (isOperationAdd() && !node.isChecked && isOverMemberMaxLimit(selectedData.size, 1)) {
            noticeManager.notice(
                NoticeEvent(
                    NoticeCode.CODE_GROUP_MEMBER_EXCEED,
                    obj = getMemberMaxLimit()
                )
            )
            membersItemChangedEvent.postValue(position)
            return
        }
        node.isChecked = !node.isChecked
        if (node.isChecked) {
            selectUser(node)
        } else {
            deselectUser(node)
        }
        notifyUserSelectedChanged(selectedData)
        calibrateGroupInfo(dataTree, currentDept)
        membersItemChangedEvent.postValue(position)
    }

    fun searchContent(content: String) {
        if (content.isEmpty()) {
            Logger.e(TAG, "search empty")
            currentSeq = appController.getOperationSeqId()
            hideLoading()
            dataTree.clear()
            membersLiveData.postValue(dataTree)
            return
        }
        Logger.i(TAG, "search content:$content")
        showLoading()
        currentSeq = appController.getOperationSeqId()
        appController.searchAddressBook(currentSeq, content)
    }

    override fun onEventReceived(arg: MessageEvent) {
        when (arg.type) {
            AppEvent.ADDRESS_BOOK_EVENT.ordinal -> {
                val response = arg.obj
                if (response !is AddressBookResponse) {
                    return
                }
                handleAddressBookResult(response)
            }
            AppEvent.MEETUP_QUERY_LIST_RESULT_EVENT.ordinal -> {
                onGroupListResponse(arg.obj)
            }
            AppEvent.MEETUP_QUERY_DETAIL_RESULT_EVENT.ordinal -> {
                onGroupDetailResponse(arg.obj)
            }
            AppEvent.MEETUP_CREATE_RESULT_EVENT.ordinal,
            AppEvent.MEETUP_UPDATE_RESULT_EVENT.ordinal,
            AppEvent.MEETUP_DELETE_RESULT_EVENT.ordinal -> {
                handleMeetupOperationResult(arg.obj)
            }
        }
    }

    private fun handleMeetupOperationResult(data: Any?) {
        if (data !is MeetupResponse) {
            return
        }
        if (data.resultType != Constants.MeetupResultType.SUCCESS.value) {
            if (data.operationType != Constants.MeetupOperationType.DELETE.value) {
                //only handle delete error
                return
            }
            when (data.resultType) {
                Constants.MeetupResultType.FAILURE_MAX.value -> {
                    noticeManager.notice(
                        NoticeEvent(
                            NoticeCode.CODE_GROUP_EXCEED,
                            obj = getGroupMaxLimit()
                        )
                    )
                }
                Constants.MeetupResultType.FAILURE_MAX_MEMBERS.value -> {
                    noticeManager.notice(
                        NoticeEvent(
                            NoticeCode.CODE_GROUP_MEMBER_EXCEED,
                            obj = getMemberMaxLimit()
                        )
                    )
                }
                Constants.MeetupResultType.FAILURE_DECRYPT_ERROR.value -> {
                    noticeManager.notice(NoticeEvent(NoticeCode.CODE_INVITE_ALERT))
                }
                Constants.MeetupResultType.FAILURE_UNKNOWN.value -> {
                    noticeManager.notice(NoticeEvent(NoticeCode.CODE_INVITE_ALERT))
                }
                else -> {
                    noticeManager.notice(NoticeEvent(NoticeCode.CODE_INVITE_ALERT))

                }
            }
            return
        }
        if (isMyGroupList(currentDept)) {
            queryMyGroup(currentDept)
        }
    }

    private fun onGroupListResponse(data: Any?) {
        if (data !is MeetupResponse) {
            return
        }
        if (data.seq != currentSeq) {
            return
        }
        hideLoading()
        val groupNode = GroupNode(
            getApplication<MyApplication>().getString(R.string.avc_group_name),
            MY_GROUP_DEPT
        )
        if (data.resultType != Constants.MeetupResultType.SUCCESS.value) {
            //add error member
            dataTree.clear()
            dataTree.add(errorMember)
            membersLiveData.postValue(dataTree)
            return
        }
        val list = data.data as? List<ContactGroup>
        dataTree.clear()
        if (list == null || list.isEmpty()) {
            //add empty member
            dataTree.add(emptyGroupMember)
            membersLiveData.postValue(dataTree)
            return
        }
        val mapList = arrayListOf<MyGroupNode>()
        list?.forEach { item ->
            mapList.add(MyGroupNode(item.name, item.id))
        }
        dataTree.addAll(mapList)
        calibrateUserSelected(
            getLocalUser(),
            dataTree,
            selectedData,
            joinedData,
            operationType,
            membersLiveData
        )
        calibrateGroupInfo(dataTree, currentDept)
        notifyDeptInfoChanged(groupNode)
    }

    private fun onGroupDetailResponse(data: Any?) {
        if (data !is MeetupResponse) {
            return
        }
        if (data.seq != currentSeq) {
            return
        }
        hideLoading()
        if (data.resultType != Constants.MeetupResultType.SUCCESS.value) {
            dataTree.clear()
            dataTree.add(errorMember)
            membersLiveData.postValue(dataTree)
            return
        }
        val group = data.data as? ContactGroup
        dataTree.clear()
        if (group == null) {
            membersLiveData.postValue(dataTree)
            return
        }
        val mapList = arrayListOf<MemberNode>()
        group?.members?.forEach {
            mapList.add(
                MemberNode(
                    it.userid,
                    it.name,
                    it.alias,
                    it.mainDepartment,
                    it.avatar,
                    department = it.department
                )
            )
        }
        dataTree.addAll(mapList)
        calibrateUserSelected(
            getLocalUser(),
            dataTree,
            selectedData,
            joinedData,
            operationType,
            membersLiveData
        )
        calibrateGroupInfo(dataTree, currentDept)
        notifyDeptInfoChanged(GroupNode(group.name, group.id))
    }

    private fun handleAddressBookResult(response: AddressBookResponse) {
        if (response.seq == currentSeq) {
            hideLoading()
            if (response.resultType != Constants.AddressBookResultType.SUCCESS.value) {
                //add error member
                dataTree.clear()
                dataTree.add(errorMember)
                membersLiveData.postValue(dataTree)
                return
            }
            val operationType = response.operationType
            if (Constants.AddressBookOperationType.GET.value == operationType) {
                onGetResponse(response.data as? GroupNode)
            } else if (Constants.AddressBookOperationType.SEARCH.value == operationType) {
                onSearchResponse(response.data as? ArrayList<AddressBookNode>)
            }
        }
    }

    private fun onGetResponse(data: GroupNode?) {
        dataTree.clear()
        if (data == null) {
            //add error member
            dataTree.add(errorMember)
            membersLiveData.postValue(dataTree)
            return
        }
        if (dataType == DATA_ORG) {
            addressBookRepository.saveDeptInfo(currentDept, data)
        }
        val list = data.children as? ArrayList<AddressBookNode>
        if (list == null || list.isEmpty()) {
            dataTree.add(emptyMember)
            membersLiveData.postValue(dataTree)
            return
        }
        dataTree.addAll(list)
        if (data.deptId == ROOT_DEPT_NO && operationInvite(operationType)) {
            val groupName = getApplication<MyApplication>().getString(R.string.avc_group_name)
            dataTree.add(MyGroupNode(groupName, MY_GROUP_DEPT))
        }
        calibrateUserSelected(
            getLocalUser(),
            dataTree,
            selectedData,
            joinedData,
            operationType,
            membersLiveData
        )
        calibrateGroupInfo(dataTree, currentDept)
        notifyDeptInfoChanged(data)
    }

    private fun onSearchResponse(data: ArrayList<AddressBookNode>?) {
        dataTree.clear()
        if (data == null || data.isEmpty()) {
            //add empty member
            dataTree.add(emptySearcher)
            membersLiveData.postValue(dataTree)
            return
        }
        dataTree.addAll(data)
        calibrateUserSelected(
            getLocalUser(),
            dataTree,
            selectedData,
            joinedData,
            operationType,
            membersLiveData
        )
    }

    override fun getUIEvents(): Array<AppEvent>? {
        return arrayOf(
            AppEvent.ADDRESS_BOOK_EVENT,
            AppEvent.MEETUP_QUERY_LIST_RESULT_EVENT,
            AppEvent.MEETUP_QUERY_DETAIL_RESULT_EVENT,
            AppEvent.MEETUP_CREATE_RESULT_EVENT,
            AppEvent.MEETUP_UPDATE_RESULT_EVENT,
            AppEvent.MEETUP_DELETE_RESULT_EVENT,
        )
    }

    fun setOperationType(operationType: Int) {
        this.operationType = operationType
        selectedData = if (isOperationAdd()) {
            addressBookRepository.queryAdded()
        } else {
            addressBookRepository.queryInvitee()
        }
    }

    private fun isOperationAdd() = operationType == OPERATION_ADD

    private fun selectAll(selected: Boolean) {
        if (selected) {
            if (isOperationAdd() && isOverMemberMaxLimit(selectedData, dataTree)) {
                noticeManager.notice(
                    NoticeEvent(
                        NoticeCode.CODE_GROUP_MEMBER_EXCEED,
                        obj = getMemberMaxLimit()
                    )
                )
                return
            }
            for (node in dataTree) {
                if (node is MemberNode && !node.isJoined) {
                    node.isChecked = true
                }
            }
            membersLiveData.postValue(dataTree)
            for (node in dataTree) {
                selectUser(node)
            }
            notifyUserSelectedChanged(selectedData)
        } else {
            for (node in dataTree) {
                if (node is MemberNode && !node.isJoined) {
                    node.isChecked = false
                }
            }
            membersLiveData.postValue(dataTree)
            for (node in dataTree) {
                deselectUser(node)
            }
            notifyUserSelectedChanged(selectedData)
        }
        calibrateGroupInfo(dataTree, currentDept)
    }

    private fun isOverMemberMaxLimit(
        selectedData: ArrayList<MemberNode>,
        dataTree: ArrayList<AddressBookNode>
    ): Boolean {
        var count = 0
        for (node in dataTree) {
            if (!node.isChecked) {
                count++
            }
        }
        return isOverMemberMaxLimit(selectedData.size, count)
    }

    private fun isOverMemberMaxLimit(selectedSize: Int, unChecked: Int): Boolean {
        val maxMeetupMembersNum = getMemberMaxLimit()
        if (selectedSize + unChecked > maxMeetupMembersNum) {
            return true
        }
        return false
    }

    private fun getMemberMaxLimit(): Int = appConfig?.app?.maxMeetupMembersNum ?: 0

    private fun getGroupMaxLimit(): Int = appConfig?.app?.maxMeetupsNum ?: 0

    private fun isOverGroupMaxLimit(groupSize: Int, addCount: Int): Boolean {
        val maxMeetupsNum = getGroupMaxLimit()
        if (groupSize + addCount > maxMeetupsNum) {
            return true
        }
        return false
    }

    private fun selectUser(node: AddressBookNode) {
        if (node is MemberNode && !node.isJoined && !selectedData.contains(node)) {
            selectedData.add(node)
        }
    }

    private fun deselectUser(node: AddressBookNode) {
        if (node is MemberNode && !node.isJoined) {
            val position = selectedData.indexOf(node)
            if (position in 0 until selectedData.size) {
                selectedData.removeAt(position)
            }
        }
    }

    fun doSelectedChanged() {
        calibrateUserSelected(
            getLocalUser(),
            dataTree,
            selectedData,
            joinedData,
            operationType,
            membersLiveData
        )
        calibrateGroupInfo(dataTree, currentDept)
    }

    fun doActionTxtClicked(groupInfo: GroupInfo?) {
        val info = groupInfo ?: return
        if (info.isGroupList) {
            if (isOverGroupMaxLimit(dataTree.size, 1)) {
                noticeManager.notice(
                    NoticeEvent(
                        NoticeCode.CODE_GROUP_EXCEED,
                        obj = getGroupMaxLimit()
                    )
                )
            } else {
                addGroupEvent.postValue(true)
            }
        } else if (info.isLeaf && info.isAllSelected) {
            selectAll(false)
        } else if (info.isLeaf && !info.isAllSelected) {
            selectAll(true)
        }
    }

    companion object {
        private const val TAG = "[VM][AddressBook]"
        private const val PAGE = 1
        private const val PAGE_SIZE = 30
    }
}