package io.agora.avc.app.address

import android.app.Application
import androidx.lifecycle.MutableLiveData
import io.agora.avc.base.AppViewModel
import io.agora.avc.biz.AppController
import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.bo.LocalUser
import io.agora.avc.manager.notice.NoticeManager
import io.agora.avc.repository.AddressBookRepository
import io.agora.logger.Logger
import javax.inject.Inject

class MemberSelectedViewModel @Inject constructor(
    application: Application,
    appController: AppController,
    private val addressBookRepository: AddressBookRepository,
    private val noticeManager: NoticeManager,
) : AppViewModel(application, appController) {

    val selectedLiveData = MutableLiveData<List<MemberNode>>()
    val localUserLiveData = MutableLiveData<LocalUser>()
    private var selectedData = arrayListOf<MemberNode>()
    private var currentSeq = -1
    private var operationType = OPERATION_INVITE
    private var dataType = DATA_ORG

    override fun onResume() {
        super.onResume()
        queryLocalUser()
        notifyUserSelectedChanged(selectedData)
    }

    private fun notifyUserSelectedChanged(list: ArrayList<MemberNode>) {
        selectedLiveData.postValue(list)
    }

    private fun queryLocalUser() {
        getLocalUser()?.let { user ->
            localUserLiveData.postValue(user)
        }
    }

    fun initParams(operationType: Int, dataType: Int) {
        Logger.i(TAG, "init params, type:$operationType, dt:$dataType")
        this.currentSeq = appController.getOperationSeqId()
        this.dataType = dataType
        setOperationType(operationType)
    }

    /**
     * deselected user
     */
    fun uncheckMember(memberNode: MemberNode) {
        deselectUser(memberNode)
        notifyUserSelectedChanged(selectedData)
    }

    override fun onEventReceived(arg: MessageEvent) {
        when (arg.type) {
            AppEvent.ADDRESS_BOOK_SELECT_DATA_CHANGED_EVENT.ordinal -> {
                notifyUserSelectedChanged(selectedData)
            }
        }
    }

    override fun getUIEvents(): Array<AppEvent>? {
        return arrayOf(
            AppEvent.ADDRESS_BOOK_SELECT_DATA_CHANGED_EVENT,
        )
    }

    private fun setOperationType(operationType: Int) {
        this.operationType = operationType
        selectedData = if (isOperationAdd()) {
            addressBookRepository.queryAdded()
        } else {
            addressBookRepository.queryInvitee()
        }
    }

    private fun isOperationAdd() = operationType == OPERATION_ADD

    private fun deselectUser(node: AddressBookNode) {
        if (node is MemberNode) {
            val position = selectedData.indexOf(node)
            if (position in 0 until selectedData.size) {
                selectedData.removeAt(position)
            }
        }
    }

    companion object {
        private const val TAG = "[VM][MemberSelected]"
    }
}