package io.agora.avc.app.attendees

import android.text.SpannableString
import android.text.SpannableStringBuilder
import android.text.Spanned
import android.text.TextUtils
import android.text.style.ForegroundColorSpan
import android.util.TypedValue
import android.view.View
import androidx.core.view.isVisible
import androidx.recyclerview.widget.RecyclerView
import io.agora.avc.R
import io.agora.avc.bo.LocalUser
import io.agora.avc.bo.valoran.ARoomUser
import io.agora.avc.databinding.AvcItemAttendeeBinding
import io.agora.avc.extensions.isMySelf
import io.agora.avc.listener.AntiShakeClickListener
import io.agora.avc.utils.StringUtils
import io.agora.avc.utils.ThemeUtils

class AttendeesViewHolder constructor(val binding: AvcItemAttendeeBinding) :
    RecyclerView.ViewHolder(binding.root), IAttendeesViewHolder {

    private var attendeeClickListener: AttendeesAdapter.IAttendeeClickListener? = null
    private var colorPrimary: Int = 0

    init {
        val typedValue = TypedValue()
        binding.root.context.theme.resolveAttribute(R.attr.colorPrimary, typedValue, true)
        colorPrimary = typedValue.data
    }

    override var user: ARoomUser? = null

    override fun bindUser(user: ARoomUser) {
        bindName(user)
        bindHost(user)
        bindFeature(user)
        bindThirdPartyStatus(user)
        bindThirdPartyDepartment(user)
        bindInviter(user)
        bindAudioStatus(user)
        bindVideoStatus(user)
        bindSpeaking(user)
        bindShareStatus(user)
        setListener()
        bindAssistantStatus(user)
    }

    override fun bindName(user: ARoomUser) {
        if (user.isThirdPartyLoggedIn || user.isAssistant) {
            if (user.isAssistant) {
                binding.tvUserName.text = StringUtils.getString(R.string.avc_as_name)
                binding.tvUserName.setTextColor(
                    ThemeUtils.getAttrValue(
                        binding.root.context,
                        R.attr.colorPrimary
                    )
                )
                if (user.isThirdPartyLoggedIn) {
                    setAssistantUserName(
                        user.thirdPartyName,
                        user.thirdPartyAlias,
                        user.thirdPartyNameStartPos,
                        user.thirdPartyNameEndPos,
                        user.thirdPartyAliasStartPos,
                        user.thirdPartyAliasEndPos
                    )
                } else {
                    setAssistantUserName(
                        user.name,
                        null,
                        user.nameStartPos,
                        user.nameEndPos,
                        0,
                        0
                    )
                }
            } else {
                binding.tvUserName.setTextColor(
                    ThemeUtils.getAttrValue(
                        binding.root.context,
                        android.R.attr.textColorPrimary
                    )
                )
                setUserName(
                    user.thirdPartyName,
                    user.thirdPartyNameStartPos,
                    user.thirdPartyNameEndPos
                )
                setUserAlias(
                    user.thirdPartyAlias,
                    user.thirdPartyAliasStartPos,
                    user.thirdPartyAliasEndPos
                )
            }
            binding.tvAlias.visibility = View.VISIBLE
        } else {
            setUserName(user.name, user.nameStartPos, user.nameEndPos)
            binding.tvAlias.text = ""
            binding.tvAlias.visibility = View.GONE
        }
    }

    private fun setUserAlias(alias: String?, startPos: Int, endPos: Int) {
        if (startPos != endPos) {
            val spannableString = SpannableString(alias)
            spannableString.setSpan(
                ForegroundColorSpan(colorPrimary),
                startPos,
                endPos,
                Spanned.SPAN_INCLUSIVE_INCLUSIVE
            )
            binding.tvAlias.text = spannableString
        } else {
            binding.tvAlias.text = alias
        }

    }

    private fun setUserName(name: String?, startPos: Int, endPos: Int) {
        if (TextUtils.isEmpty(name)) {
            binding.tvUserName.text = StringUtils.getString(R.string.avc_participants_unknown)
        } else if (startPos != endPos) {
            val spannableString = SpannableString(name)
            spannableString.setSpan(
                ForegroundColorSpan(colorPrimary),
                startPos,
                endPos,
                Spanned.SPAN_INCLUSIVE_INCLUSIVE
            )
            binding.tvUserName.text = spannableString
        } else {
            binding.tvUserName.text = name
        }
    }

    private fun setAssistantUserName(
        name: String?,
        alias: String?,
        nameStartPos: Int,
        nameEndPos: Int,
        aliasStartPos: Int,
        aliasEndPos: Int
    ) {
        if (name?.isEmpty() == true) {
            binding.tvAlias.text = StringUtils.getString(R.string.avc_participants_unknown)
            return
        }
        SpannableStringBuilder().let { _builder ->
            _builder.append(name)
            if (alias?.isEmpty() == false) {
                _builder.append("(")
                _builder.append(alias)
                _builder.append(")")
            }
            if (nameStartPos != nameEndPos) {
                _builder.setSpan(
                    ForegroundColorSpan(colorPrimary),
                    nameStartPos,
                    nameEndPos,
                    Spanned.SPAN_INCLUSIVE_INCLUSIVE
                )
            }
            if (aliasStartPos != aliasEndPos) {
                _builder.setSpan(
                    ForegroundColorSpan(colorPrimary),
                    aliasStartPos + _builder.indexOf(alias!!),
                    aliasEndPos + _builder.indexOf(alias),
                    Spanned.SPAN_INCLUSIVE_INCLUSIVE
                )
            }
            binding.tvAlias.text = _builder
        }
    }

    override fun bindHost(user: ARoomUser) {
        binding.ivHost.isVisible = user.isHost
    }

    override fun bindFeature(user: ARoomUser) {
        refreshTagLayoutVisible(user)
//        binding.tvAudioDump.isVisible = user.isAudioDumping()
//        binding.tvCloudRecording.isVisible = user.isCloudRecording
    }

    override fun bindThirdPartyStatus(user: ARoomUser) {
        refreshTagLayoutVisible(user)
//        binding.tvNonLogin.isVisible = !user.isThirdPartyLoggedIn
    }

    override fun bindThirdPartyDepartment(user: ARoomUser) {
        if (user.isThirdPartyLoggedIn) {
            binding.tvDepartment.text = binding.root.context.getString(
                R.string.avc_participants_department,
                user.thirdPartyDepartment
            )
        }
    }

    override fun bindInviter(user: ARoomUser) {
        if (!user.isThirdPartyLoggedIn) {
            binding.tvDepartment.text = if (user.inviteBy?.isNotEmpty() == true) {
                binding.root.context.getString(R.string.avc_participants_inviter, user.inviteBy)
            } else {
                val name = binding.root.context.getString(R.string.avc_participants_unknown)
                binding.root.context.getString(R.string.avc_participants_inviter, name)
            }
        }
    }

    override fun bindAudioStatus(user: ARoomUser) {
        if (user.isAudioPending) {
            binding.btnAudio.loading = true
        } else {
            binding.btnAudio.loading = false
            binding.btnAudio.mute = !user.audioState
            binding.btnAudio.isMySelf = user.isMySelf()
        }
    }

    override fun bindVideoStatus(user: ARoomUser) {
        if (user.isVideoPending) {
            binding.btnVideo.loading = true
        } else {
            binding.btnVideo.loading = false
            binding.btnVideo.opened = user.videoState
        }
    }

    override fun bindSpeaking(user: ARoomUser) {
        if (user.isAudioPending) {
            binding.btnAudio.loading = true
        } else if (user !is LocalUser) {
            binding.btnAudio.loading = false
            binding.btnAudio.speaking = user.isSpeaking
        } else {
            binding.btnAudio.loading = false
            binding.btnAudio.volume = user.volume
        }
    }

    override fun bindVolume(user: ARoomUser) {
        if (user.isMySelf()) {
            binding.btnAudio.volume = user.volume
        }
    }

    override fun bindShareStatus(user: ARoomUser) {
        binding.tvShare.isVisible = user.hasShareStream()
    }

    override fun bindAssistantStatus(user: ARoomUser) {
        binding.btnAudio.isVisible = user.showMedia
        binding.btnVideo.isVisible = user.showMedia
        if (!user.isAssistant) {
//            binding.tvAssistantLeaving.isVisible = false
            binding.ivHost.isVisible = user.isHost
            return
        }
//        binding.tvAssistantLeaving.isVisible = !user.online
        binding.ivHost.isVisible = user.isHost
    }

    private fun refreshTagLayoutVisible(user: ARoomUser) {
        binding.fblTag.isVisible =
            user.hasShareStream() || user.isAudioDumping() || !user.isThirdPartyLoggedIn || user.isCloudRecording
    }

    private fun setListener() {
        object : AntiShakeClickListener() {
            override fun onAntiShakeClicked(view: View?) {
                user?.let {
                    attendeeClickListener?.onAttendeeClicked(it)
                }
            }
        }.apply {
            binding.ivHost.setOnClickListener(this)
            binding.root.setOnClickListener(this)
        }
    }

    fun setIAttendeeClickListener(listener: AttendeesAdapter.IAttendeeClickListener?) {
        this.attendeeClickListener = listener
    }
}