package io.agora.avc.app.audio

import android.Manifest
import android.content.res.Configuration
import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import androidx.appcompat.app.AlertDialog
import androidx.constraintlayout.widget.ConstraintSet
import androidx.core.graphics.Insets
import androidx.core.view.updatePaddingRelative
import androidx.lifecycle.Observer
import com.google.android.material.dialog.MaterialAlertDialogBuilder
import io.agora.avc.R
import io.agora.avc.app.master.MainViewModel
import io.agora.avc.bo.LocalUser
import io.agora.avc.dao.PermissionDao
import io.agora.avc.databinding.AvcFragmentAudioBinding
import io.agora.avc.utils.ClipboardUtils
import io.agora.avc.utils.PermissionConstants
import io.agora.avc.utils.PermissionUtils
import io.agora.avc.widget.MediaPopupWindow
import io.agora.avc.widget.danceview.NougatBoot
import io.agora.frame.base.HolderFragment
import io.agora.logger.LogConverter
import io.agora.logger.Logger

class AudioFragment : HolderFragment<AudioViewModel, AvcFragmentAudioBinding>() {

    private var localUser: LocalUser? = null
    private var unifiedDialog: AlertDialog? = null

    var openAudioConfirm = false

    var isAudioVisible = false
        set(value) {
            field = value
            Logger.i(TAG, "audio fragment, visible:${LogConverter.visible(value)}")
            if (value) {
                mainViewModel.enterPureAudioMode()
                mViewModel?.startCountTime()
                mViewModel?.onResume()
                localUser?.let {
                    setupLocalUser(it)
                }
            } else {
                mViewModel?.stopCountTime()
                mainViewModel.exitPureAudioMode()
                setupLocalAudioStatus(false)
            }
        }

    private val nougatBoot: NougatBoot by lazy {
        NougatBoot()
    }

    private val mediaPopupWindow by lazy {
        MediaPopupWindow(requireActivity())
    }

    private val mainViewModel: MainViewModel by lazy {
        getViewModel(requireActivity().viewModelStore, MainViewModel::class.java)
    }

    var onBackPressListener: (() -> Unit)? = null

    override fun getLayoutId(): Int {
        return R.layout.avc_fragment_audio
    }

    override fun allocObserver() {
        mViewModel?.meetingDurationEvent?.observe(this, Observer {
            mBinding.tvDuration.text = it
        })

        mViewModel?.localUserStatusChanged?.observe(this, Observer { user ->
            this.localUser = user
            setupLocalUser(user)
        })

        mViewModel?.bizConnectChangedEvent?.observe(this, {
            showRoomRtmState(it)
        })

        mViewModel?.roomInfoChangedEvent?.observe(this) {
            it?.let {
                mBinding.tvRoomName.text = it.name
            }
        }
    }

    private fun setupLocalUser(user: LocalUser) {
        if (isAudioVisible) {
            setupLocalAudioStatus(user.audioState)
            setupLocalSignal(user)
        }
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
    }

    override fun initialize(savedInstanceState: Bundle?) {

        mBinding.icHangup.setOnClickListener {
            Logger.i(TAG, "user hang up the meeting")
            if (localUser?.isCloudRecording == true) {
                showKeepRecordingConfirmDialog()
            } else {
                mViewModel?.quitRoom()
            }
        }
        mBinding.icSpeaker.apply {
            indeterminateDrawable = nougatBoot
        }

        mBinding.btBack.setOnClickListener {
            onBackPressed()
        }

        mBinding.flSpeaker.setOnClickListener {
            onLocalAudioClicked()
        }

        localUser?.let {
            setupLocalUser(it)
        }
    }

    private fun onLocalAudioClicked() {
        Logger.i(TAG, "Trigger local audio click")
        if (openAudioConfirm && !audioOpened()) {
            showOpenAudioConfirmDialog {
                checkPermissionOnLocalAudioClicked()
            }
        } else {
            checkPermissionOnLocalAudioClicked()
        }
    }

    private fun checkPermissionOnLocalAudioClicked() {
        Logger.i(TAG, "Trigger speaker click")
        PermissionUtils
            .permission(PermissionConstants.MICROPHONE)
            .callback(object : PermissionUtils.FullCallback {
                override fun onGranted(permissionsGranted: MutableList<String>) {
                    if (PermissionDao.isAllGranted(permissionsGranted, getRequestPermission())) {
                        mViewModel?.changeLocalAudioStatus(audioOpened())
                    }
                }

                private fun getRequestPermission(): Array<String> {
                    return arrayOf(Manifest.permission.RECORD_AUDIO)
                }

                override fun onDenied(
                    permissionsDeniedForever: MutableList<String>,
                    permissionsDenied: MutableList<String>
                ) {
                    mainViewModel?.onPermissionDenied(permissionsDeniedForever, permissionsDenied)
                }
            })
            .request()
    }

    override fun onStart() {
        super.onStart()
        mViewModel?.startCountTime()
    }

    override fun onPause() {
        super.onPause()
        unifiedDialog?.dismiss()
    }

    override fun onStop() {
        super.onStop()
        mViewModel?.stopCountTime()
    }

    override fun onBackPressed() {
        Logger.i(TAG, "user clicks the back button to exit pure audio mode")
        onBackPressListener?.invoke()
    }

    private fun setupLocalAudioStatus(open: Boolean) {
        if (mBinding?.btnAudio?.opened != open) {
            Logger.i(TAG, "set up local audio open:$open")
            mBinding?.btnAudio?.opened = open
        }
        if (open) {
            nougatBoot.startDance()
        } else {
            nougatBoot.stopDance()
        }
    }

    private fun setupLocalSignal(user: LocalUser) {
        mBinding.ivSignal.setQuality(user.quality)
    }

    private fun showRoomRtmState(connected: Boolean) {
        mBinding.ivSignal.bizConnected = connected
    }

    override fun onConfigurationChanged(newConfig: Configuration) {
        super.onConfigurationChanged(newConfig)

        ConstraintSet().apply {
            clone(AvcFragmentAudioBinding.inflate(LayoutInflater.from(context)).container)
            applyTo(mBinding.container)
        }
    }

    fun dispatchApplyWindowInsets(insets: Insets) {
        mBinding.container.updatePaddingRelative(
            start = insets.left,
            end = insets.right,
            top = insets.top,
            bottom = insets.bottom,
        )
    }

    override fun onDestroyView() {
        super.onDestroyView()
        mediaPopupWindow.dismiss()
    }

    private fun showKeepRecordingConfirmDialog() {
        ClipboardUtils.copyText(io.agora.avc.net.constant.Constants.getAtlasRecordingUrl())
        MaterialAlertDialogBuilder(requireContext(), R.style.avc_CustomMaterialAlertDialog)
            .setTitle(R.string.avc_notice_keep_recording_title)
            .setCancelable(false)
            .setMessage(getString(R.string.avc_notice_keep_recording_message))
            .setNegativeButton(R.string.avc_notice_keep_recording_cancel) { _, _ ->
            }
            .setPositiveButton(R.string.avc_notice_keep_recording_confirm) { _, _ ->
                mViewModel?.quitRoom()
            }
            .show()
    }

    private fun showOpenAudioConfirmDialog(action: () -> Unit) {
        unifiedDialog?.dismiss()
        activity?.let {
            unifiedDialog =
                MaterialAlertDialogBuilder(it, R.style.avc_CustomMaterialAlertDialog)
                    .setCancelable(false)
                    .setMessage(it.getString(R.string.avc_microphone_multiplayer_request))
                    .setNegativeButton(R.string.avc_no_button) { _, _ ->
                    }
                    .setPositiveButton(R.string.avc_yes_button) { _, _ ->
                        action.invoke()
                    }
                    .show()
        }
    }

    private fun audioOpened() = localUser?.audioState ?: false

    companion object {
        private const val TAG = "[UI][Audio]"

    }

}
