package io.agora.avc.app.bugReport

import android.content.res.Configuration
import android.os.Bundle
import android.text.Editable
import android.text.TextWatcher
import android.view.View
import android.view.ViewGroup
import android.widget.FrameLayout
import androidx.appcompat.content.res.AppCompatResources
import androidx.core.view.children
import androidx.core.view.forEachIndexed
import androidx.core.view.updateLayoutParams
import androidx.lifecycle.Observer
import androidx.navigation.fragment.NavHostFragment
import androidx.recyclerview.widget.DividerItemDecoration
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import com.google.android.material.bottomsheet.BottomSheetBehavior
import com.google.android.material.chip.Chip
import com.google.android.material.dialog.MaterialAlertDialogBuilder
import io.agora.avc.R
import io.agora.avc.app.master.MainViewModel
import io.agora.avc.bo.*
import io.agora.avc.bo.valoran.ARoomUser
import io.agora.avc.databinding.AvcFragmentBugReportBinding
import io.agora.avc.extensions.getConferenceNicknameMaybeAssistant
import io.agora.avc.utils.*
import io.agora.avc.widget.UploadingStatus
import io.agora.frame.base.BaseSheetDialogFragment
import io.agora.logger.Logger
import java.util.*

class BugReportFragment : BaseSheetDialogFragment<BugReportViewModel, AvcFragmentBugReportBinding>() {

    private val mainViewModel: MainViewModel? by lazy {
        getViewModel(requireActivity().viewModelStore, MainViewModel::class.java)
    }

    private val pageAdapter = BugReportAdapter()

    private var oppositeAdapter: OppositeUserAdapter? = null

    private var oppositeUserStreamId: Int? = null

    private var issue: Issue? = null

    private var localUser: ARoomUser? = null

    private val audioIssueList by lazy {
        listOf(
            Pair(0, R.string.avc_dump_label_perceivable_delay),
            Pair(3, R.string.avc_dump_label_intermittent_voice),
            Pair(5, R.string.avc_dump_label_no_sound),
            Pair(2, R.string.avc_dump_label_noticeable_noise),
            Pair(4, R.string.avc_dump_label_low_sound_volume),
            Pair(DUMP_LABEL_NOTICEABLE_ECHO_TAG, R.string.avc_dump_label_noticeable_echo)
        )
    }

    private val videoIssueList by lazy {
        listOf(
            Pair(8, R.string.avc_dump_label_high_latency),
            Pair(7, R.string.avc_dump_label_intermittent_video),
            Pair(11, R.string.avc_dump_label_blurriness),
            Pair(12, R.string.avc_dump_label_obvious_noise_in_the_image),
            Pair(9, R.string.avc_dump_label_corruption_mosaic),
            Pair(6, R.string.avc_dump_label_frame_freeze),
            Pair(13, R.string.avc_dump_label_too_bright_dark_image),
            Pair(10, R.string.avc_dump_label_sync)
        )
    }

    private val linearLayoutManager: LinearLayoutManager by lazy {
        LinearLayoutManager(context, RecyclerView.HORIZONTAL, false)
    }

    private val dividerItemDecoration by lazy {
        DividerItemDecoration(context, DividerItemDecoration.HORIZONTAL).also { decoration ->
            AppCompatResources.getDrawable(requireContext(), R.drawable.avc_shape_bug_report_divider)
                ?.let {
                    decoration.setDrawable(it)
                }
        }
    }

    private val commentTextWatcher: TextWatcher = object : TextWatcher {
        override fun afterTextChanged(s: Editable?) {
            s?.apply {
                if (length > MAX_COMMENT_WORDS) {
                    mBinding.inputComment.error =
                        context?.getString(R.string.avc_bug_report_comment_words_overflow)
                } else {
                    mBinding.inputComment.error = null
                }
            }
            refreshSubmitButtonEnable()
        }

        override fun beforeTextChanged(s: CharSequence?, start: Int, count: Int, after: Int) {

        }

        override fun onTextChanged(s: CharSequence?, start: Int, before: Int, count: Int) {
        }
    }

    override fun getLayoutId(): Int {
        return R.layout.avc_fragment_bug_report
    }

    override fun allocObserver() {
        mViewModel?.reportResultEvent?.observe(this, Observer {
            if (it == UploadingStatus.COMPLETED) {
                NavHostFragment.findNavController(this).navigateUp()
            }
        })
        mViewModel?.attachmentLiveData?.observe(this, Observer {
            pageAdapter.setNewData(it)
            setAttachmentItemVisible(it != null && it.isNotEmpty())
            refreshSubmitButtonEnable()
        })
        mViewModel?.issueChangedLiveData?.observe(this, Observer {
            this.issue = it
            this.oppositeUserStreamId = it.oppositeStreamId
            renderUI()
        })
        mViewModel?.localUserChangedLiveData?.observe(this, Observer {
            this.localUser = it
            renderUI()
        })
        mainViewModel?.bugReportRetryEvent?.observe(this) {
            mViewModel.doSubmit(collectReportInfo())
        }
    }

    override fun onStart() {
        super.onStart()
        mDialog.setCanceledOnTouchOutside(true)
        mBehavior.isDraggable = true
        mDialog.setCancelable(true)
        onConfigurationChanged(resources.configuration)
    }

    override fun onConfigurationChanged(newConfig: Configuration) {
        super.onConfigurationChanged(newConfig)
        if (newConfig.orientation == Configuration.ORIENTATION_LANDSCAPE) {
            switch2Landscape()
        } else {
            switch2Portrait()
        }
    }

    private fun switch2Portrait() {
        mDialog?.window?.let {
            BarUtils.setStatusBarVisibility(it, true)
        }
        val layoutParams = mBinding.container.layoutParams as FrameLayout.LayoutParams
        layoutParams.height = (ScreenUtils.getAppScreenHeight() * 0.9f).toInt()
        mBinding.container.layoutParams = layoutParams
        mBehavior.isFitToContents = false
        mBehavior.skipCollapsed = false
        mBehavior.peekHeight = (ScreenUtils.getAppScreenHeight() * 0.6f).toInt()
        mBehavior.expandedOffset = (ScreenUtils.getScreenHeight() * 0.1f).toInt()
        mBehavior.halfExpandedRatio = 0.6f
        mBehavior.state = BottomSheetBehavior.STATE_HALF_EXPANDED
        mBinding.scrollView.post {
            if (mBinding == null) {
                return@post
            }
            mBinding.scrollView.updateLayoutParams<ViewGroup.LayoutParams> {
                height = (ScreenUtils.getAppScreenHeight() * 0.9f).toInt() - ConvertUtils.dp2px(57f)
            }
        }
    }

    private fun switch2Landscape() {
        mDialog?.window?.let {
            BarUtils.setStatusBarVisibility(it, false)
        }
        val layoutParams = mBinding.container.layoutParams as FrameLayout.LayoutParams
        layoutParams.height = ScreenUtils.getAppScreenHeight() - ConvertUtils.dp2px(54f)
        mBinding.container.layoutParams = layoutParams
        mBehavior.isFitToContents = true
        mBehavior.skipCollapsed = true
        mBehavior.state = BottomSheetBehavior.STATE_EXPANDED
        mBinding.scrollView.post {
            if (mBinding == null) {
                return@post
            }
            mBinding.scrollView.updateLayoutParams<ViewGroup.LayoutParams> {
                height = ScreenUtils.getAppScreenHeight() - ConvertUtils.dp2px(111f)
            }
        }
    }

    override fun initialize(savedInstanceState: Bundle?) {
        oppositeAdapter = OppositeUserAdapter(
            requireContext(),
            R.layout.avc_list_user_item,
            R.id.tvItem,
        )

        mBinding.audioChips.apply {
            removeAllViews()
            addItems(audioIssueList)
            forEachIndexed { _, view ->
                if (view.tag == DUMP_LABEL_NOTICEABLE_ECHO_TAG) {
                    view.setOnClickListener {
                        if ((view as Chip).isChecked) {
                            ToastUtils.showLong(getString(R.string.avc_echo_pop))
                        }
                    }
                }
            }
        }

        mBinding.videoChips.removeAllViews()
        mBinding.videoChips.addItems(videoIssueList)

        mBinding.oppositeUser.setAdapter(oppositeAdapter)

        mBinding.recyclerView.adapter = pageAdapter
        mBinding.recyclerView.addItemDecoration(dividerItemDecoration)
        mBinding.recyclerView.layoutManager = linearLayoutManager
        pageAdapter.setOnDelClickListener(object : BugReportAdapter.OnDelClickListener {
            override fun onDelClicked(attachment: Attachment) {
                showDelAttachmentDialog(attachment)
            }
        })
        //set audio chip checked change listener
        for (child in mBinding.audioChips.children) {
            if (child is Chip) {
                child.setOnCheckedChangeListener { _, _ ->
                    refreshSubmitButtonEnable()
                }
            }
        }
        //set video chip checked change listener
        for (child in mBinding.videoChips.children) {
            if (child is Chip) {
                child.setOnCheckedChangeListener { _, _ ->
                    refreshSubmitButtonEnable()
                }
            }
        }
        mBinding.btnCreate.setOnClickListener {
            if (!isPermitSubmit()) {
                Logger.w(
                    TAG,
                    "Not allowed to submit,comment length:${mBinding.etComment.text?.toString()} audioChips size:${mBinding.audioChips.getCheckedTags()} videoChips size:${mBinding.videoChips.getCheckedTags()} file count:${issue?.attachmentList?.size}"
                )
                return@setOnClickListener
            }
            if (mBinding.audioChips.hasChecked() &&
                (issue?.attachmentList == null || issue?.attachmentList?.isEmpty() == true)
            ) {
                showRecordingTipsDialog()
            } else {
                mViewModel?.doSubmit(collectReportInfo())
            }
        }
        mBinding.btnDeclined.setOnClickListener {
            onBackEvent()
        }
        mBinding.btnRecord.setOnClickListener {
            if (pageAdapter.data.size >= MAX_ATTACHMENT) {
                showAttachmentOverflowDialog()
            } else {
                toRecording()
            }
        }
        mBinding.oppositeUser.setOnItemClickListener { parent, view, position, id ->
            oppositeUserStreamId = oppositeAdapter?.getItem(position)?.streamId
        }
        mBinding.etComment.addTextChangedListener(commentTextWatcher)
    }

    private fun renderUI() {
        context?.let {
            mBinding.userName.text = localUser?.getConferenceNicknameMaybeAssistant()
        }
        mBinding.userId.text = "ID:${localUser?.streamId ?: ""}"
        renderChips(issue)
        mBinding.etComment.setText(issue?.description ?: "")
        renderOppositeUser()
        refreshSubmitButtonEnable()
    }

    private fun renderOppositeUser() {
        oppositeAdapter?.setData((issue?.userSnapshot ?: arrayListOf()) as ArrayList<ARoomUser>)
        if (oppositeUserStreamId != null && issue?.userSnapshot != null) {
            try {
                var user: ARoomUser? = null
                var position = -1
                issue?.userSnapshot?.forEachIndexed { index, aRoomUser ->
                    if (aRoomUser.streamId == oppositeUserStreamId) {
                        user = aRoomUser
                        position = index
                        return@forEachIndexed
                    }
                }
                if (user != null) {
                    mBinding.oppositeUser.setText(
                        user?.getConferenceNicknameMaybeAssistant() ?: "",
                        false
                    )
                }
            } catch (e: Exception) {
                Logger.e(TAG, "An error occurred when setting the opposite username", e)
            }
        }
    }

    private fun renderChips(issue: Issue?) {
        issue?.tags?.forEach { tag ->
            if (tag in 0..5) {
                mBinding.audioChips.setChecked(tag)
            } else {
                mBinding.videoChips.setChecked(tag)
            }
        }
    }

    private fun setAttachmentItemVisible(visible: Boolean) {
        if (visible) {
            mBinding.attachmentTitle.visibility = View.VISIBLE
            mBinding.recyclerView.visibility = View.VISIBLE
        } else {
            mBinding.attachmentTitle.visibility = View.GONE
            mBinding.recyclerView.visibility = View.GONE
        }
    }

    private fun toRecording() {
        mViewModel?.prepare2Recording(collectReportInfo())
        NavHostFragment.findNavController(this).navigateUp()
    }

    private fun collectReportInfo(): ReportInfo {
        return ReportInfo(
            mBinding.etComment.text.toString(),
            oppositeUserStreamId,
            mBinding.audioChips.getCheckedTags(),
            mBinding.videoChips.getCheckedTags()
        )
    }

    private fun onBackEvent() {
        if (mBinding.etComment.text.toString().isNotEmpty() ||
            (oppositeUserStreamId != null && oppositeUserStreamId!! > 0) ||
            mBinding.audioChips.hasChecked() ||
            mBinding.videoChips.hasChecked() ||
            pageAdapter.data.size > 0
        ) {
            MaterialAlertDialogBuilder(requireContext(), R.style.avc_CustomMaterialAlertDialog)
                .setTitle(R.string.avc_notice_leave_bug_report_title)
                .setCancelable(false)
                .setMessage(getString(R.string.avc_notice_leave_bug_report_message))
                .setNegativeButton(R.string.avc_notice_leave_bug_report_confirm) { _, _ ->
                    quitReporting()
                }
                .setPositiveButton(R.string.avc_notice_leave_bug_report_cancel) { _, _ ->

                }
                .show()
        } else {
            quitReporting()
        }
    }

    private fun quitReporting() {
        Logger.i(TAG, "The user gave up editing the bug report")
        mViewModel?.quitReporting()
        NavHostFragment.findNavController(this).navigateUp()
    }

    override fun onBackPressed() {
        onBackEvent()
    }

    private fun refreshSubmitButtonEnable() {
        mBinding.btnCreate.isEnabled = isPermitSubmit()
    }

    private fun isPermitSubmit(): Boolean {
        val commentText = mBinding.etComment.text ?: ""
        val attachmentList = issue?.attachmentList ?: arrayListOf()
        return commentText.length <= MAX_COMMENT_WORDS &&
                (attachmentList.size > 0 ||
                        commentText.trim().isNotEmpty() ||
                        mBinding.audioChips.hasChecked() ||
                        mBinding.videoChips.hasChecked())

    }

    private fun showDelAttachmentDialog(attachment: Attachment) {
        MaterialAlertDialogBuilder(requireContext(), R.style.avc_CustomMaterialAlertDialog)
            .setTitle(R.string.avc_notice_bug_report_del_title)
            .setCancelable(false)
            .setMessage(getString(R.string.avc_notice_bug_report_del_message))
            .setNegativeButton(R.string.avc_notice_bug_report_del_cancel) { _, _ ->
            }
            .setPositiveButton(R.string.avc_notice_bug_report_del_confirm) { _, _ ->
                mViewModel?.removeAttachment(attachment)
            }
            .show()
    }

    private fun showAttachmentOverflowDialog() {
        MaterialAlertDialogBuilder(requireContext(), R.style.avc_CustomMaterialAlertDialog)
            .setTitle(R.string.avc_notice_bug_report_file_overflow_title)
            .setCancelable(false)
            .setMessage(getString(R.string.avc_notice_bug_report_file_overflow_message))
            .setPositiveButton(R.string.avc_notice_bug_report_file_overflow_confirm) { _, _ ->

            }
            .show()
    }

    private fun showRecordingTipsDialog() {
        MaterialAlertDialogBuilder(requireContext(), R.style.avc_CustomMaterialAlertDialog)
            .setTitle(R.string.avc_notice_bug_report_recording_tips_title)
            .setCancelable(true)
            .setMessage(getString(R.string.avc_notice_bug_report_recording_tips_message))
            .setNegativeButton(R.string.avc_notice_bug_report_recording_tips_cancel) { _, _ ->
                mViewModel?.doSubmit(collectReportInfo())
            }
            .setPositiveButton(R.string.avc_notice_bug_report_recording_tips_confirm) { _, _ ->
                toRecording()
            }
            .show()
    }

    override fun onStop() {
        super.onStop()
        mViewModel?.stashIssue(collectReportInfo())
        hideKeyboard()
    }

    private fun hideKeyboard() {
        activity?.let {
            KeyboardUtils.hideSoftInput(it)
        }
    }

    override fun onDestroyView() {
        mBinding.etComment.removeTextChangedListener(commentTextWatcher)
        super.onDestroyView()
    }

    companion object {
        private const val TAG = "[UI][BugReport]"
        const val MAX_ATTACHMENT = 2
        const val MAX_COMMENT_WORDS = 140
        const val DUMP_LABEL_NOTICEABLE_ECHO_TAG = 1
    }

}
