package io.agora.avc.app.bugReport

import android.app.Application
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.viewModelScope
import io.agora.avc.MyApplication
import io.agora.avc.R
import io.agora.avc.base.AppViewModel
import io.agora.avc.biz.AppController
import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.AppEventBus
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.bo.Attachment
import io.agora.avc.bo.AudioDumpStatus
import io.agora.avc.bo.Issue
import io.agora.avc.bo.LocalUser
import io.agora.avc.bo.valoran.ARoomUser
import io.agora.avc.manager.audiodump.AudioDumper
import io.agora.avc.manager.bugreport.BugReporter
import io.agora.avc.manager.bugreport.DumpStatus
import io.agora.avc.repository.ProblemRepository
import io.agora.avc.repository.RoomRepository
import io.agora.avc.utils.AppUtils
import io.agora.avc.utils.DeviceUtils
import io.agora.avc.utils.SDKUtils
import io.agora.avc.utils.TimeUtils
import io.agora.frame.base.livedata.EventLiveData
import io.agora.logger.Logger
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.util.*
import javax.inject.Inject

class BugReportViewModel @Inject constructor(
    application: Application,
    appController: AppController,
    private val problemRepository: ProblemRepository,
    private val audioDumper: AudioDumper,
    private val bugReporter: BugReporter,
    private val roomRepository: RoomRepository,
    private val appEventBus: AppEventBus,
) : AppViewModel(application, appController) {
    val attachmentLiveData = MutableLiveData<List<Attachment>?>()
    val reportResultEvent = EventLiveData<Int>()
    val issueChangedLiveData = MutableLiveData<Issue>()
    val localUserChangedLiveData = MutableLiveData<LocalUser?>()

    override fun onCreate() {
        super.onCreate()
    }

    override fun onResume() {
        super.onResume()
        queryIssue()
        queryAttachmentList()
        queryLocalUser()
    }

    private fun queryIssue() {
        var issue = problemRepository.getIssue()
        if (issue == null) {
            val localUser = getLocalUser()
            val issueName = if (localUser?.isThirdPartyLoggedIn == true) {
                localUser.thirdPartyName ?: ""
            } else {
                localUser?.name ?: ""
            }
            val oppositeUsers = localUser?.let {
                val oppositeSnapshot = roomRepository.getOppositeSnapshot(it)
                val default = ARoomUser(
                    name = getApplication<MyApplication>().getString(R.string.avc_dump_peer_none),
                    streamId = -1
                )
                oppositeSnapshot.add(0, default)
                oppositeSnapshot
            }

            issue = Issue(
                issueName = issueName,
                issueStreamId = localUser?.streamId ?: 0,
                appVersion = AppUtils.getAppVersionName(),
                rid = roomRepository.getRoom()?.name ?: "",
                sdkVersion = SDKUtils.getRTCSdkVersion(),
                platform = DeviceUtils.getPlatform(),
                userSnapshot = oppositeUsers?.toMutableList(),
                createTime = TimeUtils.getNowMills()
            )
        }
        problemRepository.updateIssue(issue)
        issue?.apply {
            issueChangedLiveData.postValue(this)
        }
    }

    private fun queryLocalUser() {
        localUserChangedLiveData.postValue(getLocalUser())
    }

    private fun queryAttachmentList() {
        viewModelScope.launch(Dispatchers.Main) {
            withContext(Dispatchers.IO) {
                findAttachmentCompleted(problemRepository.getIssue()?.attachmentList)
            }
        }
    }

    private fun findAttachmentCompleted(list: ArrayList<Attachment>?) {
        val ret = arrayListOf<Attachment>()
        list?.forEach { attachment ->
            if (attachment.dumpingStatus == AudioDumpStatus.COMPLETED.value ||
                attachment.dumpingStatus == AudioDumpStatus.STOPPED.value
            ) {
                ret.add(attachment)
            }
        }
        attachmentLiveData.postValue(ret)
    }

    private fun reportIssue(issue: Issue) {
        viewModelScope.launch(Dispatchers.Main) {
            withContext(Dispatchers.IO) {
                val problemId = problemRepository.addProblem(
                    issue,
                    getLocalUser()?.streamId,
                    getLocalUser()?.uid
                )

            }
        }
    }

    override fun onEventReceived(arg: MessageEvent) {
        when (arg.type) {
            AppEvent.AUDIO_DUMPING_STATUS_CHANGE.ordinal -> {
                val data = arg.obj
                if (data is Issue) {
                    if (data.dumpStatus == DumpStatus.FINISHED) {
                        queryAttachmentList()
                    }
                }
            }
            AppEvent.BUG_REPORT_STATUS_CHANGE.ordinal -> {
                val data = arg.obj
                if (data is Issue) {
                    reportResultEvent.postValue(data.status)
                }
            }
        }
    }

    override fun getUIEvents(): Array<AppEvent>? {
        return arrayOf(
            AppEvent.AUDIO_DUMPING_STATUS_CHANGE,
            AppEvent.BUG_REPORT_STATUS_CHANGE,
        )
    }

    fun doSubmit(info: ReportInfo) {
        val issue = problemRepository.getIssue()
        stashIssue(info)
        issue?.let {
            bugReporter.reportIssue(issue)
        }
    }

    //TODO:remove file from disk
    fun removeAttachment(attachment: Attachment) {
        problemRepository.getIssue()?.attachmentList?.remove(attachment)
        queryAttachmentList()
    }

    fun quitReporting() {
        audioDumper.destroy()
        problemRepository.clearIssue()
    }

    fun stashIssue(info: ReportInfo) {
        val issue = problemRepository.getIssue()
        if (issue == null) {
            Logger.e(TAG, "cloud not stash issue, please check why issue is empty")
            return
        }
        issue.description = info.comment
        issue.oppositeStreamId = info.oppositeStreamId
        issue.packet = ""
        val tags = IntArray(info.audioChips.size.plus(info.videoChips.size))
        var chipIndex = 0
        for (tag in info.audioChips) {
            tags[chipIndex++] = tag
        }
        for (tag in info.videoChips) {
            tags[chipIndex++] = tag
        }
        issue.tags = tags.toMutableList()
    }

    fun prepare2Recording(info: ReportInfo) {
        val issue = problemRepository.getIssue()
        stashIssue(info)
        issue?.dumpStatus = DumpStatus.PREPARE
        appEventBus.notifyObservers(
            MessageEvent(
                AppEvent.AUDIO_DUMPING_STATUS_CHANGE.ordinal,
                issue
            )
        )
    }

    companion object {
        private const val TAG = "[VM][BugReport]"
    }
}