package io.agora.avc.app.chat

import android.app.Application
import androidx.lifecycle.MutableLiveData
import io.agora.avc.base.AppViewModel
import io.agora.avc.biz.AppController
import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.AppEventBus
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.bo.Message
import io.agora.avc.extensions.getConferenceNicknameMaybeAssistant
import io.agora.avc.repository.ChatRepository
import io.agora.avc.utils.GsonUtils
import io.agora.avc.utils.TimeUtils
import io.agora.frame.base.livedata.EventLiveData
import io.agora.logger.Logger
import javax.inject.Inject

class ChatViewModel @Inject constructor(
    application: Application,
    appController: AppController,
    private val appEventBus: AppEventBus,
    private val chatRepository: ChatRepository
) : AppViewModel(application, appController) {
    val messageListLiveData = MutableLiveData<List<Message>>()
    val messageStatusChangedLiveData = EventLiveData<HashMap<String, Any>>()
    val chatMessageReceivedLiveData = EventLiveData<Message>()

    override fun onCreate() {
        super.onCreate()
    }

    override fun onResume() {
        super.onResume()
        queryMessage()
    }

    private fun queryMessage() {
        messageListLiveData.postValue(chatRepository.getMessages())
    }

    fun sendChatMessage(msg: String) {
        createChatMessage(msg).apply {
            sendChatMessage(this)
        }
    }

    private fun createChatMessage(msg: String): Message {
        val localUser = getLocalUser()
        val message = Message(Message.CHAT_RIGHT)
        message.textContent = msg
        message.senderId = localUser?.uid
        message.senderName = localUser?.getConferenceNicknameMaybeAssistant()
        message.senderUrl = localUser?.portraitId
        message.timestamp = TimeUtils.getNowMills()
        message.isRead = true
        return message
    }

    private fun sendChatMessage(message: Message) {
        if (message.textContent == null) {
            Logger.e(TAG, "Not allow sending empty message,content:${message.textContent}")
            return
        }
        appController.sendChat(message.textContent)
    }

    fun resendMessage(id: Int) {
        appController.resendChat(id)
    }

    fun remarkAllMessageRead() {
        var index = chatRepository.getMessages().size
        while (index > 0) {
            val message = chatRepository.getMessages()[index - 1]
            if (message?.isRead == true) {
                break
            }
            message?.isRead = true
            index--
        }
        val event = MessageEvent(AppEvent.CHAT_MESSAGE_READ.ordinal)
        appEventBus.notifyObservers(event)
    }

    override fun onEventReceived(arg: MessageEvent) {
        when (arg.type) {
            AppEvent.CHAT_MESSAGE_ADD.ordinal -> {
                val data = arg.data?.getString(Message.KEY)
                val message = GsonUtils.fromJson(data, Message::class.java)
                chatMessageReceivedLiveData.postValue(message)
            }
            AppEvent.CHAT_MESSAGE_UPDATE.ordinal -> {
                val index = arg.obj as Int
                val data = hashMapOf<String, Any>()
                data["index"] = index
                data["message"] = chatRepository.getMessages()[index]
                messageStatusChangedLiveData.postValue(data)
            }
        }
    }

    override fun getUIEvents(): Array<AppEvent>? {
        return arrayOf(
            AppEvent.CHAT_MESSAGE_ADD,
            AppEvent.CHAT_MESSAGE_UPDATE,
        )
    }

    companion object {
        private const val TAG = "[VM][Chat]"
    }
}