package io.agora.avc.app.enter

import android.Manifest
import android.app.Activity
import android.content.Intent
import android.content.pm.ActivityInfo
import android.content.res.Configuration
import android.net.Uri
import android.os.Build
import android.os.Bundle
import android.provider.Settings
import android.text.Editable
import android.text.InputFilter
import android.text.TextWatcher
import android.util.Log
import android.view.View
import android.view.ViewGroup
import android.view.WindowManager
import android.view.inputmethod.EditorInfo
import androidx.annotation.RequiresApi
import androidx.appcompat.app.AlertDialog
import androidx.appcompat.content.res.AppCompatResources
import androidx.core.os.bundleOf
import androidx.lifecycle.Observer
import androidx.navigation.fragment.findNavController
import com.google.android.material.dialog.MaterialAlertDialogBuilder
import io.agora.avc.BuildConfig
import io.agora.avc.R
import io.agora.avc.app.master.ActionType
import io.agora.avc.app.master.AgoraActivity
import io.agora.avc.app.master.MainViewModel
import io.agora.avc.biz.RoomChecker
import io.agora.avc.bo.AppTips
import io.agora.avc.bo.LastMile
import io.agora.avc.bo.RoomMode
import io.agora.avc.bo.valoran.ARoomUser
import io.agora.avc.config.PREFIX_INTERNAL_MEETING
import io.agora.avc.dao.PermissionDao
import io.agora.avc.databinding.AvcFragmentEnterBinding
import io.agora.avc.extensions.getThirdPartyNickName
import io.agora.avc.extensions.safeNavigate
import io.agora.avc.extensions.showPop
import io.agora.avc.manager.splite.SPLiteProxy
import io.agora.avc.push.notification.CallingNotification
import io.agora.avc.utils.*
import io.agora.avc.widget.*
import io.agora.frame.base.NovelFragment
import io.agora.frame.base.livedata.StatusEvent
import io.agora.logger.Logger
import java.util.*

const val KEY_PAGE_TYPE = "key_page_type"

class EnterFragment : NovelFragment<EnterViewModel, AvcFragmentEnterBinding>() {

    private var lastMile: LastMile? = null
    private var localUser: ARoomUser? = null

    private var microphoneOpened: Boolean = false

    private var cameraOpened: Boolean = false

    private var signalPopWindow: SignalQualityPopupWindow2? = null

    @ActionType
    private var pageType: Int = ActionType.CREATE_ROOM

    private val loadingView by lazy { LoadingDialog(requireContext()) }

    private val entrancePopupWindow by lazy {
        EntrancePopupWindow(context as Activity, R.string.avc_login_entrance_tip)
    }

    private var commonDialog: AlertDialog? = null

    private val roomNameTextWatcher: TextWatcher = object : TextWatcher {
        override fun afterTextChanged(s: Editable?) {
            if (s != null && s.length >= RoomChecker.MIN_ROOM_NAME_LENGTH) {
                mBinding.inputRoomName.error = null
            } else {
                mBinding.inputRoomName.error = getString(R.string.avc_room_name_min_length)
            }
        }

        override fun beforeTextChanged(s: CharSequence?, start: Int, count: Int, after: Int) {
        }

        override fun onTextChanged(s: CharSequence?, start: Int, before: Int, count: Int) {
        }
    }

    private val nickNameTextWatcher: TextWatcher = object : TextWatcher {
        override fun afterTextChanged(s: Editable?) {
            if (s != null && s.length >= RoomChecker.MIN_NICK_NAME_LENGTH) {
                mBinding.inputNickName.error = null
            } else {
                mBinding.inputNickName.error = getString(R.string.avc_toast_name_empty)
            }
        }

        override fun beforeTextChanged(p0: CharSequence?, p1: Int, p2: Int, p3: Int) {
        }

        override fun onTextChanged(p0: CharSequence?, p1: Int, p2: Int, p3: Int) {
        }
    }

    private val mainViewModel: MainViewModel by lazy {
        getViewModel(requireActivity().viewModelStore, MainViewModel::class.java)
    }

    private var roomMode = RoomMode.AGORA.value

    override fun getLayoutId(): Int {
        return R.layout.avc_fragment_enter
    }

    /**
     *  Clear colorStateCaches,
     *  the getColorStateList method below api23 will fetch the cache colorStateCaches,
     *  which causes the cache to be fetched after switching the theme
     */
    @RequiresApi(21)
    private fun flushLollipops() {
        val colorStateCachesField =
            AppCompatResources::class.java.getDeclaredField("sColorStateCaches")
        colorStateCachesField.isAccessible = true
        var colorStateCaches: WeakHashMap<*, *>? = null
        try {
            colorStateCaches = colorStateCachesField.get(null) as WeakHashMap<*, *>
        } catch (e: IllegalAccessException) {
            Log.e(
                TAG,
                "Could not retrieve value from AppCompatResources#sColorStateCaches",
                e
            )
        }
        colorStateCaches?.clear()
    }

    override fun allocObserver() {
        mainViewModel.roomModeChangedLiveData.observe(this) {
//            roomMode = it
            val act = activity
            if (act is AgoraActivity) {
                act.setRoomTheme(it)
                flushLollipops()
            }
            refreshUI()
        }

        mainViewModel.syncRoomModeEvent.observe(this) {
            flushLollipops()
            refreshUI()
        }

        mainViewModel.joinRoomEvent.observe(this) { room ->
            if (room?.isInternal() == true) {
                if (isLocalUserLoggedIn()) {
                    mBinding.editRoomName.setText(room.name)
                    mBinding.editRoomPwd.setText(room.pwd)
                }
            } else {
                mBinding.editRoomName.setText(room?.name)
                mBinding.editRoomPwd.setText(room?.pwd)
            }
            val nickName = mBinding?.inputNickName?.editText?.text
            if (nickName == null || nickName.isEmpty()) {
                showUserNameError()
                return@observe
            }
            if (roomMode != room.mode) {
                mainViewModel.changeRoomMode(room.mode)
            }
            onJoinClick(
                room.name,
                room.pwd,
                nickName.toString(),
                RoomMode.AGORA.value
            )
        }

        mViewModel?.statusEvent?.observe(this, Observer {
            when (it) {
                StatusEvent.Status.LOADING -> {
                    loadingView.show()
                }
                StatusEvent.Status.SUCCESS -> {
                    loadingView.dismiss()
                }
                StatusEvent.Status.ERROR -> {
                    loadingView.dismiss()
                }
                else -> {
                }
            }
        })

        mViewModel?.errorUserNameLiveData?.observe(this, Observer {
            showUserNameError()
        })

        mViewModel?.joinRoomLiveData?.observe(this) {
            if (findNavController().currentDestination?.id == R.id.callRating) {
                findNavController().navigateUp()
            }
            safeNavigate(R.id.action_enter_to_meeting, it)
        }

        mViewModel?.roomChecker?.errorLiveData?.observe(this, Observer {
            when (it) {
                RoomChecker.Error.RID_INVALID -> {
                    mBinding.inputRoomName.error = getString(R.string.avc_room_name_limit)
                    requestFocusForRoomName()
                }
                RoomChecker.Error.RID_TOO_SHORT -> {
                    mBinding.inputRoomName.error = getString(R.string.avc_room_name_min_length)
                    requestFocusForRoomName()
                }
                RoomChecker.Error.RID_TOO_LONG -> {
                    mBinding.inputRoomName.error = getString(R.string.avc_room_name_max_length)
                    requestFocusForRoomName()
                }
                RoomChecker.Error.PWD_TOO_LONG -> {
                    mBinding.inputRoomPwd.error = getString(R.string.avc_room_name_max_length)
                    requestFocusForRoomName()
                }
            }
        })

        mViewModel?.enterNewRoom?.observe(this) { room ->
            if (room?.isInternal() == true) {
                if (isLocalUserLoggedIn()) {
                    mBinding.editRoomName.setText(room.name)
                    mBinding.editRoomPwd.setText(room.pwd)
                }
            } else {
                mBinding.editRoomName.setText(room?.name)
                mBinding.editRoomPwd.setText(room?.pwd)
            }
        }

        mainViewModel.oemActionEvent.observe(this) {
            this.pageType = it.action
            refreshActionType()
            it.rid?.let { rid ->
                mBinding.editRoomName.setText(rid)
            }
            it.pwd?.let { pwd ->
                mBinding.editRoomPwd.setText(pwd)
            }
        }

        mViewModel?.lastMileChangedLiveData?.observe(this, Observer {
            lastMile = it
            it?.let { _lastMile ->
                mBinding.signal.setQuality(_lastMile.quality)
            }
            signalPopWindow?.setData(it)
        })

        mViewModel?.localUserInfoChanged?.observe(this, Observer { user ->
            if (user == null) {
                return@Observer
            }
            this.localUser = user
            if (user.isThirdPartyLoggedIn) {
                showEntranceTips(user)
            }
            initAccount()
        })

        mViewModel?.localMicrophoneChangedLiveData?.observe(this) {
            microphoneOpened = it
            initAccount()
        }

        mViewModel?.localCameraChangedLiveData?.observe(this) {
            cameraOpened = it
            initAccount()
        }

        mViewModel?.openNotificationGuideEvent?.observe(this) {
            if (it) {
                showNotificationRequestConfirmDialog()
            }
        }

        mViewModel?.openBannerNotificationGuideEvent?.observe(this) {
            if (it) {
                showBannerNotificationRequestConfirmDialog()
            }
        }

        mViewModel?.callRatingEvent?.observe(this) {
//            safeNavigate(
//                R.id.action_enter_to_call_rating,
//                bundleOf("data" to GsonUtils.toJson(it))
//            )
        }

        mViewModel?.needShowFraudTipsEvent?.observe(this) {
//            mBinding.layoutNotification.root.isVisible = it
        }

        mainViewModel.appConfigLiveData.observe(this) {
            mViewModel.queryNeedShowFraudTips()
        }
    }

    /**
     * Need to reset ui elements
     */
    private fun refreshUI() {
        val roomName = mBinding.editRoomName.text?.toString()
        val roomPwd = mBinding.editRoomPwd.text?.toString()
        val viewGroup = view as ViewGroup
        viewGroup.removeAllViews()
        val layout = onCreateView(layoutInflater, viewGroup, null)
        viewGroup.addView(layout)
        mBinding.editRoomName.setText(roomName)
        mBinding.editRoomPwd.setText(roomPwd)
        initAccount()
    }

    private fun showUserNameError() {
        //switch scenes and set username
        mBinding.inputNickName.error = getString(R.string.avc_toast_name_empty)
        mBinding.inputNickName.requestFocus()
    }

    private fun showEntranceTips(user: ARoomUser) {
        val appTips = SPLiteProxy.getAppTips() ?: AppTips()
        if (!user.isThirdPartyLoggedIn || appTips.innerRoom) {
            return
        }
        appTips.innerRoom = true
        SPLiteProxy.setAppTips(appTips)
    }

    private fun refreshActionType() {
        //todo replace string
        if (pageType == ActionType.JOIN_ROOM) {
            mBinding.tvTitle.text = "加入会议"
            mBinding.btJoin.text = "加入会议"
        } else if (pageType == ActionType.CREATE_ROOM) {
            mBinding.tvTitle.text = "发起会议"
            mBinding.btJoin.text = "发起会议"
        }
    }

    override fun initialize(savedInstanceState: Bundle?) {
        refreshActionType()
        mBinding.btBack.setOnClickListener {
            activity?.finish()
        }
        mBinding.btJoin.setOnClickListener {
            if ((mBinding.editRoomName.text?.length ?: 0) < RoomChecker.MIN_ROOM_NAME_LENGTH) {
                mBinding.inputRoomName.error = getString(R.string.avc_room_name_min_length)
                return@setOnClickListener
            }
            if ((mBinding.editNickName.text?.length ?: 0) <= 0) {
                mBinding.inputNickName.error = getString(R.string.avc_toast_name_empty)
                return@setOnClickListener
            }
            val roomName = (mBinding.inputRoomName.prefixText?.toString()
                ?: "").plus(mBinding.editRoomName.text.toString().uppercase(Locale.ENGLISH))
            val roomPwd = mBinding.inputRoomPwd.editText?.text?.toString()
            val userName = mBinding.inputNickName.editText?.text?.toString()
            onJoinClick(roomName, roomPwd, userName, roomMode)
        }
        mBinding.editRoomName.transformationMethod = UpperCaseTransform()
        mBinding.editRoomName.filters = arrayOf(InputFilter { source, start, end, _, _, _ ->
            if (source?.startsWith(PREFIX_INTERNAL_MEETING) == true) {
                Logger.i(TAG, "Remove the AG- in the copied content:$source")
                return@InputFilter source.removePrefix(PREFIX_INTERNAL_MEETING)
            }
            return@InputFilter source
        }, InputFilter.LengthFilter(RoomChecker.MAX_ROOM_NAME_LENGTH))
        mBinding.editRoomPwd.filters = arrayOf(InputFilter { source, start, end, _, _, _ ->
            for (i in start until end) {
                if (source[i] !in RoomChecker.PWD_FILTER) {
                    return@InputFilter ""
                }
            }
            null
        }, InputFilter.LengthFilter(RoomChecker.MAX_ROOM_PWD_LENGTH))

        mBinding.editNickName.apply {
            setOnFocusChangeListener { v, hasFocus ->
                Logger.e(TAG, "nick name focus changed:$hasFocus")
                if (!hasFocus) {
                    updateUsername()
                }
            }
            setOnEditorActionListener { v, actionId, _ ->
                if (actionId == EditorInfo.IME_ACTION_DONE) {
                    updateUsername()
                    hideKeyboard()
                }
                true
            }

            filters = arrayOf(InputFilter { source, start, end,
                                            dest, dstart, dend ->
                // 判断是否输入空格
                if (" " == source && length() == 0) {
                    ""
                } else {
                    null
                }
            }, InputFilter.LengthFilter(18))
        }

        mBinding.btSettings.setOnClickListener {
            safeNavigate(R.id.action_enter_to_mine)
        }

        mBinding.signal.setOnClickListener {
            showSignalTip(lastMile)
        }

        mBinding.inputRoomPwd.setEndIconOnClickListener {
            showPwdTip()
        }

        mBinding.cbCamera.setOnCheckedChangeListener { buttonView, isChecked ->
            mViewModel?.setLocalCamera(isChecked)
        }

        mBinding.cbMicrophone.setOnCheckedChangeListener { buttonView, isChecked ->
            mViewModel?.setLocalMicrophone(isChecked)
        }

        mBinding.constraintLayout.setOnClickListener {
            KeyboardUtils.hideSoftInput(it)
        }

        initRoomName()
        requestFocusForRoomName()
    }

    private fun showNotificationRequestConfirmDialog() {
        commonDialog?.dismiss()
        commonDialog =
            MaterialAlertDialogBuilder(requireContext(), R.style.avc_CustomMaterialAlertDialog)
                .setTitle(getString(R.string.avc_Android_open_notification))
                .setCancelable(false)
                .setNegativeButton(R.string.avc_cancel) { _, _ ->
                    commonDialog = null
                }
                .setPositiveButton(R.string.avc_confirm) { _, _ ->
                    jumpToNotificationSetting()
                    commonDialog = null
                }
                .show()
    }

    private fun showBannerNotificationRequestConfirmDialog() {
        commonDialog?.dismiss()
        commonDialog =
            MaterialAlertDialogBuilder(requireContext(), R.style.avc_CustomMaterialAlertDialog)
                .setTitle(getString(R.string.avc_android_notification_banner))
                .setCancelable(false)
                .setNegativeButton(R.string.avc_Androdi_open_notification_no) { _, _ ->
                    commonDialog = null
                }
                .setPositiveButton(R.string.avc_Androdi_open_notification_yes) { _, _ ->
                    jumpToNotificationSetting(CallingNotification.CHANNEL_ID)
                    commonDialog = null
                }
                .show()
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        activity?.requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_PORTRAIT
    }

    private fun requestFocusForRoomName() {
        mBinding.editRoomName.post {
            if (mBinding == null) {
                return@post
            }
            mBinding.editRoomName.text?.length?.let {
                if (it > 0) {
                    mBinding.editRoomName.setSelection(it)
                }
            }
            mBinding.editRoomName.requestFocus()
        }
    }

    override fun onBackPressed() {
        mViewModel?.onBackPressed()
        finish()
    }

    private fun initRoomName() {
        if (BuildConfig.DEBUG) {
            mBinding.editRoomName.setText("H2020")
        }
    }

    private fun initAccount() {
        if (roomMode == RoomMode.AGORA.value) {
            mBinding.inputRoomName.prefixText = PREFIX_INTERNAL_MEETING
            mBinding.inputNickName.hint = StringUtils.getString(R.string.avc_join_room_agora_name_title)
            mBinding.inputNickName.isEnabled = false
            mBinding.editRoomPwd.imeOptions = EditorInfo.IME_ACTION_DONE
            mBinding.editRoomPwd.nextFocusDownId = View.NO_ID
            mBinding.editNickName.setTextColor(
                ThemeUtils.getAttrValue(
                    requireContext(),
                    R.attr.inputHintColorPrimary
                )
            )
            if (!StringUtils.isEmpty(localUser?.getThirdPartyNickName())) {
                mBinding.editNickName.setText(localUser?.getThirdPartyNickName())
            }
        } else {
            mBinding.inputRoomName.prefixText = ""
            mBinding.inputNickName.hint = StringUtils.getString(R.string.avc_join_room_nickname_title)
            mBinding.inputNickName.isEnabled = false
            mBinding.editRoomPwd.imeOptions = EditorInfo.IME_ACTION_DONE
            mBinding.editRoomPwd.nextFocusDownId = View.NO_ID
            mBinding.editNickName.setTextColor(
                ThemeUtils.getAttrValue(
                    requireContext(),
                    R.attr.inputHintColorPrimary
                )
            )
            if (!StringUtils.isEmpty(localUser?.name)) {
                mBinding.editNickName.setText(localUser?.name)
            }
        }

        mBinding.cbCamera.isChecked = cameraOpened
        mBinding.cbMicrophone.isChecked = microphoneOpened
    }

    private fun updateUsername() {
        mViewModel.updateUserName(mBinding.editNickName.text.toString().trim())
    }

    private fun showSignalTip(lastMile: LastMile?) {
        activity?.let { _activity ->
            if (signalPopWindow == null) {
                showPop(mBinding.signal) {
                    signalPopWindow = SignalQualityPopupWindow2(
                        _activity,
                        lastMile
                    )
                    signalPopWindow!!
                }
            } else {
                signalPopWindow!!.setData(lastMile)
                showPop(mBinding.signal) {
                    signalPopWindow!!
                }
            }
        }
    }

    private fun showPwdTip() {
        val pos = IntArray(2)
        mBinding.inputRoomPwd.findViewById<View>(R.id.text_input_end_icon)?.let {
            it.getLocationOnScreen(pos)
            if (context?.resources?.configuration?.orientation == Configuration.ORIENTATION_PORTRAIT) {
                pos[0] += it.width / 2
                pos[1] += it.height * 4 / 5
                TextPopupWindow(requireActivity(), R.string.avc_join_room_password_info)
                    .showAtLocation(pos, TipPopupWindow.DIRECTION_DOWN)
            } else {
                pos[0] += it.width / 2
                pos[1] += it.height * 1 / 3
                TextPopupWindow(requireActivity(), R.string.avc_join_room_password_info)
                    .showAtLocation(pos, TipPopupWindow.DIRECTION_UP)
            }
        }
    }

    private fun onJoinClick(rid: String, roomPwd: String?, userName: String?, mode: Int) {
        Logger.i(TAG, "User clicks the join button")
        val doAction: (Boolean, Boolean) -> Unit = { audioPermit, videoPermit ->
            if (pageType == ActionType.JOIN_ROOM) {
                mViewModel?.joinRoom(
                    rid,
                    roomPwd,
                    userName,
                    mode,
                    audioPermit = audioPermit,
                    videoPermit = videoPermit
                )
            } else {
                mViewModel?.createRoom(
                    rid,
                    roomPwd,
                    userName,
                    mode,
                    audioPermit = audioPermit,
                    videoPermit = videoPermit
                )
            }
        }
        PermissionUtils
            .permission(PermissionConstants.MICROPHONE, PermissionConstants.CAMERA)
            .callback(object : PermissionUtils.FullCallback {
                override fun onGranted(permissionsGranted: MutableList<String>) {
                    if (PermissionDao.isAllGranted(permissionsGranted, getRequestPermission())) {
                        doAction(true, true)
                    }
                }

                private fun getRequestPermission(): Array<String> {
                    return arrayOf(Manifest.permission.CAMERA, Manifest.permission.RECORD_AUDIO)
                }

                override fun onDenied(
                    permissionsDeniedForever: MutableList<String>,
                    permissionsDenied: MutableList<String>
                ) {
                    mainViewModel.onPermissionDenied(permissionsDeniedForever, permissionsDenied)
                    val audioPermit = !permissionsDenied.contains(Manifest.permission.RECORD_AUDIO)
                    val videoPermit = !permissionsDenied.contains(Manifest.permission.CAMERA)
                    doAction(audioPermit, videoPermit)
                }
            })
            .request()
    }

    private fun isLocalUserLoggedIn() = localUser?.isThirdPartyLoggedIn == true

    override fun onStart() {
        super.onStart()
        mBinding.editRoomName.addTextChangedListener(roomNameTextWatcher)
        mBinding.editNickName.addTextChangedListener(nickNameTextWatcher)
        activity?.window?.setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_ADJUST_NOTHING)
    }

    override fun onStop() {
        super.onStop()
        mBinding.editRoomName.removeTextChangedListener(roomNameTextWatcher)
        mBinding.editNickName.removeTextChangedListener(nickNameTextWatcher)
        activity?.window?.setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_ADJUST_RESIZE or WindowManager.LayoutParams.SOFT_INPUT_STATE_HIDDEN)
        hideKeyboard()
    }

    private fun redisplayEntranceTips() {
        if (entrancePopupWindow.isShowing) {
            entrancePopupWindow.dismiss()
        }
    }

    private fun hideKeyboard() {
        activity?.let {
            KeyboardUtils.hideSoftInput(it)
        }
    }

    override fun onDestroyView() {
        mBinding.editNickName.setOnEditorActionListener(null)
        mBinding.editNickName.onFocusChangeListener = null
        if (loadingView.isShowing) {
            loadingView.dismiss()
        }
        mBinding.editRoomName.clearFocus()
        mBinding.editRoomPwd.clearFocus()
        mBinding.editNickName.clearFocus()
        super.onDestroyView()
    }

    private fun jumpToNotificationSetting(channel: String? = null) {
        context?.apply {
            val intent = Intent()
            when {
                Build.VERSION.SDK_INT >= Build.VERSION_CODES.P -> {
                    intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
                    if (channel != null) {
                        intent.action = Settings.ACTION_CHANNEL_NOTIFICATION_SETTINGS
                        intent.putExtra(Settings.EXTRA_CHANNEL_ID, channel)
                    } else {
                        intent.action = Settings.ACTION_APP_NOTIFICATION_SETTINGS
                    }
                    intent.putExtra(Settings.EXTRA_APP_PACKAGE, packageName)
                }
                Build.VERSION.SDK_INT >= Build.VERSION_CODES.O -> {
                    if (channel != null) {
                        intent.action = Settings.ACTION_CHANNEL_NOTIFICATION_SETTINGS
                        intent.putExtra(Settings.EXTRA_CHANNEL_ID, channel)
                    } else {
                        intent.action = Settings.ACTION_APP_NOTIFICATION_SETTINGS
                    }
                    intent.putExtra(Settings.EXTRA_APP_PACKAGE, packageName)
                }
                Build.VERSION.SDK_INT >= Build.VERSION_CODES.N_MR1 -> {
                    intent.action = Settings.ACTION_APP_NOTIFICATION_SETTINGS
                    intent.putExtra(Settings.EXTRA_APP_PACKAGE, packageName)
                }
                Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP -> {
                    intent.action = Settings.ACTION_APP_NOTIFICATION_SETTINGS
                    intent.putExtra("app_package", packageName)
                    intent.putExtra("app_uid", applicationInfo.uid)
                }
                Build.VERSION.SDK_INT == Build.VERSION_CODES.KITKAT -> {
                    intent.action = Settings.ACTION_APPLICATION_DETAILS_SETTINGS
                    intent.addCategory(Intent.CATEGORY_DEFAULT)
                    intent.data = Uri.parse("package:$packageName")
                }
            }
            if (intent.resolveActivity(packageManager) == null) {
                Logger.e(TAG, "can not jump to notification setting")
                return
            }
            startActivity(intent)
        }
    }

    companion object {
        private const val TAG = "[UI][Enter]"
    }
}
