package io.agora.avc.app.enter

import android.Manifest
import android.app.Application
import android.app.NotificationManager
import android.content.pm.PackageManager
import android.os.Build
import android.os.Bundle
import androidx.core.app.ActivityCompat
import androidx.core.app.NotificationManagerCompat
import androidx.core.os.bundleOf
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.viewModelScope
import com.agora.valoran.bean.UserJoinInfo
import io.agora.avc.MyApplication
import io.agora.avc.base.AppViewModel
import io.agora.avc.biz.AppController
import io.agora.avc.biz.RoomChecker
import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.AppEventBus
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.bo.*
import io.agora.avc.bo.valoran.ARoomUser
import io.agora.avc.net.BaseObserver
import io.agora.avc.net.bean.BaseNo
import io.agora.avc.push.notification.CallingNotification
import io.agora.avc.repository.*
import io.agora.avc.utils.StringUtils
import io.agora.frame.base.livedata.EventLiveData
import io.agora.logger.Logger
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.schedulers.Schedulers
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import javax.inject.Inject

class EnterViewModel @Inject constructor(
    application: Application,
    appController: AppController,
    private val mediaConfigRepository: MediaConfigRepository,
    private val appConfigRepository: AppConfigRepository,
    private val userRepository: UserRepository,
    private val roomRepository: RoomRepository,
    private val equipmentRepository: EquipmentRepository,
    private val appEventBus: AppEventBus,
) : AppViewModel(application, appController) {
    val joinRoomLiveData: EventLiveData<Bundle> = EventLiveData()
    val errorUserNameLiveData: EventLiveData<Boolean> = EventLiveData()
    val roomChecker: RoomChecker = RoomChecker()
    val lastMileChangedLiveData = EventLiveData<LastMile?>()
    val localUserInfoChanged = EventLiveData<ARoomUser?>()
    val enterNewRoom = EventLiveData<Room>()
    val localMicrophoneChangedLiveData = MutableLiveData<Boolean>()
    val localCameraChangedLiveData = MutableLiveData<Boolean>()
    var lastMile: LastMile? = null
    val openNotificationGuideEvent = EventLiveData<Boolean>()
    val openBannerNotificationGuideEvent = EventLiveData<Boolean>()
    val callRatingEvent = EventLiveData<MosScore>()
    val needShowFraudTipsEvent = EventLiveData<Boolean>()

    override fun onCreate() {
        super.onCreate()
        checkNotificationPermission(CallingNotification.CHANNEL_ID)
    }

    override fun onResume() {
        super.onResume()
        queryLocalUser()
        queryCameraSetting()
        queryMicrophoneSetting()
        queryLastMile()
        queryNeedShowFraudTips()
    }

    private fun checkNotificationPermission(channelName: String) {
        val manager = NotificationManagerCompat.from(getApplication())
        val notificationEnabled = manager.areNotificationsEnabled()
        val notificationGuideTime = appConfigRepository.getOpenNotificationGuideTime() ?: 0
        val currentTime = System.currentTimeMillis()
        Logger.i(TAG, "ntEnable:$notificationEnabled")
        if (!notificationEnabled && currentTime - notificationGuideTime > WEEK_MILLISECONDS) {
            appConfigRepository.saveOpenNotificationGuideTime(currentTime)
            openNotificationGuideEvent.postValue(true)
            return
        }
        val bannerNotificationGuideTime = appConfigRepository.getBannerNotificationGuideTime() ?: 0
        val bannerNotificationEnable = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            val channel = manager.getNotificationChannel(channelName)
            if (channel != null) {
                channel.importance > NotificationManager.IMPORTANCE_DEFAULT
            } else {
                null
            }
        } else {
            manager.importance > NotificationManager.IMPORTANCE_DEFAULT
        }

        Logger.i(TAG, "bntEnable:$bannerNotificationEnable")
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O &&
            bannerNotificationEnable != null && !bannerNotificationEnable &&
            currentTime - bannerNotificationGuideTime > WEEK_MILLISECONDS
        ) {
            appConfigRepository.saveBannerNotificationGuideTime(currentTime)
            openBannerNotificationGuideEvent.postValue(true)
            return
        }
    }

    private fun queryMicrophoneSetting() {
        localMicrophoneChangedLiveData.postValue(mediaConfigRepository.getAudioSetting())
    }

    private fun queryCameraSetting() {
        localCameraChangedLiveData.postValue(mediaConfigRepository.getVideoSetting())
    }

    private fun queryLocalUser() {
        localUserInfoChanged.postValue(getLocalUser())
    }

    /**
     * logic:https://confluence.agoralab.co/display/BEC/AVC+6.2.1
     */
    fun queryNeedShowFraudTips() {
        appConfigRepository.needShowFraudTips = appConfigRepository.needShowFraudTips
                && appConfigRepository.getAppConfig()?.app?.enableHomeBanner ?: true
                && getLocalUser()?.isThirdPartyLoggedIn != true
                && appConfigRepository.getRoomMode() == RoomMode.NORMAL.value
        needShowFraudTipsEvent.postValue(appConfigRepository.needShowFraudTips)
    }

    fun closeFraudTips() {
        appConfigRepository.needShowFraudTips = false
        queryNeedShowFraudTips()
    }

    fun updateUserName(userName: String) {
        if (StringUtils.isEmpty(userName) || getLocalUser()?.name.equals(userName)) {
            Logger.i(TAG, "The username is the same, no need to change")
            return
        }
        userRepository.updateUserName(userName)
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(object : BaseObserver<BaseNo>() {
                override fun onSuccess(it: BaseNo?) {
                    queryLocalUser()
                }

                override fun onFail(t: BaseNo?) {

                }
            })
    }

    fun createRoom(
        roomName: String,
        roomPwd: String?,
        nameText: String?,
        mode: Int,
        audioPermit: Boolean,
        videoPermit: Boolean
    ) {
        Logger.i(
            TAG,
            "createRoom: roomName = $roomName, roomPwd = $roomPwd, nameText = $nameText, mode = $mode, audioPermit = $audioPermit, videoPermit = $videoPermit"
        )
        joinRoom(
            roomName = roomName,
            roomPwd = roomPwd,
            nameText = nameText,
            mode = mode,
            audioPermit = audioPermit,
            videoPermit = videoPermit,
            isCreateRoom = true,
        )
    }

    fun joinRoom(
        roomName: String,
        roomPwd: String?,
        nameText: String?,
        mode: Int,
        audioPermit: Boolean,
        videoPermit: Boolean
    ) {
        Logger.i(
            TAG,
            "joinRoom: roomName = $roomName, roomPwd = $roomPwd, nameText = $nameText, mode = $mode, audioPermit = $audioPermit, videoPermit = $videoPermit"
        )
        joinRoom(
            roomName = roomName,
            roomPwd = roomPwd,
            nameText = nameText,
            mode = mode,
            audioPermit = audioPermit,
            videoPermit = videoPermit,
            isCreateRoom = false,
        )
    }

    /**
     * request permission prefer, join meeting
     * check roomName & pwd by mode
     * check user name, if user name is empty, need prompt input
     * set user default audio & video status by user-setting & permission
     *
     */
    private fun joinRoom(
        roomName: String,
        roomPwd: String?,
        nameText: String?,
        mode: Int,
        audioPermit: Boolean,
        videoPermit: Boolean,
        isCreateRoom: Boolean,
    ) {
        viewModelScope.launch(Dispatchers.Main) {
            if (!roomChecker.checkRoom(roomName, roomPwd, mode)) return@launch
            userRepository.resetLocalUser()
            val localUser = getLocalUser()
            var localUserName = localUser?.name
            if (mode == RoomMode.NORMAL.value && localUserName.isNullOrEmpty() && nameText.isNullOrEmpty()) {
                Logger.w(TAG, "Join in the room:$roomName exception,user has not nickname")
                errorUserNameLiveData.postValue(true)
                return@launch
            }
            if (!localUserName.equals(nameText) && mode == RoomMode.NORMAL.value) {
                localUserName = nameText
                withContext(Dispatchers.IO) {
                    updateUserName(nameText!!)
                }
            }
            val audioSetting = mediaConfigRepository.getAudioSetting()
            val videoSetting = mediaConfigRepository.getVideoSetting()
            val userJoinInfo = UserJoinInfo()
            //If the setUserName interface is abnormal,
            // it is also ensured that the user can see his name normally after logging in
            userJoinInfo.name = localUserName
            userJoinInfo.audioDefault = audioSetting && audioPermit
            userJoinInfo.videoDefault = videoSetting && videoPermit
            userJoinInfo.resolution = mediaConfigRepository.getResolutionOption()
            userJoinInfo.isThirdPartyLoggedIn = localUser?.isThirdPartyLoggedIn ?: false
            userJoinInfo.thirdPartyName = localUser?.thirdPartyName
            userJoinInfo.thirdPartyAlias = localUser?.thirdPartyAlias
            userJoinInfo.thirdPartyDepartment = localUser?.thirdPartyDepartment
            Logger.i(
                TAG,
                "I join the room,roomName:$roomName," +
                        "pwd:$roomPwd," +
                        "mode:$mode," +
                        "audio:$audioSetting " +
                        "video:$videoSetting " +
                        "audioPermit:$audioPermit " +
                        "videoPermit:$videoPermit " +
                        "userInfo:$localUser "
            )
            val room = roomRepository.createRoom(roomName, roomPwd, mode, userJoinInfo.resolution)
            val application = getApplication<MyApplication>()
            application.appContainer.room = room
            appController.configClientInfo(
                localUser?.uid ?: "",
                equipmentRepository.getLangType(),
                equipmentRepository.getFeatureType(),
                equipmentRepository.getAppVersionName(),
                equipmentRepository.getEquipmentInfo(),
                equipmentRepository.getPlatform()
            )
            if (isCreateRoom) {
                appController.createRoom(roomName, roomPwd, userJoinInfo)
            } else {
                appController.joinRoom(roomName, roomPwd, userJoinInfo)
            }
            appEventBus.notifyObservers(MessageEvent(AppEvent.ROOM_INFO_CHANGED.ordinal))
            val data = bundleOf("roomName" to roomName, "roomPwd" to (roomPwd ?: ""))
            joinRoomLiveData.postValue(data)
        }
    }

    private fun joinRoom(
        roomName: String,
        roomPwd: String?,
        nameText: String?,
        mode: Int,
    ) {
        val audioPermit = ActivityCompat.checkSelfPermission(
            getApplication(),
            Manifest.permission.CAMERA
        ) == PackageManager.PERMISSION_GRANTED
        val videoPermit = ActivityCompat.checkSelfPermission(
            getApplication(),
            Manifest.permission.CAMERA
        ) == PackageManager.PERMISSION_GRANTED

        joinRoom(roomName, roomPwd, nameText, mode, audioPermit, videoPermit)
    }

    override fun onEventReceived(arg: MessageEvent) {
        when (arg.type) {
            AppEvent.LAST_MILE_QUALITY_CHANGED.ordinal -> {
                val lastMile = arg.obj
                if (lastMile is LastMile) {
                    this.lastMile = lastMile
                    queryLastMile()
                }
            }
            AppEvent.LOCAL_USER_CHANGED.ordinal -> {
                queryLocalUser()
            }
            AppEvent.ENTER_ROOM_EVENT.ordinal -> {
                arg.obj?.let { data ->
                    if (data is Room) {
                        enterNewRoom.postValue(data)
                        joinRoom(data.name, data.pwd, getLocalUser()?.name, data.mode)
                    }
                }
            }
            AppEvent.CALL_RATING_EVENT.ordinal -> {
                arg.obj?.let { data ->
                    if (data is MosScore) {
                        callRatingEvent.postValue(data)
                    }
                }
            }
        }
    }

    private fun queryLastMile() {
        lastMileChangedLiveData.postValue(lastMile)
    }

    override fun getUIEvents(): Array<AppEvent>? {
        return arrayOf(
            AppEvent.LAST_MILE_QUALITY_CHANGED,
            AppEvent.LOCAL_USER_CHANGED,
            AppEvent.ENTER_ROOM_EVENT,
            AppEvent.CALL_RATING_EVENT,
        )
    }

    fun onBackPressed() {
        Logger.i(TAG, "user exits normally")
        appController.destroy()
    }

    fun setLocalCamera(opened: Boolean) {
        mediaConfigRepository.saveVideoSetting(opened)
        localCameraChangedLiveData.postValue(opened)
    }

    fun setLocalMicrophone(opened: Boolean) {
        mediaConfigRepository.saveAudioSetting(opened)
        localMicrophoneChangedLiveData.postValue(opened)
    }

    companion object {
        private const val TAG = "[VM][Enter]"
        private const val WEEK_MILLISECONDS = 7 * 24 * 60 * 60 * 1000
    }
}