package io.agora.avc.app.group

import android.app.Application
import androidx.lifecycle.MutableLiveData
import com.agora.valoran.Constants
import io.agora.avc.app.address.AddressBookNode
import io.agora.avc.app.address.ErrorNode
import io.agora.avc.app.address.MemberNode
import io.agora.avc.base.AppViewModel
import io.agora.avc.biz.AppController
import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.bo.LocalUser
import io.agora.avc.bo.NoticeCode
import io.agora.avc.manager.network.NetWorkInfo
import io.agora.avc.manager.notice.NoticeEvent
import io.agora.avc.manager.notice.NoticeManager
import io.agora.avc.manager.notice.NoticeType
import io.agora.avc.net.bean.ErrorNo
import io.agora.avc.repository.AddressBookRepository
import io.agora.avc.utils.StringUtils
import io.agora.logger.Logger
import javax.inject.Inject

class GroupTailoredViewModel @Inject constructor(
    application: Application,
    appController: AppController,
    private val addressBookRepository: AddressBookRepository,
    private val noticeManager: NoticeManager,
) : AppViewModel(application, appController) {

    val groupNameLiveData = MutableLiveData<String>()
    val membersLiveData = MutableLiveData<ArrayList<AddressBookNode>>()
    val localUserLiveData = MutableLiveData<LocalUser>()
    val exitLiveData = MutableLiveData<Boolean>()
    val contactGroupLiveData = MutableLiveData<ContactGroup?>()

    private val memberList = arrayListOf<AddressBookNode>()

    var groupId: String? = null
    private var currentSeq = -1
    private val errorMember = ErrorNode()
    private var contactGroup: ContactGroup? = null

    fun queryGroupDetails(groupId: String) {
        Logger.i(TAG, "query group details, groupId = $groupId")
        this.groupId = groupId
        showLoading()
        currentSeq = appController.getOperationSeqId()
        appController.getMeetupDetail(currentSeq, groupId)
    }

    fun submit(groupName: String) {
        Logger.i(TAG, "update groupName, groupId = $groupId, groupName = $groupName")
        if (StringUtils.isEmpty(this.groupId)) {
            addGroup(groupName)
        } else {
            updateGroupInfo(groupName)
        }
    }

    override fun onResume() {
        super.onResume()
        queryLocalUser()
        membersLiveData.postValue(memberList)
    }

    fun isChanged(): Boolean {
        if (memberList.size != contactGroup?.members?.size ?: -1) {
            return true
        }
        for (i in 0 until memberList.size) {
            if ((memberList[i] as? MemberNode)?.userid != contactGroup?.members?.get(i)?.userid) {
                return true
            }
        }
        return false
    }


    private fun queryLocalUser() {
        getLocalUser()?.let { user ->
            localUserLiveData.postValue(user)
        }
    }

    override fun onEventReceived(arg: MessageEvent) {
        when (arg.type) {
            AppEvent.MEETUP_QUERY_DETAIL_RESULT_EVENT.ordinal -> {
                onMeetupQueryDetailResponse(arg)
            }
            AppEvent.MEETUP_CREATE_RESULT_EVENT.ordinal,
            AppEvent.MEETUP_UPDATE_RESULT_EVENT.ordinal -> {
                doOnMeetUpResponse(arg)
            }
            AppEvent.MEETUP_EDIT_NAME_RESULT_EVENT.ordinal -> {
                (arg.obj as? Pair<*, *>)?.let {
                    this.groupNameLiveData.postValue(it.second as String)
                }
            }
            AppEvent.MEETUP_ADD_MEMBER_EVENT.ordinal -> {
                ((arg.obj as? MeetupResponse)?.data as? ArrayList<AddressBookNode>)?.let {
                    memberList.clear()
                    memberList.addAll(it)
                    membersLiveData.postValue(memberList)
                }
            }
            AppEvent.NET_WORK_CHANGED_EVENT.ordinal -> {
                //retry request group details
                if (contactGroup == null) {
                    groupId?.let { _groupId ->
                        (arg.obj as? NetWorkInfo)?.let { _netWorkInfo ->
                            if (_netWorkInfo.reConnected) {
                                queryGroupDetails(_groupId)
                            }
                        }
                    }
                }
            }
        }
    }

    private fun doOnMeetUpResponse(arg: MessageEvent) {
        Logger.i(TAG, "on meetup response success")
        (arg.obj as? MeetupResponse)?.let { _meetUpResponse ->
            if (currentSeq == _meetUpResponse.seq) {
                hideLoading()
                when (_meetUpResponse.resultType) {
                    Constants.MeetupResultType.SUCCESS.value -> {
                        (_meetUpResponse.data as? String).let { _groupId ->
                            Logger.i(TAG, "on meetup response success, groupId = $_groupId")
                        }
                        exitLiveData.postValue(true)
                    }
                    Constants.MeetupResultType.FAILURE_MAX.value -> {
                        noticeManager.notice(
                            NoticeEvent(
                                type = NoticeType.TOAST,
                                code = NoticeCode.CODE_GROUP_EXCEED
                            )
                        )
                    }
                    Constants.MeetupResultType.FAILURE_MAX_MEMBERS.value -> {
                        noticeManager.notice(
                            NoticeEvent(
                                type = NoticeType.TOAST,
                                code = NoticeCode.CODE_GROUP_MEMBER_EXCEED
                            )
                        )
                    }
                    Constants.MeetupResultType.FAILURE_UNKNOWN.value -> {
                        noticeManager.notice(
                            NoticeEvent(
                                type = NoticeType.TOAST,
                                code = NoticeCode.CODE_INVITE_ALERT
                            )
                        )
                    }
                }
            }
        }
    }

    private fun addGroup(groupName: String) {
        if (StringUtils.isEmpty(groupName)) {
            Logger.e(TAG, "add group failed, group name is empty")
            return
        }
        showLoading()
        Logger.i(TAG, "add group info, groupName = $groupName")
        currentSeq = appController.getOperationSeqId()
        appController.createMeetup(
            currentSeq,
            groupName,
            ArrayList<String>(memberList.size).apply {
                memberList.forEach {
                    add((it as MemberNode).userid)
                }
            }
        )
    }

    private fun updateGroupInfo(groupName: String) {
        fun convertToUserIds(_minusList: List<Any?>): ArrayList<String> {
            val newIds = ArrayList<String>(_minusList.size)
            _minusList.forEach { _memberNode ->
                if (_memberNode is MemberNode) {
                    newIds.add(_memberNode.userid)
                }
            }
            return newIds
        }

        if (StringUtils.isEmpty(groupName)) {
            Logger.e(TAG, "update group failed, group name is empty")
            return
        }
        showLoading()
        Logger.i(TAG, "update group info, groupName = $groupName")
        currentSeq = appController.getOperationSeqId()
        appController.updateMeetup(
            seq = currentSeq,
            id = groupId!!,
            name = groupName,
            newIds = convertToUserIds((memberList.minus(contactGroup?.members))),
            deleteIds = convertToUserIds(
                (contactGroup?.members?.minus(memberList)) ?: arrayListOf()
            )
        )
    }

    private fun onMeetupQueryDetailResponse(arg: MessageEvent) {
        (arg.obj as? MeetupResponse)?.let { _meetUpResponse ->
            if (currentSeq == _meetUpResponse.seq) {
                hideLoading()
                memberList.clear()
                if (_meetUpResponse.resultType != Constants.MeetupResultType.SUCCESS.value) {
                    memberList.add(errorMember)
                    membersLiveData.postValue(memberList)
                    return
                }

                (_meetUpResponse.data as ContactGroup).let { _contactGroup ->
                    this.contactGroup = _contactGroup
                    this.groupNameLiveData.postValue(_contactGroup.name)
                    _contactGroup.members?.let { _members ->
                        memberList.addAll(_members)
                    }
                    membersLiveData.postValue(memberList)
                    contactGroupLiveData.postValue(contactGroup)
                }
            }
        }
    }

    fun onAddClicked() {
        if (memberList.isNotEmpty()) {
            ArrayList<MemberNode>(memberList.size).let { _array ->
                memberList.forEach {
                    if (it is MemberNode) {
                        _array.add(it)
                    }
                }
                addressBookRepository.saveAdded(_array)
            }
        }
    }

    fun membersIsEmpty(): Boolean {
        return memberList.size == 0
                || (memberList.size == 1 && memberList[0] is ErrorNo)
    }

    override fun getUIEvents(): Array<AppEvent> {
        return arrayOf(
            AppEvent.MEETUP_QUERY_DETAIL_RESULT_EVENT,
            AppEvent.MEETUP_CREATE_RESULT_EVENT,
            AppEvent.MEETUP_UPDATE_RESULT_EVENT,
            AppEvent.MEETUP_EDIT_NAME_RESULT_EVENT,
            AppEvent.MEETUP_ADD_MEMBER_EVENT,
            AppEvent.NET_WORK_CHANGED_EVENT,
        )
    }

    companion object {
        private const val TAG = "[VM][GroupTailored]"
    }
}