package io.agora.avc.app.group

import android.os.Bundle
import androidx.core.os.bundleOf
import androidx.core.widget.doAfterTextChanged
import androidx.fragment.app.Fragment
import androidx.navigation.fragment.findNavController
import com.google.android.material.dialog.MaterialAlertDialogBuilder
import io.agora.avc.R
import io.agora.avc.databinding.AvcFragmentNameEditBinding
import io.agora.avc.extensions.safeNavigate
import io.agora.avc.utils.KeyboardUtils
import io.agora.avc.utils.StringUtils
import io.agora.avc.widget.OnActionTxtClickListener
import io.agora.frame.base.NovelFragment

class NameEditFragment : NovelFragment<NameEditViewModel, AvcFragmentNameEditBinding>() {

    private var sourceGroupName: String? = null

    override fun getLayoutId() = R.layout.avc_fragment_name_edit

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        sourceGroupName = arguments?.getString(KEY_NAME)
    }

    override fun initialize(savedInstanceState: Bundle?) {

        mBinding.etName.doAfterTextChanged {
            mBinding.etName.text?.toString().let { _groupName ->
                validGroupName(_groupName)
            }
        }

        arguments?.getString(KEY_GROUP_ID)?.let {
            mViewModel.saveGroupId(it)
        }
        arguments?.getString(KEY_NAME)?.let {
            mBinding.etName.setText(it)
        }

        mBinding.actionBar.setOnActionTxtClickListener(object : OnActionTxtClickListener {
            override fun onActionTxtClicked() {
                doOnActionTxtClicked()
            }
        })
    }

    private fun doOnActionTxtClicked() {
        mBinding.etName.text?.toString().let {
            if (!validGroupName(it)) {
                return@let
            }
            if (it == sourceGroupName) {
                onBackPressed()
                return@let
            }
            mViewModel.submit(it!!)
        }
    }

    private fun validGroupName(_groupName: String?): Boolean {
        if (StringUtils.isEmpty(_groupName)) {
            mBinding.tilName.error =
                getString(R.string.avc_group_name_null)
            return false
        }
        if (_groupName!!.length > mBinding.tilName.counterMaxLength) {
            mBinding.tilName.error =
                getString(R.string.avc_bug_report_comment_words_overflow)
            return false
        }
        mBinding.tilName.error = null
        return true
    }

    override fun allocObserver() {
        super.allocObserver()
        mViewModel.exitLiveData.observe(this) {
            findNavController().navigateUp()
            KeyboardUtils.hideSoftInput(mBinding.etName)
        }
    }

    override fun onBackPressed() {
        val editName = mBinding.etName.text.toString()
        if (!StringUtils.isEmpty(editName) && sourceGroupName != editName) {
            MaterialAlertDialogBuilder(requireContext(), R.style.avc_CustomMaterialAlertDialog)
                .setTitle(R.string.avc_group_quit_ask_title)
                .setNegativeButton(R.string.avc_group_quit_ask_button_no) { _, _ ->
                    super.onBackPressed()
                    KeyboardUtils.hideSoftInput(mBinding.etName)
                }
                .setPositiveButton(R.string.avc_group_save_mobile) { _, _ ->
                    doOnActionTxtClicked()
                }
                .show()
        } else {
            super.onBackPressed()
            KeyboardUtils.hideSoftInput(mBinding.etName)
        }

    }

    companion object {
        private const val KEY_GROUP_ID = "key_group_id"
        private const val KEY_NAME = "key_name"

        fun navigateTo(
            fragment: Fragment,
            groupId: String? = null,
            name: String? = null,
        ) {
            if (StringUtils.isEmpty(groupId) && StringUtils.isEmpty(name)) {
                fragment.safeNavigate(R.id.action_group_tailored_to_name_edit)
            } else {
                fragment.safeNavigate(
                    R.id.action_group_tailored_to_name_edit,
                    bundleOf(KEY_GROUP_ID to groupId, KEY_NAME to name)
                )
            }
        }
    }
}