package io.agora.avc.app.group

import android.app.Application
import androidx.lifecycle.MutableLiveData
import com.agora.valoran.Constants
import io.agora.avc.base.AppViewModel
import io.agora.avc.biz.AppController
import io.agora.avc.biz.event.AppEvent
import io.agora.avc.biz.event.AppEventBus
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.bo.NoticeCode
import io.agora.avc.manager.notice.NoticeEvent
import io.agora.avc.manager.notice.NoticeManager
import io.agora.avc.manager.notice.NoticeType
import io.agora.avc.utils.StringUtils
import io.agora.logger.Logger
import javax.inject.Inject

class NameEditViewModel @Inject constructor(
    application: Application,
    appController: AppController,
    private val appEventBus: AppEventBus,
    private val noticeManager: NoticeManager,
) : AppViewModel(application, appController) {

    val exitLiveData = MutableLiveData<Boolean>()

    private var currentSeq = -1
    private var groupId: String? = null
    private var groupName: String? = null

    fun saveGroupId(groupId: String) {
        this.groupId = groupId
    }

    override fun onEventReceived(arg: MessageEvent) {
        when (arg.type) {
            AppEvent.MEETUP_CREATE_RESULT_EVENT.ordinal,
            AppEvent.MEETUP_UPDATE_RESULT_EVENT.ordinal -> {
                doOnMeetUpResponse(arg)
            }
        }
    }

    private fun doOnMeetUpResponse(arg: MessageEvent) {
        Logger.i(TAG, "on meetup response success")
        (arg.obj as? MeetupResponse)?.let { _meetUpResponse ->
            if (currentSeq == _meetUpResponse.seq) {
                hideLoading()
                when (_meetUpResponse.resultType) {
                    Constants.MeetupResultType.SUCCESS.value -> {
                        (_meetUpResponse.data as? String).let { _groupId ->
                            if (StringUtils.isEmpty(_groupId)) {
                                if (arg.type == AppEvent.MEETUP_CREATE_RESULT_EVENT.ordinal) {
                                    Logger.e(TAG, "meetup create response, groupId is empty")
                                } else {
                                    Logger.e(TAG, "meetup update response, groupId is empty")
                                }
                            } else {
                                this.groupName?.let { _groupName ->
                                    notifyUpdateNameSuccessEvent(_groupId, _groupName)
                                }
                            }
                        }
                        exitLiveData.postValue(true)
                    }
                    Constants.MeetupResultType.FAILURE_MAX.value -> {
                        noticeManager.notice(
                            NoticeEvent(
                                type = NoticeType.TOAST,
                                code = NoticeCode.CODE_GROUP_EXCEED
                            )
                        )
                    }
                    Constants.MeetupResultType.FAILURE_MAX_MEMBERS.value -> {
                        noticeManager.notice(
                            NoticeEvent(
                                type = NoticeType.TOAST,
                                code = NoticeCode.CODE_GROUP_MEMBER_EXCEED
                            )
                        )
                    }
                    Constants.MeetupResultType.FAILURE_UNKNOWN.value -> {
                        noticeManager.notice(
                            NoticeEvent(
                                type = NoticeType.TOAST,
                                code = NoticeCode.CODE_GROUP_NAME_ERROR
                            )
                        )
                    }
                }
            }
        }
    }

    fun submit(groupName: String) {
        Logger.i(TAG, "update groupName, groupId = $groupId, groupName = $groupName")
        this.groupName = groupName
        if (StringUtils.isEmpty(groupId)) {
            notifyUpdateNameSuccessEvent(groupName = groupName)
            exitLiveData.postValue(true)
        } else {
            updateGroupInfo(groupName)
        }
    }

    private fun notifyUpdateNameSuccessEvent(_groupId: String? = null, groupName: String) {
        appEventBus.notifyObservers(
            MessageEvent(
                AppEvent.MEETUP_EDIT_NAME_RESULT_EVENT.ordinal,
                Pair(_groupId, groupName),
            )
        )
    }

    private fun updateGroupInfo(groupName: String) {
        Logger.i(TAG, "update group info, groupName = $groupName")
        currentSeq = appController.getOperationSeqId()
        appController.updateMeetup(
            currentSeq,
            groupId.toString(),
            groupName,
            arrayListOf(),
            arrayListOf(),
        )
    }

    override fun getUIEvents(): Array<AppEvent> {
        return arrayOf(
            AppEvent.MEETUP_CREATE_RESULT_EVENT,
            AppEvent.MEETUP_UPDATE_RESULT_EVENT,
        )
    }

    companion object {
        private const val TAG = "[VM][EditGroupName]"
    }
}