package io.agora.avc.app.master

import android.content.Context
import android.content.Intent
import android.media.AudioManager
import android.view.KeyEvent
import io.agora.avc.R
import io.agora.avc.receiver.VolumeBroadcastReceiver.Companion.VOLUME_CHANGED_ACTION
import io.agora.avc.utils.AppUtils
import io.agora.avc.utils.ToastUtils
import io.agora.logger.Logger

class VolumeBootHelper(var context: Context? = null) {

    private val maxVolume by lazy {
        (context?.getSystemService(Context.AUDIO_SERVICE) as AudioManager).getStreamMaxVolume(
            AudioManager.STREAM_MUSIC
        )
    }

    var isInPictureInPictureMode = false

    private var needBoot = false
        set(value) {
            field = value
            Logger.i(TAG, "need boot: $value")
            boostPrompt = value
        }
    private var bootVolume = BOOST_MIN
    private var isKeyDowning = false

    private var boostPrompt = false
        set(value) {
            Logger.i(TAG, "boost prompt field, field: $field, value:$value")
            if (field == value) {
                return
            }
            field = value
            if (value) {
                boostPrompt()
            }
        }

    var onVolumeBoost: ((Int) -> Unit)? = null

    fun reset() {
        Logger.i(TAG, "reset volume boot")
        needBoot = false
        bootVolume = BOOST_MIN
    }

    fun onVolumeChange(isInRoom: Boolean, intent: Intent?) {
        Logger.i(TAG, "on volume change, is in room:$isInRoom")
        if (!isInRoom) {
            return
        }
        intent?.let {
            if (VOLUME_CHANGED_ACTION == intent.action
                && intent.getIntExtra(EXTRA_VOLUME_STREAM_TYPE, -1) == AudioManager.STREAM_MUSIC
            ) {
                val preVolume = intent.getIntExtra(EXTRA_PREV_VOLUME_STREAM_VALUE, 0)
                val currentVolume = intent.getIntExtra(EXTRA_VOLUME_STREAM_VALUE, 0)
                val isMaxVolume = currentVolume == maxVolume

                if (preVolume == maxVolume
                    || currentVolume == maxVolume
                ) {
                    needBoot = isMaxVolume && !isKeyDowning
                }
            }
        }
    }

    fun onKeyDown(isInRoom: Boolean, keyCode: Int): Boolean {
        Logger.i(TAG, "on key down, is in room:$isInRoom")
        isKeyDowning = true
        //4、interceptor when in boost status and bootVolume more than the #BOOST_MIN,
        // Because when the volume is reduced to BOOST_MIN,
        // the system volume should be adjusted by pressing the volume down button
        if (isInRoom
            && needBoot
            && (keyCode == KeyEvent.KEYCODE_VOLUME_UP
                    || (keyCode == KeyEvent.KEYCODE_VOLUME_DOWN && bootVolume > BOOST_MIN))
        ) {
            Logger.i(TAG, "volume in boot mode")
            return true
        }
        return false
    }

    fun onKeyUp(isInRoom: Boolean, keyCode: Int): Boolean {
        Logger.i(TAG, "on key up, is in room:$isInRoom")
        isKeyDowning = false
        if (isInRoom
            && (keyCode == KeyEvent.KEYCODE_VOLUME_UP
                    || keyCode == KeyEvent.KEYCODE_VOLUME_DOWN)
        ) {
            if (needBoot) {
                //2、volume up
                if (keyCode == KeyEvent.KEYCODE_VOLUME_UP) {
                    bootVolume += BOOST_MIN
                    bootVolume = bootVolume.coerceAtMost(BOOST_MAX)
                    Logger.i(TAG, "on volume boost up, volume: $bootVolume")
                    onVolumeBoost?.invoke(bootVolume)
                    ToastUtils.showShort(
                        context?.getString(
                            R.string.avc_Android_boost,
                            bootVolume.toString()
                        )
                    )
                    return true
                } else if (keyCode == KeyEvent.KEYCODE_VOLUME_DOWN && bootVolume > BOOST_MIN) {
                    //3、volume down
                    bootVolume -= BOOST_MIN
                    Logger.i(TAG, "on volume boost down, volume: $bootVolume")
                    onVolumeBoost?.invoke(bootVolume)
                    ToastUtils.showShort(
                        context?.getString(
                            R.string.avc_Android_boost,
                            bootVolume.toString()
                        )
                    )
                    return true
                }
            } else {
                //1、into boost status and prompt user
                needBoot = isMaxStreamVolume(context)
                bootVolume = BOOST_MIN
            }
        }
        return false
    }

    private fun boostPrompt() {
        Logger.i(TAG, "volume is max, promp user click key volume up can boost volume")

        if (!isInPictureInPictureMode
            && AppUtils.isAppForeground()
        ) {
            ToastUtils.showShort(context?.getString(R.string.avc_Android_boost_prompt))
        }
    }

    fun destroy() {
        context = null
        onVolumeBoost = null
    }

    companion object {
        private const val TAG = "[UI][MeetingVolumeBootHelper]"
        const val EXTRA_VOLUME_STREAM_TYPE = "android.media.EXTRA_VOLUME_STREAM_TYPE"

        /**
         * The volume associated with the stream for the volume changed intent.
         */
        const val EXTRA_VOLUME_STREAM_VALUE = "android.media.EXTRA_VOLUME_STREAM_VALUE"

        /**
         * The previous volume associated with the stream for the volume changed intent.
         */
        const val EXTRA_PREV_VOLUME_STREAM_VALUE = "android.media.EXTRA_PREV_VOLUME_STREAM_VALUE"
        private const val BOOST_MIN = 100
        private const val BOOST_MAX = 400

        private fun isMaxStreamVolume(context: Context?) =
            (context?.getSystemService(Context.AUDIO_SERVICE) as AudioManager).let {
                it.getStreamVolume(AudioManager.STREAM_MUSIC) == it.getStreamMaxVolume(AudioManager.STREAM_MUSIC)
            }
    }
}
