package io.agora.avc.app.meeting

import android.os.Bundle
import android.util.SparseArray
import android.view.LayoutInflater
import android.view.ViewGroup
import androidx.core.util.forEach
import androidx.recyclerview.widget.DiffUtil
import androidx.recyclerview.widget.ListAdapter
import io.agora.avc.bo.valoran.ARoomUser
import io.agora.avc.databinding.AvcItemMeetingBinding
import io.agora.avc.extensions.getVideoStats
import io.agora.avc.extensions.isMySelf
import io.agora.avc.utils.ScreenUtils
import io.agora.frame.base.livedata.EventLiveData

class MeetingAdapter :
    ListAdapter<ARoomUser, MeetingViewHolder>(MeetingDiffItemCallback()) {

    private var viewHolders = SparseArray<MeetingViewHolder>()
    private var onItemListener: OnItemListener? = null
    val showVideoData by lazy {
        object : EventLiveData<Boolean>() {
            override fun postValue(value: Boolean?) {
                super.postValue(value)
                MeetingDiffItemCallback.showVideoData = value == true
            }
        }
    }

    fun setItemClickListener(listener: OnItemListener?) {
        this.onItemListener = listener
    }

    interface OnItemListener {
        fun onClicked(user: ARoomUser)
        fun onLongClick(user: ARoomUser)
        fun onCameraClick(user: ARoomUser)
    }

    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): MeetingViewHolder {
        val viewHolder = MeetingViewHolder(
            AvcItemMeetingBinding.inflate(
                LayoutInflater.from(parent.context),
                parent,
                false
            )
        )

        viewHolder.setItemClickListener(onItemListener)

        //adjust screen orientation
        val isPortrait = ScreenUtils.isPortrait()
        viewHolder.binding.root.rotation = if (isPortrait) 0f else -90f
        viewHolders.put(viewHolders.size(), viewHolder)
        return viewHolder
    }

    override fun onBindViewHolder(holder: MeetingViewHolder, position: Int) {
        holder.bindUser(getItem(position), showVideoData.value == true)
    }

    override fun onBindViewHolder(
        holder: MeetingViewHolder,
        position: Int,
        payloads: MutableList<Any>
    ) {
        if (payloads.isEmpty()) {
            onBindViewHolder(holder, position)
        } else {
            changePieceUI(payloads, holder, getItem(position), position)
        }
    }

    override fun onViewAttachedToWindow(holder: MeetingViewHolder) {
        super.onViewAttachedToWindow(holder)
        showVideoData.observeForever(holder.observer)
    }

    override fun onViewDetachedFromWindow(holder: MeetingViewHolder) {
        super.onViewDetachedFromWindow(holder)
        showVideoData.removeObserver(holder.observer)
    }

    fun destroy() {
        onItemListener = null
        viewHolders.forEach { _, value ->
            value.release()
        }
    }

    private fun changePieceUI(
        payloads: MutableList<Any>,
        holder: MeetingViewHolder,
        user: ARoomUser,
        position: Int
    ) {
        val bundle = payloads[0] as Bundle
        if (bundle.get(KEY_USER) != null) {
            holder.bindUser(user, showVideoData.value == true)
        }
        if (bundle.get(KEY_NAME) != null) {
            holder.bindName(user)
        }
        if (bundle.get(KEY_AUDIO_STATUS) != null) {
            holder.bindAudioStatus(user)
        }
        if (bundle.get(KEY_VIDEO_STATUS) != null) {
            holder.bindVideoStatus(user)
            holder.bindWatermark(user)
        }
        if (bundle.get(KEY_SPEAKING) != null) {
            holder.bindSpeaking(user)
        }
        if (bundle.get(KEY_QUALITY) != null) {
            holder.bindQuality(user)
        }
        if (bundle.get(KEY_SHARE_ID) != null) {
            holder.bindShareStatus(user)
        }
        if (bundle.get(KEY_MEDIA_STATICS) != null) {
            holder.setupMediaStatics(user, showVideoData.value == true)
        }
        if (bundle.get(KEY_WATERMARK) != null) {
            holder.bindWatermark(user)
        }
        if (bundle.get(KEY_VOLUME) != null) {
            holder.bindSpeaking(user)
        }
        if (bundle.get(KEY_ASSISTANT) != null) {
            holder.bindAssistant(user)
        }
    }

    class MeetingDiffItemCallback : DiffUtil.ItemCallback<ARoomUser>() {

        override fun areItemsTheSame(oldItem: ARoomUser, newItem: ARoomUser): Boolean {
            return oldItem.areTheSame(newItem)
        }

        override fun areContentsTheSame(oldItem: ARoomUser, newItem: ARoomUser): Boolean {
            return oldItem.streamId == newItem.streamId
                    && oldItem.uid == newItem.uid
                    && oldItem.name == newItem.name
                    && oldItem.isInterrupt == newItem.isInterrupt
                    && oldItem.audioState == newItem.audioState
                    && oldItem.videoState == newItem.videoState
                    && oldItem.isSpeaking == newItem.isSpeaking
                    && oldItem.quality == newItem.quality
                    && oldItem.shareId == newItem.shareId
                    && oldItem.parentStreamId == newItem.parentStreamId
                    && oldItem.hasWatermark == newItem.hasWatermark
                    && oldItem.volume == newItem.volume
                    && oldItem.isAssistant == newItem.isAssistant
                    && oldItem.online == newItem.online
                    && areContentsTheSameVideoStats(oldItem, newItem)
        }

        private fun areContentsTheSameVideoStats(oldItem: ARoomUser, newItem: ARoomUser): Boolean {
            return !showVideoData ||
                    (oldItem.bitrate == newItem.bitrate
                            && oldItem.fps == newItem.fps
                            && oldItem.streamType == newItem.streamType
                            && oldItem.width == newItem.width
                            && oldItem.height == newItem.height
                            && oldItem.rotation == newItem.rotation)
        }

        override fun getChangePayload(oldItem: ARoomUser, newItem: ARoomUser): Any? {
            val bundle = Bundle()

            if (oldItem.streamId != newItem.streamId || oldItem.uid != newItem.uid) {
                bundle.putBoolean(KEY_USER, true)
            }

            if (oldItem.name != newItem.name || oldItem.thirdPartyName != newItem.thirdPartyName || oldItem.thirdPartyAlias != newItem.thirdPartyAlias) {
                bundle.putBoolean(KEY_NAME, true)
            }

            if (oldItem.audioState != newItem.audioState) {
                bundle.putBoolean(KEY_AUDIO_STATUS, true)
            }

            if (oldItem.videoState != newItem.videoState) {
                bundle.putBoolean(KEY_VIDEO_STATUS, true)
            }

            if (oldItem.isSpeaking != newItem.isSpeaking) {
                bundle.putBoolean(KEY_SPEAKING, true)
            }

            if (oldItem.quality != newItem.quality) {
                bundle.putBoolean(KEY_QUALITY, true)
            }

            if (oldItem.shareId != newItem.shareId) {
                bundle.putBoolean(KEY_SHARE_ID, true)
            }

            if (oldItem.parentStreamId != newItem.parentStreamId) {
                bundle.putBoolean(KEY_PARENT_STREAM, true)
            }

            if (showVideoData && oldItem.getVideoStats() != newItem.getVideoStats()) {
                bundle.putBoolean(KEY_MEDIA_STATICS, true)
            }

            if (oldItem.hasWatermark != newItem.hasWatermark) {
                bundle.putBoolean(KEY_WATERMARK, true)
            }

            if (newItem.isMySelf()
                && oldItem.volume != newItem.volume
            ) {
                bundle.putBoolean(KEY_VOLUME, true)
            }

            if (oldItem.isAssistant != newItem.isAssistant
                || oldItem.online != newItem.online
            ) {
                bundle.putBoolean(KEY_ASSISTANT, true)
            }

            return if (bundle.size() > 0) {
                bundle
            } else {
                null
            }
        }

        companion object {
            var showVideoData = false
        }
    }

    companion object {
        private const val KEY_USER: String = "key_user"
        private const val KEY_NAME: String = "key_name"
        private const val KEY_AUDIO_STATUS: String = "key_audio_status"
        private const val KEY_VIDEO_STATUS: String = "key_video_status"
        private const val KEY_SPEAKING: String = "key_speaking"
        private const val KEY_QUALITY: String = "key_quality"
        private const val KEY_SHARE_ID: String = "key_share_id"
        private const val KEY_PARENT_STREAM: String = "key_parent_stream"
        private const val KEY_MEDIA_STATICS: String = "key_media_statics"
        private const val KEY_WATERMARK: String = "key_watermark"
        private const val KEY_VOLUME: String = "key_volume"
        private const val KEY_ASSISTANT: String = "key_assistant"
    }
}