package io.agora.avc.app.meeting

import android.app.Activity
import android.graphics.Rect
import android.view.View
import androidx.appcompat.app.AlertDialog
import androidx.core.view.doOnLayout
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.MutableLiveData
import com.google.android.material.dialog.MaterialAlertDialogBuilder
import io.agora.avc.R
import io.agora.avc.app.developer.DeveloperOptions
import io.agora.avc.app.master.AgoraActivity
import io.agora.avc.biz.event.MessageEvent
import io.agora.avc.bo.LocalUser
import io.agora.avc.bo.NetworkSignal
import io.agora.avc.bo.NoticeCode
import io.agora.avc.bo.Room
import io.agora.avc.databinding.AvcFragmentMeetingBinding
import io.agora.avc.extensions.getBegin
import io.agora.avc.extensions.showPop
import io.agora.avc.net.constant.Constants
import io.agora.avc.utils.ClipboardUtils
import io.agora.avc.utils.ConvertUtils
import io.agora.avc.utils.StringUtils
import io.agora.avc.widget.*
import io.agora.frame.base.livedata.EventLiveData
import io.agora.logger.Logger

class MeetingSystemWindowHelper(
    var activity: Activity?
) {

    lateinit var mBinding: AvcFragmentMeetingBinding

    private var menuPopupWindow: RoomMenuPopupWindow? = null
    private val mediaPopupWindow by lazy {
        if (activity == null) {
            null
        } else {
            MediaPopupWindow(activity!!)
        }
    }
    private var copyLinkPopupWindow: CopyLinkPopupWindow? = null
    private var signalQualityPopupWindow: SignalQualityPopupWindow? = null
    private var mediaVideoPopupWindow: MediaVideoPopupWindow? = null
    private var dumpTipPopupWindow: TextPopupWindow? = null

    private var unifiedDialog: AlertDialog? = null

    private var isVideoRating = false
    private var videoRatingTiming = 0

    private var networkSignal: NetworkSignal? = null

    fun bind(binding: AvcFragmentMeetingBinding) {
        mBinding = binding
    }

    fun showSignalPop() {
        showSignalPop(networkSignal)
    }

    private fun showSignalPop(networkSignal: NetworkSignal?) {
        val pos = IntArray(2)
        mBinding.flSignal.getLocationInWindow(pos)
        pos[0] += mBinding.flSignal.width / 2
        pos[1] += mBinding.flSignal.height
        activity?.let {
            signalQualityPopupWindow?.dismiss()
            signalQualityPopupWindow = SignalQualityPopupWindow(
                it,
                networkSignal
            ).apply {
                showAtLocation(pos, TipPopupWindow.DIRECTION_DOWN)
                setOnDismissListener {
                    signalQualityPopupWindow = null
                }
            }
        }
    }

    fun showCloudRecordingConfirmDialog(start: Boolean, action: () -> Unit) {
        var title = -1
        val content: Int
        val negativeText: Int
        val positiveText: Int
        if (start) {
            content = R.string.avc_cloud_recording_start_reminder
            negativeText = R.string.avc_cancel_button
            positiveText = R.string.avc_cloud_recording_start_button
        } else {
            ClipboardUtils.copyText(Constants.getAtlasRecordingUrl())
            title = R.string.avc_cloud_recording_end_reminder_title
            content = R.string.avc_cloud_recording_end_content
            negativeText = R.string.avc_cancel_button
            positiveText = R.string.avc_terminate_button
        }
        unifiedDialog?.dismiss()
        activity?.let {
            unifiedDialog =
                MaterialAlertDialogBuilder(it, R.style.avc_CustomMaterialAlertDialog)
                    .apply {
                        if (title != -1) {
                            setTitle(title)
                        }
                    }
                    .setCancelable(false)
                    .setMessage(content)
                    .setNegativeButton(negativeText) { _, _ ->

                    }
                    .setPositiveButton(positiveText) { _, _ ->
                        action.invoke()
                    }
                    .show()
        }
    }

    fun showKeepRecordingConfirmDialog(action: () -> Unit) {
        ClipboardUtils.copyText(Constants.getAtlasRecordingUrl())
        unifiedDialog?.dismiss()
        activity?.let {
            unifiedDialog =
                MaterialAlertDialogBuilder(it, R.style.avc_CustomMaterialAlertDialog)
                    .setTitle(R.string.avc_notice_keep_recording_title)
                    .setCancelable(false)
                    .setMessage(it.getString(R.string.avc_notice_keep_recording_message))
                    .setNegativeButton(R.string.avc_notice_keep_recording_cancel) { _, _ ->
                    }
                    .setPositiveButton(R.string.avc_notice_keep_recording_confirm) { _, _ ->
                        action.invoke()
                    }
                    .show()
        }
    }

    fun showOpenAudioConfirmDialog(action: () -> Unit) {
        unifiedDialog?.dismiss()
        activity?.let {
            unifiedDialog =
                MaterialAlertDialogBuilder(it, R.style.avc_CustomMaterialAlertDialog)
                    .setCancelable(false)
                    .setMessage(it.getString(R.string.avc_microphone_multiplayer_request))
                    .setNegativeButton(R.string.avc_no_button) { _, _ ->
                    }
                    .setPositiveButton(R.string.avc_yes_button) { _, _ ->
                        action.invoke()
                    }
                    .show()
        }
    }

    fun showApplyOpenAudioDialog(action: () -> Unit){
        unifiedDialog?.dismiss()
        activity?.let {
            unifiedDialog =
                MaterialAlertDialogBuilder(it, R.style.avc_CustomMaterialAlertDialog)
                    .setCancelable(false)
                    .setMessage(StringUtils.getString(R.string.avc_mute_all_raise_pop))
                    .setNegativeButton(R.string.avc_mute_all_raise_no_button) { _, _ ->
                    }
                    .setPositiveButton(R.string.avc_mute_all_raise_yes_button) { _, _ ->
                        action.invoke()
                    }
                    .show()
        }
    }

    private fun showMediaChangedTips(isScreenPortrait: Boolean) {
        Logger.i(TAG, "show media changed tips")
        mBinding.bottomBar.doOnLayout {
            val pos1 = IntArray(2)
            val pos2 = IntArray(2)
            val pos3 = IntArray(2)
            mBinding.bottomBar.getLocationInWindow(pos3)
            activity?.let {
                if (isScreenPortrait) {
                    mBinding.bottomBar.getLocalAudio().getLocationInWindow(pos1)
                    pos1[0] += mBinding.bottomBar.getLocalAudio().width / 2 // view middle
                    mBinding.bottomBar.getLocalVideo().getLocationInWindow(pos2)
                    pos2[0] += mBinding.bottomBar.getLocalVideo().width / 2
                } else {
                    mBinding.bottomBar.getLocalAudioL().getLocationInWindow(pos1)
                    pos1[0] += mBinding.bottomBar.getLocalAudioL().width / 2 // view middle
                    mBinding.bottomBar.getLocalVideoL().getLocationInWindow(pos2)
                    pos2[0] += mBinding.bottomBar.getLocalVideoL().width / 2
                }
            }

            mediaPopupWindow?.showAtLocation(when (mediaPopupWindow?.tipType) {
                TipPopupWindow.TIP_TYPE_MICROPHONE -> pos1
                TipPopupWindow.TIP_TYPE_CAMERA -> pos2
                else -> IntArray(2).apply {
                    set(0, (pos1[0] + pos2[0]) / 2) // position between audio and video
                    set(1, pos1[1])
                }
            }, TipPopupWindow.DIRECTION_UP)
        }
    }

    fun showMediaVideoPop(
        onCloseVideo: () -> Unit,
        onConvertCamera: () -> Unit
    ) {
        activity?.let {
            if (mediaVideoPopupWindow?.isShowing == true) {
                mediaVideoPopupWindow?.dismiss()
            }
            val target = mBinding.bottomBar.getLocalVideo()
            val pos = IntArray(2)
            target.getLocationInWindow(pos)
            pos[0] += target.width / 2
            pos[1] -= ConvertUtils.dp2px(32f)
            mediaVideoPopupWindow = MediaVideoPopupWindow(it) { _view ->
                when (_view.id) {
                    R.id.llCloseVideo -> {
                        onCloseVideo.invoke()
                    }
                    R.id.llConvertCamera -> {
                        onConvertCamera.invoke()
                    }
                }
            }.apply {
                showAtLocation(pos, TipPopupWindow.DIRECTION_UP)
            }
        }
    }

    fun showRoomInfoPop(
        room: Room?,
        onClickListener: View.OnClickListener
    ) {
        val pos = IntArray(2)
        mBinding.tvTitle.getLocationInWindow(pos)
        pos[0] += mBinding.tvTitle.width - ConvertUtils.dp2px(3f)
        pos[1] += mBinding.tvTitle.height
        activity?.let {
            copyLinkPopupWindow?.dismiss()

            copyLinkPopupWindow = CopyLinkPopupWindow(
                it,
                roomName = room?.name ?: "",
                password = room?.pwd ?: "",
                onClickListener = onClickListener
            ).apply {
                showAtLocation(
                    pos,
                    TipPopupWindow.DIRECTION_DOWN,
                    Rect(mBinding.insetLeftView.getBegin(), 0, 0, 0)
                )
                setOnDismissListener {
                    copyLinkPopupWindow = null
                }
            }
        }
    }

    fun showDumpTipPop() {
        dumpTipPopupWindow?.dismiss()
        activity?.let {
            showPop(
                target = mBinding.flDumpProgressBar,
                direction = TipPopupWindow.DIRECTION_DOWN
            ) {
                dumpTipPopupWindow = TextPopupWindow(it).apply {
                    updateText(it.getString(R.string.avc_room_dump))
                    setOnDismissListener {
                        dumpTipPopupWindow = null
                    }
                }
                dumpTipPopupWindow!!
            }
        }
    }

    fun showExitConfirmDialog(
        room: Room?,
        onNegativeClick: () -> Unit,
        onPositiveClick: () -> Unit
    ) {
        room?.name?.let { _name ->
            activity?.let {
                unifiedDialog =
                    MaterialAlertDialogBuilder(it, R.style.avc_CustomMaterialAlertDialog)
                        .setTitle(R.string.avc_notice_leave_room_title)
                        .setCancelable(false)
                        .setMessage(it.getString(R.string.avc_notice_leave_room_message, _name))
                        .setNegativeButton(R.string.avc_notice_leave_room_cancel) { _, _ ->
                            onNegativeClick.invoke()
                        }
                        .setPositiveButton(R.string.avc_notice_leave_room_confirm) { _, _ ->
                            onPositiveClick.invoke()
                        }
                        .show()
            }
        }
    }

    fun showMenuPop(
        isSharingScreen: Boolean,
        localUser: LocalUser?,
        room: Room?,
        developerOptions: DeveloperOptions?,
        hasUnreadMessage: Boolean,
        callback: RoomMenuPopupWindow.MenuCallback
    ) {
        activity?.let { _context ->
            menuPopupWindow?.apply {
                dismiss()
                return@let
            }
            menuPopupWindow = RoomMenuPopupWindow(
                ctx = _context,
                isSharingScreen = isSharingScreen,
                isThirdPartyLoggedIn = localUser?.isThirdPartyLoggedIn == true,
                isCloudRecording = localUser?.isCloudRecording == true,
                isAgoraRoom = room?.isInternal() == true,
                developerOptions = developerOptions,
                isVideoRating = isVideoRating,
                hasUnreadMessage = hasUnreadMessage,
                videoRateTime = videoRatingTiming,
                callback = callback
            )
            menuPopupWindow!!.showUpAnchorView(mBinding.bottomBar)
            menuPopupWindow!!.setOnDismissListener {
                menuPopupWindow = null
            }
        }
    }

    fun setupMenuPopupWindow(
        isScreenSharing: Boolean,
        isThirdPartyLoggedIn: Boolean,
        isCloudRecording: Boolean,
        isInternal: Boolean
    ) {
        menuPopupWindow?.apply {
            updateCloudRecord(
                isThirdPartyLoggedIn = isThirdPartyLoggedIn,
                isCloudRecording = isCloudRecording,
                isAgoraRoom = isInternal
            )
            updateShareItem(isScreenSharing)
        }
    }

    fun dismissPops() {
        menuPopupWindow?.dismiss()
        mediaPopupWindow?.dismiss()
        copyLinkPopupWindow?.dismiss()
        signalQualityPopupWindow?.dismiss()
        mediaVideoPopupWindow?.dismiss()
        dumpTipPopupWindow?.dismiss()
        unifiedDialog?.dismiss()
    }

    fun hasPopupWindowShowing(): Boolean {
        return mediaPopupWindow?.isShowing == true
                || copyLinkPopupWindow?.isShowing == true
                || mediaVideoPopupWindow?.isShowing == true
                || signalQualityPopupWindow?.isShowing == true
                || dumpTipPopupWindow?.isShowing == true
                || menuPopupWindow?.isShowing == true
                || unifiedDialog?.isShowing == true
                || isUploadStatusDialogShowing()
                || isErrorAlertDialogShowing()
                || isPermissionSettingDialogShowing()
                || isShareLinkConfirmDialogShowing()
    }

    private fun isUploadStatusDialogShowing(): Boolean {
        val host = activity
        if (host is AgoraActivity) {
            return host.isUploadDialogShowing()
        }
        return false
    }

    private fun isErrorAlertDialogShowing(): Boolean {
        val host = activity
        if (host is AgoraActivity) {
            return host.isErrorAlertDialogShowing()
        }
        return false
    }

    private fun isPermissionSettingDialogShowing(): Boolean {
        val host = activity
        if (host is AgoraActivity) {
            return host.isPermissionSettingDialogShowing()
        }
        return false
    }

    private fun isShareLinkConfirmDialogShowing(): Boolean {
        val host = activity
        if (host is AgoraActivity) {
            return host.isShareLinkConfirmDialogShowing()
        }
        return false
    }

    fun observeUnReadMessageChanged(
        lifecycleOwner: LifecycleOwner,
        unReadMessageChangedLiveData: MutableLiveData<Int>?
    ) {
        unReadMessageChangedLiveData?.observe(lifecycleOwner) {
            menuPopupWindow?.updateHasUnreadMessage(it > 0)
        }
    }

    fun observeVideoRating(
        lifecycleOwner: LifecycleOwner,
        videoRatingChangedLiveData: MutableLiveData<Boolean>?,
        videoRatingTimingLiveData: EventLiveData<Long>?
    ) {
        videoRatingChangedLiveData?.observe(lifecycleOwner) {
            isVideoRating = it
            if (!it) {
                videoRatingTiming = 0
                menuPopupWindow?.videoRateReset()
            }
        }
        videoRatingTimingLiveData?.observe(lifecycleOwner) {
            videoRatingTiming = 60 - (it / 1000).toInt()
            menuPopupWindow?.updateVideoRateTime(videoRatingTiming)
        }
    }

    fun observeNetworkSignalChanged(
        lifecycleOwner: LifecycleOwner,
        networkSignalChangedLiveData: MutableLiveData<NetworkSignal>?
    ) {
        networkSignalChangedLiveData?.observe(lifecycleOwner) {
            if (networkSignal != null && it.bizConnected == networkSignal?.bizConnected) {
                signalQualityPopupWindow?.setNetWorkSignal(it)
            } else if (signalQualityPopupWindow?.isShowing == true) {
                showSignalPop(it)
            }
            networkSignal = it
        }
    }

    fun observePopup(
        lifecycleOwner: LifecycleOwner,
        popupLiveData: EventLiveData<MessageEvent>?,
        isScreenPortrait: () -> Boolean
    ) {
        popupLiveData?.observe(lifecycleOwner) {
            it.obj?.let { _obj ->
                if (_obj is NoticeCode) {
                    when (_obj) {
                        NoticeCode.CODE_DEFAULT_MICROPHONE -> {
                            mediaPopupWindow?.tipType = TipPopupWindow.TIP_TYPE_MICROPHONE
                            showMediaChangedTips(isScreenPortrait.invoke())
                        }
                        NoticeCode.CODE_DEFAULT_CAMERA -> {
                            mediaPopupWindow?.tipType = TipPopupWindow.TIP_TYPE_CAMERA
                            showMediaChangedTips(isScreenPortrait.invoke())
                        }
                        NoticeCode.CODE_DEFAULT_BOTH -> {
                            mediaPopupWindow?.tipType = TipPopupWindow.TIP_TYPE_MEDIA
                            showMediaChangedTips(isScreenPortrait.invoke())
                        }
                    }
                }
            }
        }
    }

    fun onDestroy() {
        activity = null
    }

    companion object {
        private const val TAG = "[UI][MeetingWindowHelper]"
    }
}